<?php
/**
 * Storage Result Class
 *
 * @package SwiftOffload\Storage
 */

namespace SwiftOffload\Storage;

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Storage result wrapper for operation responses
 */
class Storage_Result {

	/**
	 * Success status
	 *
	 * @var bool
	 */
	private $success;

	/**
	 * Result data
	 *
	 * @var mixed
	 */
	private $data;

	/**
	 * Error message
	 *
	 * @var string|null
	 */
	private $error;

	/**
	 * HTTP status code
	 *
	 * @var int
	 */
	private $status_code;

	/**
	 * Additional metadata
	 *
	 * @var array
	 */
	private $metadata;

	/**
	 * Constructor
	 *
	 * @param bool        $success     Success status
	 * @param mixed       $data        Result data
	 * @param string|null $error       Error message
	 * @param int         $status_code HTTP status code
	 * @param array       $metadata    Additional metadata
	 */
	public function __construct( $success, $data = null, $error = null, $status_code = 200, $metadata = array() ) {
		$this->success     = $success;
		$this->data        = $data;
		$this->error       = $error;
		$this->status_code = $status_code;
		$this->metadata    = $metadata;
	}

	/**
	 * Create a success result
	 *
	 * @param mixed $data       Result data
	 * @param array $metadata   Additional metadata
	 * @return Storage_Result
	 */
	public static function success( $data = null, $metadata = array() ) {
		return new self( true, $data, null, 200, $metadata );
	}

	/**
	 * Create an error result
	 *
	 * @param string $error       Error message
	 * @param int    $status_code HTTP status code
	 * @param array  $metadata    Additional metadata
	 * @return Storage_Result
	 */
	public static function error( $error, $status_code = 500, $metadata = array() ) {
		return new self( false, null, $error, $status_code, $metadata );
	}

	/**
	 * Check if operation was successful
	 *
	 * @return bool
	 */
	public function is_success() {
		return $this->success;
	}

	/**
	 * Check if operation failed
	 *
	 * @return bool
	 */
	public function is_error() {
		return ! $this->success;
	}

	/**
	 * Get result data
	 *
	 * @return mixed
	 */
	public function get_data() {
		return $this->data;
	}

	/**
	 * Get error message
	 *
	 * @return string|null
	 */
	public function get_error() {
		return $this->error;
	}

	/**
	 * Get status code
	 *
	 * @return int
	 */
	public function get_status_code() {
		return $this->status_code;
	}

	/**
	 * Get metadata
	 *
	 * @param string $key Optional key to get specific metadata
	 * @return mixed
	 */
	public function get_metadata( $key = null ) {
		if ( $key !== null ) {
			return $this->metadata[ $key ] ?? null;
		}
		return $this->metadata;
	}

	/**
	 * Set metadata
	 *
	 * @param string $key   Metadata key
	 * @param mixed  $value Metadata value
	 */
	public function set_metadata( $key, $value ) {
		$this->metadata[ $key ] = $value;
	}

	/**
	 * Convert to array
	 *
	 * @return array
	 */
	public function to_array() {
		return array(
			'success'     => $this->success,
			'data'        => $this->data,
			'error'       => $this->error,
			'status_code' => $this->status_code,
			'metadata'    => $this->metadata,
		);
	}

	/**
	 * Convert to JSON
	 *
	 * @return string
	 */
	public function to_json() {
		return wp_json_encode( $this->to_array() );
	}

	/**
	 * Magic method for string conversion
	 *
	 * @return string
	 */
	public function __toString() {
		if ( $this->success ) {
			return sprintf( 'Success: %s', wp_json_encode( $this->data ) );
		} else {
			return sprintf( 'Error: %s (Code: %d)', $this->error, $this->status_code );
		}
	}
}
