<?php
/**
 * Settings Routes Class
 *
 * @package SwiftOffload\Rest\Routes
 */

namespace SwiftOffload\Rest\Routes;

use WP_REST_Controller;
use WP_REST_Server;
use WP_Error;

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Settings Routes handles settings-related API endpoints
 */
class Routes_Settings extends WP_REST_Controller {

	/**
	 * Namespace
	 *
	 * @var string
	 */
	protected $namespace = 'swift-offload/v1';

	/**
	 * Rest base
	 *
	 * @var string
	 */
	protected $rest_base = 'settings';

	/**
	 * Register routes
	 */
	public function register_routes() {
		// Get all settings
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_settings' ),
				'permission_callback' => array( $this, 'check_permissions' ),
			)
		);

		// Update settings
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => array( $this, 'update_settings' ),
				'permission_callback' => array( $this, 'check_permissions' ),
				'args'                => array(
					'settings'       => array(
						'type'     => 'object',
						'required' => false,
					),
					'storage_config' => array(
						'type'     => 'object',
						'required' => false,
					),
					'cdn_config'     => array(
						'type'     => 'object',
						'required' => false,
					),
				),
			)
		);

		// Get storage providers
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/providers',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_providers' ),
				'permission_callback' => array( $this, 'check_permissions' ),
			)
		);

		// Reset settings to defaults
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/reset',
			array(
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => array( $this, 'reset_settings' ),
				'permission_callback' => array( $this, 'check_permissions' ),
			)
		);

		// Export settings
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/export',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'export_settings' ),
				'permission_callback' => array( $this, 'check_permissions' ),
			)
		);

		// Import settings
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/import',
			array(
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => array( $this, 'import_settings' ),
				'permission_callback' => array( $this, 'check_permissions' ),
				'args'                => array(
					'settings_data' => array(
						'type'     => 'object',
						'required' => true,
					),
				),
			)
		);
	}

	/**
	 * Get all settings
	 *
	 * @param \WP_REST_Request $request Request object
	 * @return \WP_REST_Response
	 */
	public function get_settings( $request ) {
		$settings = array(
			'general' => get_option( 'swift_offload_settings', array() ),
			'storage' => swift_offload_get_storage_config(),
			'cdn'     => swift_offload_get_cdn_config(),
		);

		// Mask sensitive data
		if ( isset( $settings['storage']['secret_key'] ) ) {
			$settings['storage']['secret_key'] = $this->mask_secret( $settings['storage']['secret_key'] );
		}

		return rest_ensure_response( $settings );
	}

	/**
	 * Update settings
	 *
	 * @param \WP_REST_Request $request Request object
	 * @return \WP_REST_Response|\WP_Error
	 */
	public function update_settings( $request ) {
		$settings       = $request->get_param( 'settings' );
		$storage_config = $request->get_param( 'storage_config' );
		$cdn_config     = $request->get_param( 'cdn_config' );

		$updated = array();

		// Update general settings
		if ( $settings && is_array( $settings ) ) {
			$validated_settings = $this->validate_general_settings( $settings );
			if ( is_wp_error( $validated_settings ) ) {
				return $validated_settings;
			}

			update_option( 'swift_offload_settings', $validated_settings );
			$updated['settings'] = true;
		}

		// Update storage configuration
		if ( $storage_config && is_array( $storage_config ) ) {
			$validated_storage = $this->validate_storage_config( $storage_config );
			if ( is_wp_error( $validated_storage ) ) {
				return $validated_storage;
			}

			update_option( 'swift_offload_storage_config', $validated_storage );
			$updated['storage_config'] = true;

			// Reset storage provider to reload with new config
			$storage_registry = \SwiftOffload\Plugin::get_instance()->get_storage_registry();
			$storage_registry->reset_active_provider();
		}

		// Update CDN configuration
		if ( $cdn_config && is_array( $cdn_config ) ) {
			$validated_cdn = $this->validate_cdn_config( $cdn_config );
			if ( is_wp_error( $validated_cdn ) ) {
				return $validated_cdn;
			}

			update_option( 'swift_offload_cdn_config', $validated_cdn );
			$updated['cdn_config'] = true;
		}

		return rest_ensure_response(
			array(
				'success' => true,
				'message' => __( 'Settings updated successfully', 'swift-offload' ),
				'updated' => $updated,
			)
		);
	}

	/**
	 * Get available storage providers
	 *
	 * @param \WP_REST_Request $request Request object
	 * @return \WP_REST_Response
	 */
	public function get_providers( $request ) {
		$storage_registry = \SwiftOffload\Plugin::get_instance()->get_storage_registry();
		$providers        = $storage_registry->get_providers();

		$provider_info = array();
		foreach ( $providers as $id => $class ) {
			$provider_info[ $id ] = $storage_registry->get_provider_info( $id );
		}

		return rest_ensure_response( $provider_info );
	}

	/**
	 * Reset settings to defaults
	 *
	 * @param \WP_REST_Request $request Request object
	 * @return \WP_REST_Response
	 */
	public function reset_settings( $request ) {
		delete_option( 'swift_offload_settings' );
		delete_option( 'swift_offload_storage_config' );
		delete_option( 'swift_offload_cdn_config' );

		// Recreate default settings
		\SwiftOffload\Plugin::set_default_options();

		return rest_ensure_response(
			array(
				'success' => true,
				'message' => __( 'Settings reset to defaults', 'swift-offload' ),
			)
		);
	}

	/**
	 * Export settings
	 *
	 * @param \WP_REST_Request $request Request object
	 * @return \WP_REST_Response
	 */
	public function export_settings( $request ) {
		$export_data = array(
			'version'        => SWIFT_OFFLOAD_VERSION,
			'exported_at'    => current_time( 'mysql' ),
			'settings'       => get_option( 'swift_offload_settings', array() ),
			'storage_config' => swift_offload_get_storage_config(),
			'cdn_config'     => swift_offload_get_cdn_config(),
		);

		// Mask sensitive data
		if ( isset( $export_data['storage_config']['secret_key'] ) ) {
			unset( $export_data['storage_config']['secret_key'] );
		}

		return rest_ensure_response( $export_data );
	}

	/**
	 * Import settings
	 *
	 * @param \WP_REST_Request $request Request object
	 * @return \WP_REST_Response|\WP_Error
	 */
	public function import_settings( $request ) {
		$settings_data = $request->get_param( 'settings_data' );

		if ( ! $settings_data || ! is_array( $settings_data ) ) {
			return new WP_Error( 'invalid_data', __( 'Invalid settings data', 'swift-offload' ), array( 'status' => 400 ) );
		}

		$imported = array();

		// Import general settings
		if ( isset( $settings_data['settings'] ) ) {
			update_option( 'swift_offload_settings', $settings_data['settings'] );
			$imported[] = 'settings';
		}

		// Import storage config (but preserve sensitive keys)
		if ( isset( $settings_data['storage_config'] ) ) {
			$current_storage = swift_offload_get_storage_config();
			$new_storage     = array_merge( $current_storage, $settings_data['storage_config'] );
			update_option( 'swift_offload_storage_config', $new_storage );
			$imported[] = 'storage_config';
		}

		// Import CDN config
		if ( isset( $settings_data['cdn_config'] ) ) {
			update_option( 'swift_offload_cdn_config', $settings_data['cdn_config'] );
			$imported[] = 'cdn_config';
		}

		return rest_ensure_response(
			array(
				'success'  => true,
				'message'  => __( 'Settings imported successfully', 'swift-offload' ),
				'imported' => $imported,
			)
		);
	}

	/**
	 * Validate general settings
	 *
	 * @param array $settings Settings data
	 * @return array|\WP_Error
	 */
	private function validate_general_settings( $settings ) {
		$rules = array(
			'offload_on_upload'  => array( 'type' => 'boolean' ),
			'remove_local_files' => array( 'type' => 'boolean' ),
			'object_key_format'  => array(
				'type'       => 'string',
				'max_length' => 255,
			),
			'storage_class'      => array(
				'type' => 'string',
				'enum' => array( 'STANDARD', 'REDUCED_REDUNDANCY', 'STANDARD_IA', 'ONEZONE_IA', 'INTELLIGENT_TIERING', 'GLACIER', 'DEEP_ARCHIVE' ),
			),
			'object_acl'         => array(
				'type' => 'string',
				'enum' => array( 'private', 'public-read', 'public-read-write', 'authenticated-read' ),
			),
		);

		return $this->validate_data( $settings, $rules );
	}

	/**
	 * Validate storage configuration
	 *
	 * @param array $config Storage config
	 * @return array|\WP_Error
	 */
	private function validate_storage_config( $config ) {
		$rules = array(
			'provider'       => array(
				'type'     => 'string',
				'required' => true,
				'enum'     => array( 's3' ),
			),
			'access_key'     => array(
				'type'       => 'string',
				'required'   => true,
				'min_length' => 16,
			),
			'secret_key'     => array(
				'type'       => 'string',
				'required'   => true,
				'min_length' => 32,
			),
			'region'         => array(
				'type'     => 'string',
				'required' => true,
			),
			'bucket'         => array(
				'type'     => 'string',
				'required' => true,
			),
			'endpoint'       => array( 'type' => 'string' ),
			'use_path_style' => array( 'type' => 'boolean' ),
		);

		return $this->validate_data( $config, $rules );
	}

	/**
	 * Validate CDN configuration
	 *
	 * @param array $config CDN config
	 * @return array|\WP_Error
	 */
	private function validate_cdn_config( $config ) {
		$rules = array(
			'enabled'             => array( 'type' => 'boolean' ),
			'distribution_domain' => array( 'type' => 'string' ),
			'custom_domain'       => array( 'type' => 'string' ),
			'signed_urls'         => array( 'type' => 'boolean' ),
			'query_string_auth'   => array( 'type' => 'boolean' ),
		);

		return $this->validate_data( $config, $rules );
	}

	/**
	 * Validate data against rules
	 *
	 * @param array $data Data to validate
	 * @param array $rules Validation rules
	 * @return array|\WP_Error
	 */
	private function validate_data( $data, $rules ) {
		$validated = array();
		$errors    = array();

		foreach ( $rules as $field => $rule ) {
			$value = $data[ $field ] ?? null;

			// Check required fields
			if ( ! empty( $rule['required'] ) && ( null === $value || '' === $value ) ) {
				/* translators: %s: field name */
				$errors[] = sprintf( __( 'Field %s is required', 'swift-offload' ), $field );
				continue;
			}

			// Skip if not required and empty
			if ( null === $value || '' === $value ) {
				continue;
			}

			// Type validation and sanitization
			switch ( $rule['type'] ) {
				case 'string':
					$validated[ $field ] = sanitize_text_field( $value );
					break;
				case 'boolean':
					$validated[ $field ] = (bool) $value;
					break;
				case 'integer':
					$validated[ $field ] = (int) $value;
					break;
				default:
					$validated[ $field ] = $value;
			}

			// Enum validation
			if ( isset( $rule['enum'] ) && ! in_array( $validated[ $field ], $rule['enum'] ) ) {
				/* translators: %1$s: field name, %2$s: list of allowed values */
				$errors[] = sprintf( __( 'Field %1$s must be one of: %2$s', 'swift-offload' ), $field, implode( ', ', $rule['enum'] ) );
			}

			// Length validation
			if ( isset( $rule['min_length'] ) && strlen( $validated[ $field ] ) < $rule['min_length'] ) {
				/* translators: %1$s: field name, %2$d: minimum character count */
				$errors[] = sprintf( __( 'Field %1$s must be at least %2$d characters', 'swift-offload' ), $field, $rule['min_length'] );
			}

			if ( isset( $rule['max_length'] ) && strlen( $validated[ $field ] ) > $rule['max_length'] ) {
				/* translators: %1$s: field name, %2$d: maximum character count */
				$errors[] = sprintf( __( 'Field %1$s must be no more than %2$d characters', 'swift-offload' ), $field, $rule['max_length'] );
			}
		}

		if ( ! empty( $errors ) ) {
			return new WP_Error( 'validation_failed', implode( ', ', $errors ), array( 'status' => 400 ) );
		}

		return $validated;
	}

	/**
	 * Check permissions
	 *
	 * @return bool|\WP_Error
	 */
	public function check_permissions() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return new WP_Error( 'rest_forbidden', __( 'You do not have permission to manage settings', 'swift-offload' ), array( 'status' => 403 ) );
		}

		return true;
	}

	/**
	 * Mask sensitive values
	 *
	 * @param string $value Value to mask
	 * @return string
	 */
	private function mask_secret( $value ) {
		if ( strlen( $value ) <= 8 ) {
			return str_repeat( '*', strlen( $value ) );
		}

		return substr( $value, 0, 4 ) . str_repeat( '*', strlen( $value ) - 8 ) . substr( $value, -4 );
	}
}
