<?php
/**
 * Media Routes Class
 *
 * @package SwiftOffload\Rest\Routes
 */

namespace SwiftOffload\Rest\Routes;

use WP_REST_Controller;
use WP_REST_Server;
use WP_Error;
use SwiftOffload\Plugin;

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Media Routes handles media-related API endpoints
 */
class Routes_Media extends WP_REST_Controller {

	/**
	 * Namespace
	 *
	 * @var string
	 */
	protected $namespace = 'swift-offload/v1';

	/**
	 * Rest base
	 *
	 * @var string
	 */
	protected $rest_base = 'media';

	/**
	 * Register routes
	 */
	public function register_routes() {
		// Offload specific attachments
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/offload',
			array(
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => array( $this, 'offload_attachments' ),
				'permission_callback' => array( $this, 'check_permissions' ),
				'args'                => array(
					'attachment_ids' => array(
						'required' => true,
						'type'     => 'array',
						'items'    => array(
							'type' => 'integer',
						),
					),
				),
			)
		);

		// Get attachment offload status
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>\d+)/status',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_attachment_status' ),
				'permission_callback' => array( $this, 'check_permissions' ),
			)
		);

		// Get media library with offload status
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_media_library' ),
				'permission_callback' => array( $this, 'check_permissions' ),
				'args'                => array(
					'page'           => array(
						'type'    => 'integer',
						'default' => 1,
						'minimum' => 1,
					),
					'per_page'       => array(
						'type'    => 'integer',
						'default' => 20,
						'minimum' => 1,
						'maximum' => 100,
					),
					'search'         => array(
						'type' => 'string',
					),
					'mime_type'      => array(
						'type' => 'string',
					),
					'offload_status' => array(
						'type'    => 'string',
						'enum'    => array( 'all', 'offloaded', 'local' ),
						'default' => 'all',
					),
				),
			)
		);

		// Remove local files
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/remove-local',
			array(
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => array( $this, 'remove_local_files' ),
				'permission_callback' => array( $this, 'check_permissions' ),
				'args'                => array(
					'attachment_ids' => array(
						'required' => true,
						'type'     => 'array',
						'items'    => array(
							'type' => 'integer',
						),
					),
					'verify_s3'      => array(
						'type'    => 'boolean',
						'default' => true,
					),
				),
			)
		);

		// Regenerate URLs
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/regenerate-urls',
			array(
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => array( $this, 'regenerate_urls' ),
				'permission_callback' => array( $this, 'check_permissions' ),
				'args'                => array(
					'attachment_ids' => array(
						'required' => true,
						'type'     => 'array',
						'items'    => array(
							'type' => 'integer',
						),
					),
				),
			)
		);
	}

	/**
	 * Offload attachments
	 *
	 * @param \WP_REST_Request $request Request object
	 * @return \WP_REST_Response|\WP_Error
	 */
	public function offload_attachments( $request ) {
		$attachment_ids = $request->get_param( 'attachment_ids' );

		if ( empty( $attachment_ids ) ) {
			return new WP_Error( 'no_attachments', __( 'No attachment IDs provided', 'swift-offload' ), array( 'status' => 400 ) );
		}

		// Validate attachment IDs
		foreach ( $attachment_ids as $attachment_id ) {
			$validation = \SwiftOffload\Utils\Validators::validate_attachment_id( $attachment_id );
			if ( ! $validation['valid'] ) {
				return new WP_Error( 'invalid_attachment', $validation['errors'][0], array( 'status' => 400 ) );
			}
		}

		// Check rate limit
		if ( ! \SwiftOffload\Utils\Validators::check_rate_limit( 'offload_media', 20 ) ) {
			return new WP_Error( 'rate_limit', __( 'Too many offload requests', 'swift-offload' ), array( 'status' => 429 ) );
		}

		$media_hooks = Plugin::get_instance()->get_media_hooks();
		$results     = $media_hooks->bulk_offload( $attachment_ids );

		/* translators: %1$d: number of successful offloads, %2$d: number of failed offloads, %3$d: number of skipped offloads */
		return rest_ensure_response(
			array(
				'success' => true,
				'results' => $results,
				'message' => sprintf(
					__( 'Offload completed. Success: %1$d, Failed: %2$d, Skipped: %3$d', 'swift-offload' ),
					$results['success'],
					$results['failed'],
					$results['skipped']
				),
			)
		);
	}

	/**
	 * Get attachment offload status
	 *
	 * @param \WP_REST_Request $request Request object
	 * @return \WP_REST_Response|\WP_Error
	 */
	public function get_attachment_status( $request ) {
		$attachment_id = (int) $request->get_param( 'id' );

		$validation = \SwiftOffload\Utils\Validators::validate_attachment_id( $attachment_id );
		if ( ! $validation['valid'] ) {
			return new WP_Error( 'invalid_attachment', $validation['errors'][0], array( 'status' => 400 ) );
		}

		$is_offloaded      = swift_offload_is_attachment_offloaded( $attachment_id );
		$s3_data           = null;
		$local_file_exists = false;

		if ( $is_offloaded ) {
			$s3_data = swift_offload_get_attachment_s3_data( $attachment_id );
		}

		$local_path = get_attached_file( $attachment_id );
		if ( $local_path ) {
			$local_file_exists = file_exists( $local_path );
		}

		return rest_ensure_response(
			array(
				'attachment_id'     => $attachment_id,
				'is_offloaded'      => $is_offloaded,
				'local_file_exists' => $local_file_exists,
				's3_data'           => $s3_data,
				'urls'              => array(
					'original' => wp_get_attachment_url( $attachment_id ),
					's3'       => $is_offloaded ? swift_offload_get_attachment_s3_url( $attachment_id ) : null,
				),
			)
		);
	}

	/**
	 * Get media library with offload status
	 *
	 * @param \WP_REST_Request $request Request object
	 * @return \WP_REST_Response
	 */
	public function get_media_library( $request ) {
		$page           = (int) $request->get_param( 'page' );
		$per_page       = (int) $request->get_param( 'per_page' );
		$search         = $request->get_param( 'search' );
		$mime_type      = $request->get_param( 'mime_type' );
		$offload_status = $request->get_param( 'offload_status' );

		global $wpdb;

		// Build query
		$where_conditions = array( "p.post_type = 'attachment'" );
		$join_conditions  = array();
		$values           = array();

		// Search
		if ( ! empty( $search ) ) {
			$where_conditions[] = 'p.post_title LIKE %s';
			$values[]           = '%' . $wpdb->esc_like( $search ) . '%';
		}

		// Mime type filter
		if ( ! empty( $mime_type ) ) {
			$where_conditions[] = 'p.post_mime_type = %s';
			$values[]           = $mime_type;
		}

		// Offload status filter
		$items_table = $wpdb->prefix . 'swift_offload_items';
		if ( $offload_status === 'offloaded' ) {
			$join_conditions[] = "INNER JOIN {$items_table} ci ON p.ID = ci.attachment_id";
		} elseif ( $offload_status === 'local' ) {
			$join_conditions[]  = "LEFT JOIN {$items_table} ci ON p.ID = ci.attachment_id";
			$where_conditions[] = 'ci.attachment_id IS NULL';
		} else {
			$join_conditions[] = "LEFT JOIN {$items_table} ci ON p.ID = ci.attachment_id";
		}

		// Count total
		$count_sql = "SELECT COUNT(DISTINCT p.ID) FROM {$wpdb->posts} p " .
					implode( ' ', $join_conditions ) .
					' WHERE ' . implode( ' AND ', $where_conditions );

		$total = $wpdb->get_var( ! empty( $values ) ? $wpdb->prepare( $count_sql, $values ) : $count_sql );

		// Get results
		$offset      = ( $page - 1 ) * $per_page;
		$results_sql = "SELECT p.ID, p.post_title, p.post_mime_type, p.post_date, ci.bucket, ci.object_key, ci.offloaded_at 
                        FROM {$wpdb->posts} p " .
						implode( ' ', $join_conditions ) .
						' WHERE ' . implode( ' AND ', $where_conditions ) .
						' ORDER BY p.post_date DESC LIMIT %d OFFSET %d';

		$query_values = array_merge( $values, array( $per_page, $offset ) );
		$results      = $wpdb->get_results( $wpdb->prepare( $results_sql, $query_values ), ARRAY_A );

		// Enhance results with additional data
		foreach ( $results as &$item ) {
			$item['is_offloaded']      = ! empty( $item['bucket'] );
			$item['file_size']         = 0;
			$item['local_file_exists'] = false;

			$local_path = get_attached_file( $item['ID'] );
			if ( $local_path && file_exists( $local_path ) ) {
				$item['file_size']         = filesize( $local_path );
				$item['local_file_exists'] = true;
			}

			$item['thumbnail_url'] = wp_get_attachment_image_url( $item['ID'], 'thumbnail' );
		}

		return rest_ensure_response(
			array(
				'items'      => $results,
				'pagination' => array(
					'total'       => (int) $total,
					'page'        => $page,
					'per_page'    => $per_page,
					'total_pages' => ceil( $total / $per_page ),
				),
			)
		);
	}

	/**
	 * Remove local files
	 *
	 * @param \WP_REST_Request $request Request object
	 * @return \WP_REST_Response|\WP_Error
	 */
	public function remove_local_files( $request ) {
		$attachment_ids = $request->get_param( 'attachment_ids' );
		$verify_s3      = $request->get_param( 'verify_s3' );

		if ( empty( $attachment_ids ) ) {
			return new WP_Error( 'no_attachments', __( 'No attachment IDs provided', 'swift-offload' ), array( 'status' => 400 ) );
		}

		$media_hooks      = Plugin::get_instance()->get_media_hooks();
		$storage_registry = Plugin::get_instance()->get_storage_registry();
		$provider         = $storage_registry->get_active_provider();

		$results = array(
			'success' => 0,
			'failed'  => 0,
			'skipped' => 0,
			'errors'  => array(),
		);

		foreach ( $attachment_ids as $attachment_id ) {
			$validation = \SwiftOffload\Utils\Validators::validate_attachment_id( $attachment_id );
			if ( ! $validation['valid'] ) {
				++$results['failed'];
				$results['errors'][] = "Invalid attachment ID: {$attachment_id}";
				continue;
			}

			// Check if attachment is offloaded
			if ( ! swift_offload_is_attachment_offloaded( $attachment_id ) ) {
				++$results['skipped'];
				continue;
			}

			// Verify S3 object exists if requested
			if ( $verify_s3 ) {
				$s3_data = swift_offload_get_attachment_s3_data( $attachment_id );
				if ( ! $s3_data || ! $provider->object_exists( $s3_data['object_key'] ) ) {
					++$results['skipped'];
					$results['errors'][] = "S3 object not found for attachment {$attachment_id}";
					continue;
				}
			}

			$success = $media_hooks->remove_local_file( $attachment_id );

			if ( $success ) {
				++$results['success'];
			} else {
				++$results['failed'];
				$results['errors'][] = "Failed to remove local file for attachment {$attachment_id}";
			}
		}

		/* translators: %1$d: number of successful removals, %2$d: number of failed removals, %3$d: number of skipped removals */
		return rest_ensure_response(
			array(
				'success' => true,
				'results' => $results,
				'message' => sprintf(
					__( 'Local file removal completed. Success: %1$d, Failed: %2$d, Skipped: %3$d', 'swift-offload' ),
					$results['success'],
					$results['failed'],
					$results['skipped']
				),
			)
		);
	}

	/**
	 * Regenerate URLs for attachments
	 *
	 * @param \WP_REST_Request $request Request object
	 * @return \WP_REST_Response|\WP_Error
	 */
	public function regenerate_urls( $request ) {
		$attachment_ids = $request->get_param( 'attachment_ids' );

		if ( empty( $attachment_ids ) ) {
			return new WP_Error( 'no_attachments', __( 'No attachment IDs provided', 'swift-offload' ), array( 'status' => 400 ) );
		}

		$media_hooks = Plugin::get_instance()->get_media_hooks();
		$results     = array(
			'success' => 0,
			'failed'  => 0,
			'errors'  => array(),
		);

		foreach ( $attachment_ids as $attachment_id ) {
			$validation = \SwiftOffload\Utils\Validators::validate_attachment_id( $attachment_id );
			if ( ! $validation['valid'] ) {
				++$results['failed'];
				$results['errors'][] = "Invalid attachment ID: {$attachment_id}";
				continue;
			}

			$success = $media_hooks->regenerate_urls( $attachment_id );

			if ( $success ) {
				++$results['success'];
			} else {
				++$results['failed'];
				$results['errors'][] = "Failed to regenerate URLs for attachment {$attachment_id}";
			}
		}

		/* translators: %1$d: number of successful regenerations, %2$d: number of failed regenerations */
		return rest_ensure_response(
			array(
				'success' => true,
				'results' => $results,
				'message' => sprintf(
					__( 'URL regeneration completed. Success: %1$d, Failed: %2$d', 'swift-offload' ),
					$results['success'],
					$results['failed']
				),
			)
		);
	}

	/**
	 * Check permissions
	 *
	 * @return bool|\WP_Error
	 */
	public function check_permissions() {
		if ( ! current_user_can( 'upload_files' ) ) {
			return new WP_Error( 'rest_forbidden', __( 'You do not have permission to manage media', 'swift-offload' ), array( 'status' => 403 ) );
		}

		return true;
	}
}
