<?php
/**
 * Jobs Routes Class
 *
 * @package SwiftOffload\Rest\Routes
 */

namespace SwiftOffload\Rest\Routes;

use WP_REST_Controller;
use WP_REST_Server;
use WP_Error;
use SwiftOffload\Jobs\Job_Runner;

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Jobs Routes handles job-related API endpoints
 */
class Routes_Jobs extends WP_REST_Controller {

	/**
	 * Namespace
	 *
	 * @var string
	 */
	protected $namespace = 'swift-offload/v1';

	/**
	 * Rest base
	 *
	 * @var string
	 */
	protected $rest_base = 'jobs';

	/**
	 * Job runner instance
	 *
	 * @var Job_Runner
	 */
	private $job_runner;

	/**
	 * Constructor
	 */
	public function __construct() {
		$this->job_runner = new Job_Runner();
	}

	/**
	 * Register routes
	 */
	public function register_routes() {
		// Create job
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => array( $this, 'create_job' ),
				'permission_callback' => array( $this, 'check_permissions' ),
				'args'                => array(
					'job_type' => array(
						'required' => true,
						'type'     => 'string',
						'enum'     => array( 'offload', 'remove_local', 'backfill', 'rewrite' ),
					),
					'job_data' => array(
						'type'    => 'object',
						'default' => array(),
					),
				),
			)
		);

		// Get job
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>\d+)',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_job' ),
				'permission_callback' => array( $this, 'check_permissions' ),
				'args'                => array(
					'id' => array(
						'validate_callback' => function ( $param ) {
							return is_numeric( $param );
						},
					),
				),
			)
		);

		// Get jobs list
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_jobs' ),
				'permission_callback' => array( $this, 'check_permissions' ),
				'args'                => array(
					'status' => array(
						'type' => 'string',
						'enum' => array( 'pending', 'running', 'completed', 'failed', 'cancelled', 'paused' ),
					),
					'limit'  => array(
						'type'    => 'integer',
						'default' => 20,
						'minimum' => 1,
						'maximum' => 100,
					),
				),
			)
		);

		// Control job (pause, resume, cancel)
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>\d+)/control',
			array(
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => array( $this, 'control_job' ),
				'permission_callback' => array( $this, 'check_permissions' ),
				'args'                => array(
					'id'     => array(
						'validate_callback' => function ( $param ) {
							return is_numeric( $param );
						},
					),
					'action' => array(
						'required' => true,
						'type'     => 'string',
						'enum'     => array( 'pause', 'resume', 'cancel' ),
					),
				),
			)
		);

		// Get job progress
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>\d+)/progress',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_job_progress' ),
				'permission_callback' => array( $this, 'check_permissions' ),
				'args'                => array(
					'id' => array(
						'validate_callback' => function ( $param ) {
							return is_numeric( $param );
						},
					),
				),
			)
		);
	}

	/**
	 * Create a new job
	 *
	 * @param \WP_REST_Request $request Request object
	 * @return \WP_REST_Response|\WP_Error
	 */
	public function create_job( $request ) {
		$job_type = $request->get_param( 'job_type' );
		$job_data = $request->get_param( 'job_data' );

		// Validate job data
		$validation = \SwiftOffload\Utils\Validators::validate_job_data( $job_data );
		if ( ! $validation['valid'] ) {
			return new WP_Error( 'invalid_job_data', implode( ', ', $validation['errors'] ), array( 'status' => 400 ) );
		}

		// Check rate limit
		if ( ! \SwiftOffload\Utils\Validators::check_rate_limit( 'create_job', 10 ) ) {
			return new WP_Error( 'rate_limit', __( 'Too many job creation requests', 'swift-offload' ), array( 'status' => 429 ) );
		}

		$job_id = $this->job_runner->create_job( $job_type, $job_data );

		if ( ! $job_id ) {
			return new WP_Error( 'job_creation_failed', __( 'Failed to create job', 'swift-offload' ), array( 'status' => 500 ) );
		}

		$job = $this->job_runner->get_job( $job_id );

		return rest_ensure_response(
			array(
				'success' => true,
				'job_id'  => $job_id,
				'job'     => $job,
				'message' => __( 'Job created successfully', 'swift-offload' ),
			)
		);
	}

	/**
	 * Get job details
	 *
	 * @param \WP_REST_Request $request Request object
	 * @return \WP_REST_Response|\WP_Error
	 */
	public function get_job( $request ) {
		$job_id = (int) $request->get_param( 'id' );
		$job    = $this->job_runner->get_job( $job_id );

		if ( ! $job ) {
			return new WP_Error( 'job_not_found', __( 'Job not found', 'swift-offload' ), array( 'status' => 404 ) );
		}

		// Get detailed progress
		$progress = $this->job_runner->get_job_progress( $job_id );

		return rest_ensure_response(
			array(
				'job'      => $job,
				'progress' => $progress,
			)
		);
	}

	/**
	 * Get jobs list
	 *
	 * @param \WP_REST_Request $request Request object
	 * @return \WP_REST_Response
	 */
	public function get_jobs( $request ) {
		$status = $request->get_param( 'status' );
		$limit  = (int) $request->get_param( 'limit' );

		global $wpdb;
		$table = $wpdb->prefix . 'swift_offload_jobs';

		$where_clause = '';
		$values       = array();

		if ( $status ) {
			$where_clause = ' WHERE status = %s';
			$values[]     = $status;
		}

		$sql      = "SELECT * FROM {$table}{$where_clause} ORDER BY created_at DESC LIMIT %d";
		$values[] = $limit;

		$jobs = $wpdb->get_results( $wpdb->prepare( $sql, $values ), ARRAY_A );

		// Add progress data for each job
		foreach ( $jobs as &$job ) {
			$job['progress'] = $this->job_runner->get_job_progress( $job['id'] );
		}

		return rest_ensure_response( $jobs );
	}

	/**
	 * Control job (pause, resume, cancel)
	 *
	 * @param \WP_REST_Request $request Request object
	 * @return \WP_REST_Response|\WP_Error
	 */
	public function control_job( $request ) {
		$job_id = (int) $request->get_param( 'id' );
		$action = $request->get_param( 'action' );

		$job = $this->job_runner->get_job( $job_id );
		if ( ! $job ) {
			return new WP_Error( 'job_not_found', __( 'Job not found', 'swift-offload' ), array( 'status' => 404 ) );
		}

		$success = false;
		$message = '';

		switch ( $action ) {
			case 'pause':
				$success = $this->job_runner->pause_job( $job_id );
				$message = __( 'Job paused successfully', 'swift-offload' );
				break;

			case 'resume':
				$success = $this->job_runner->resume_job( $job_id );
				$message = __( 'Job resumed successfully', 'swift-offload' );
				break;

			case 'cancel':
				$success = $this->job_runner->cancel_job( $job_id );
				$message = __( 'Job cancelled successfully', 'swift-offload' );
				break;
		}

		if ( ! $success ) {
			return new WP_Error( 'job_control_failed', __( 'Failed to control job', 'swift-offload' ), array( 'status' => 500 ) );
		}

		// Get updated job data
		$updated_job = $this->job_runner->get_job( $job_id );

		return rest_ensure_response(
			array(
				'success' => true,
				'message' => $message,
				'job'     => $updated_job,
			)
		);
	}

	/**
	 * Get job progress
	 *
	 * @param \WP_REST_Request $request Request object
	 * @return \WP_REST_Response|\WP_Error
	 */
	public function get_job_progress( $request ) {
		$job_id = (int) $request->get_param( 'id' );

		$job = $this->job_runner->get_job( $job_id );
		if ( ! $job ) {
			return new WP_Error( 'job_not_found', __( 'Job not found', 'swift-offload' ), array( 'status' => 404 ) );
		}

		$progress = $this->job_runner->get_job_progress( $job_id );

		return rest_ensure_response( $progress );
	}

	/**
	 * Check permissions
	 *
	 * @return bool|\WP_Error
	 */
	public function check_permissions() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return new WP_Error( 'rest_forbidden', __( 'You do not have permission to manage jobs', 'swift-offload' ), array( 'status' => 403 ) );
		}

		return true;
	}
}
