<?php
/**
 * REST Controller Class
 *
 * @package SwiftOffload\Rest
 */

namespace SwiftOffload\Rest;

use WP_REST_Controller;
use WP_REST_Server;
use WP_Error;

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * REST Controller manages API routes
 */
class Rest_Controller extends WP_REST_Controller {

	/**
	 * Namespace
	 *
	 * @var string
	 */
	protected $namespace = 'swift-offload/v1';

	/**
	 * Route controllers
	 *
	 * @var array
	 */
	private $route_controllers = array();

	/**
	 * Constructor
	 */
	public function __construct() {
		$this->init_route_controllers();
	}

	/**
	 * Register REST API routes
	 */
	public function register_routes() {
		// Register routes from each controller
		foreach ( $this->route_controllers as $controller ) {
			$controller->register_routes();
		}

		// Register core routes
		$this->register_core_routes();
	}

	/**
	 * Initialize route controllers
	 */
	private function init_route_controllers() {
		$this->route_controllers = array(
			new Routes\Routes_Settings(),
			new Routes\Routes_Jobs(),
			new Routes\Routes_Media(),
		);
	}

	/**
	 * Register core routes
	 */
	private function register_core_routes() {
		// Health check endpoint
		register_rest_route(
			$this->namespace,
			'/health',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'health_check' ),
				'permission_callback' => array( $this, 'check_permissions' ),
			)
		);

		// Test storage connection
		register_rest_route(
			$this->namespace,
			'/test-connection',
			array(
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => array( $this, 'test_connection' ),
				'permission_callback' => array( $this, 'check_manage_permissions' ),
				'args'                => array(
					'provider' => array(
						'required' => true,
						'type'     => 'string',
						'enum'     => array( 's3' ),
					),
					'config'   => array(
						'required' => true,
						'type'     => 'object',
					),
				),
			)
		);

		// Get plugin info
		register_rest_route(
			$this->namespace,
			'/info',
			array(
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => array( $this, 'get_plugin_info' ),
				'permission_callback' => array( $this, 'check_permissions' ),
			)
		);
	}

	/**
	 * Health check endpoint
	 *
	 * @param \WP_REST_Request $request Request object
	 * @return array
	 */
	public function health_check( $request ) {
		$storage_registry = \SwiftOffload\Plugin::get_instance()->get_storage_registry();
		$provider         = $storage_registry->get_active_provider();

		$health = array(
			'status'            => 'ok',
			'version'           => SWIFT_OFFLOAD_VERSION,
			'timestamp'         => time(),
			'storage_connected' => false,
			'cdn_enabled'       => false,
		);

		// Check storage connection
		if ( $provider ) {
			$test_result                 = $provider->test_connection();
			$health['storage_connected'] = $test_result->is_success();
		}

		// Check CDN status
		$cdn_config            = swift_offload_get_cdn_config();
		$health['cdn_enabled'] = ! empty( $cdn_config['enabled'] );

		return rest_ensure_response( $health );
	}

	/**
	 * Test storage connection
	 *
	 * @param \WP_REST_Request $request Request object
	 * @return array|\WP_Error
	 */
	public function test_connection( $request ) {
		$provider = $request->get_param( 'provider' );
		$config   = $request->get_param( 'config' );

		if ( ! $config || ! is_array( $config ) ) {
			return new WP_Error( 'invalid_config', __( 'Invalid configuration provided', 'swift-offload' ), array( 'status' => 400 ) );
		}

		$storage_registry = \SwiftOffload\Plugin::get_instance()->get_storage_registry();
		$result           = $storage_registry->test_provider( $provider, $config );

		if ( $result['success'] ) {
			return rest_ensure_response(
				array(
					'success' => true,
					'message' => __( 'Connection successful', 'swift-offload' ),
					'data'    => $result['data'],
				)
			);
		} else {
			return new WP_Error( 'connection_failed', $result['error'], array( 'status' => 400 ) );
		}
	}

	/**
	 * Get plugin information
	 *
	 * @param \WP_REST_Request $request Request object
	 * @return array
	 */
	public function get_plugin_info( $request ) {
		global $wpdb;

		// Get statistics
		$items_table = $wpdb->prefix . 'swift_offload_items';
		$jobs_table  = $wpdb->prefix . 'swift_offload_jobs';

		$total_offloaded   = $wpdb->get_var( "SELECT COUNT(*) FROM {$items_table}" );
		$total_attachments = $wpdb->get_var( "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = 'attachment'" );
		$pending_jobs      = $wpdb->get_var( "SELECT COUNT(*) FROM {$jobs_table} WHERE status = 'pending'" );
		$running_jobs      = $wpdb->get_var( "SELECT COUNT(*) FROM {$jobs_table} WHERE status = 'running'" );

		// Get storage usage
		$storage_size = $wpdb->get_var(
			"
            SELECT SUM(CAST(JSON_UNQUOTE(JSON_EXTRACT(meta_data, '$.size')) AS UNSIGNED))
            FROM {$items_table} 
            WHERE meta_data IS NOT NULL
        "
		);

		return rest_ensure_response(
			array(
				'version'     => SWIFT_OFFLOAD_VERSION,
				'php_version' => PHP_VERSION,
				'wp_version'  => get_bloginfo( 'version' ),
				'statistics'  => array(
					'total_attachments'  => (int) $total_attachments,
					'total_offloaded'    => (int) $total_offloaded,
					'offload_percentage' => $total_attachments > 0 ? round( ( $total_offloaded / $total_attachments ) * 100, 2 ) : 0,
					'storage_size'       => (int) $storage_size,
					'pending_jobs'       => (int) $pending_jobs,
					'running_jobs'       => (int) $running_jobs,
				),
				'config'      => array(
					'storage'  => swift_offload_get_storage_config(),
					'cdn'      => swift_offload_get_cdn_config(),
					'settings' => get_option( 'swift_offload_settings', array() ),
				),
			)
		);
	}

	/**
	 * Check basic permissions
	 *
	 * @param \WP_REST_Request $request Request object
	 * @return bool|\WP_Error
	 */
	public function check_permissions( $request ) {
		if ( ! is_user_logged_in() ) {
			return new WP_Error( 'rest_forbidden', __( 'You must be logged in', 'swift-offload' ), array( 'status' => 401 ) );
		}

		return true;
	}

	/**
	 * Check management permissions
	 *
	 * @param \WP_REST_Request $request Request object
	 * @return bool|\WP_Error
	 */
	public function check_manage_permissions( $request ) {
		if ( ! current_user_can( 'manage_options' ) ) {
			return new WP_Error( 'rest_forbidden', __( 'You do not have permission to manage Swift Offload', 'swift-offload' ), array( 'status' => 403 ) );
		}

		// Verify nonce
		$nonce = $request->get_header( 'X-WP-Nonce' );
		if ( ! wp_verify_nonce( $nonce, 'wp_rest' ) ) {
			return new WP_Error( 'rest_forbidden', __( 'Invalid nonce', 'swift-offload' ), array( 'status' => 403 ) );
		}

		return true;
	}

	/**
	 * Sanitize and validate request data
	 *
	 * @param array $data Request data
	 * @param array $rules Validation rules
	 * @return array|\WP_Error
	 */
	protected function validate_request_data( $data, $rules ) {
		$sanitized = array();
		$errors    = array();

		foreach ( $rules as $field => $rule ) {
			$value = $data[ $field ] ?? null;

			// Check required fields
			if ( ! empty( $rule['required'] ) && ( null === $value || '' === $value ) ) {
				/* translators: %s: field name */
				$errors[] = sprintf( __( 'Field %s is required', 'swift-offload' ), $field );
				continue;
			}

			// Skip validation if field is not required and empty
			if ( null === $value || '' === $value ) {
				continue;
			}

			// Type validation
			if ( isset( $rule['type'] ) ) {
				switch ( $rule['type'] ) {
					case 'string':
						$sanitized[ $field ] = sanitize_text_field( $value );
						break;
					case 'email':
						$sanitized[ $field ] = sanitize_email( $value );
						if ( ! is_email( $sanitized[ $field ] ) ) {
							/* translators: %s: field name */
							$errors[] = sprintf( __( 'Field %s must be a valid email', 'swift-offload' ), $field );
						}
						break;
					case 'url':
						$sanitized[ $field ] = esc_url_raw( $value );
						break;
					case 'boolean':
						$sanitized[ $field ] = (bool) $value;
						break;
					case 'integer':
						$sanitized[ $field ] = (int) $value;
						break;
					case 'array':
						$sanitized[ $field ] = (array) $value;
						break;
					default:
						$sanitized[ $field ] = $value;
				}
			} else {
				$sanitized[ $field ] = $value;
			}

			// Enum validation
			if ( isset( $rule['enum'] ) && ! in_array( $sanitized[ $field ], $rule['enum'] ) ) {
				/* translators: %1$s: field name, %2$s: list of allowed values */
				$errors[] = sprintf( __( 'Field %1$s must be one of: %2$s', 'swift-offload' ), $field, implode( ', ', $rule['enum'] ) );
			}

			// Length validation
			if ( isset( $rule['min_length'] ) && strlen( $sanitized[ $field ] ) < $rule['min_length'] ) {
				/* translators: %1$s: field name, %2$d: minimum character count */
				$errors[] = sprintf( __( 'Field %1$s must be at least %2$d characters', 'swift-offload' ), $field, $rule['min_length'] );
			}

			if ( isset( $rule['max_length'] ) && strlen( $sanitized[ $field ] ) > $rule['max_length'] ) {
				/* translators: %1$s: field name, %2$d: maximum character count */
				$errors[] = sprintf( __( 'Field %1$s must be no more than %2$d characters', 'swift-offload' ), $field, $rule['max_length'] );
			}
		}

		if ( ! empty( $errors ) ) {
			return new WP_Error( 'validation_failed', implode( ', ', $errors ), array( 'status' => 400 ) );
		}

		return $sanitized;
	}

	/**
	 * Send error response
	 *
	 * @param string $code Error code
	 * @param string $message Error message
	 * @param int    $status HTTP status code
	 * @return \WP_Error
	 */
	protected function send_error( $code, $message, $status = 400 ) {
		return new WP_Error( $code, $message, array( 'status' => $status ) );
	}

	/**
	 * Send success response
	 *
	 * @param mixed $data Response data
	 * @param int   $status HTTP status code
	 * @return \WP_REST_Response
	 */
	protected function send_success( $data, $status = 200 ) {
		return rest_ensure_response( $data );
	}
}
