<?php
/**
 * Job Runner Class
 *
 * @package SwiftOffload\Jobs
 */

namespace SwiftOffload\Jobs;

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Job Runner handles background job processing
 */
class Job_Runner {

	/**
	 * Maximum execution time per batch (seconds)
	 */
	const MAX_EXECUTION_TIME = 45;

	/**
	 * Maximum memory usage percentage
	 */
	const MAX_MEMORY_USAGE = 0.8;

	/**
	 * Batch size for processing
	 */
	const BATCH_SIZE = 10;

	/**
	 * Process pending jobs
	 */
	public function process_pending_jobs() {
		$start_time   = time();
		$start_memory = memory_get_usage();

		while ( time() - $start_time < self::MAX_EXECUTION_TIME ) {
			$job = $this->get_next_pending_job();

			if ( ! $job ) {
				break; // No more pending jobs
			}

			$this->process_job( $job );

			// Check memory usage
			$current_memory = memory_get_usage();
			$memory_limit   = $this->get_memory_limit();

			if ( $current_memory > ( $memory_limit * self::MAX_MEMORY_USAGE ) ) {
				swift_offload_log( 'Job processing stopped due to memory limit', 'warning' );
				break;
			}
		}
	}

	/**
	 * Create a new job
	 *
	 * @param string $job_type Job type
	 * @param array  $job_data Job data
	 * @return int|false Job ID
	 */
	public function create_job( $job_type, $job_data = array() ) {
		global $wpdb;

		$table = $wpdb->prefix . 'swift_offload_jobs';

		$insert_data = array(
			'job_type'   => $job_type,
			'status'     => 'pending',
			'job_data'   => maybe_serialize( $job_data ),
			'created_at' => current_time( 'mysql' ),
		);

		$result = $wpdb->insert( $table, $insert_data, array( '%s', '%s', '%s', '%s' ) );

		if ( $result ) {
			$job_id = $wpdb->insert_id;
			swift_offload_log( "Created job {$job_id} of type {$job_type}", 'info' );
			return $job_id;
		}

		return false;
	}

	/**
	 * Get job by ID
	 *
	 * @param int $job_id Job ID
	 * @return array|null
	 */
	public function get_job( $job_id ) {
		global $wpdb;

		$table = $wpdb->prefix . 'swift_offload_jobs';

		return $wpdb->get_row(
			$wpdb->prepare( "SELECT * FROM {$table} WHERE id = %d", $job_id ),
			ARRAY_A
		);
	}

	/**
	 * Update job status
	 *
	 * @param int    $job_id Job ID
	 * @param string $status New status
	 * @param array  $data Additional data to update
	 * @return bool
	 */
	public function update_job_status( $job_id, $status, $data = array() ) {
		global $wpdb;

		$table = $wpdb->prefix . 'swift_offload_jobs';

		$update_data = array(
			'status'     => $status,
			'updated_at' => current_time( 'mysql' ),
		);

		if ( in_array( $status, array( 'running', 'processing' ) ) ) {
			$update_data['started_at'] = current_time( 'mysql' );
		} elseif ( in_array( $status, array( 'completed', 'failed', 'cancelled' ) ) ) {
			$update_data['completed_at'] = current_time( 'mysql' );
		}

		// Merge additional data
		foreach ( $data as $key => $value ) {
			if ( in_array( $key, array( 'total_items', 'processed_items', 'succeeded_items', 'failed_items', 'error_data' ) ) ) {
				if ( in_array( $key, array( 'job_data', 'error_data' ) ) ) {
					$update_data[ $key ] = maybe_serialize( $value );
				} else {
					$update_data[ $key ] = $value;
				}
			}
		}

		$format = array_fill( 0, count( $update_data ), '%s' );
		$format[ array_search( 'total_items', array_keys( $update_data ) ) ]     = '%d';
		$format[ array_search( 'processed_items', array_keys( $update_data ) ) ] = '%d';
		$format[ array_search( 'succeeded_items', array_keys( $update_data ) ) ] = '%d';
		$format[ array_search( 'failed_items', array_keys( $update_data ) ) ]    = '%d';

		return $wpdb->update( $table, $update_data, array( 'id' => $job_id ), $format, array( '%d' ) );
	}

	/**
	 * Process a single job
	 *
	 * @param array $job Job data
	 */
	private function process_job( $job ) {
		$job_id   = $job['id'];
		$job_type = $job['job_type'];

		// Update status to running
		$this->update_job_status( $job_id, 'running' );

		try {
			$processor = $this->get_job_processor( $job_type );

			if ( ! $processor ) {
				throw new \Exception( "No processor found for job type: {$job_type}" );
			}

			$result = $processor->process( $job );

			if ( $result['completed'] ) {
				$this->update_job_status( $job_id, 'completed', $result );
			} else {
				$this->update_job_status( $job_id, 'paused', $result );
			}
		} catch ( \Exception $e ) {
			$this->update_job_status(
				$job_id,
				'failed',
				array(
					'error_data' => $e->getMessage(),
				)
			);

			swift_offload_log( "Job {$job_id} failed: " . $e->getMessage(), 'error' );
		}
	}

	/**
	 * Get next pending job
	 *
	 * @return array|null
	 */
	private function get_next_pending_job() {
		global $wpdb;

		$table = $wpdb->prefix . 'swift_offload_jobs';

		return $wpdb->get_row(
			"SELECT * FROM {$table} WHERE status IN ('pending', 'paused') ORDER BY created_at ASC LIMIT 1",
			ARRAY_A
		);
	}

	/**
	 * Get job processor instance
	 *
	 * @param string $job_type Job type
	 * @return object|null
	 */
	private function get_job_processor( $job_type ) {
		$processors = array(
			'offload'      => 'SwiftOffload\\Jobs\\Job_Offload',
			'remove_local' => 'SwiftOffload\\Jobs\\Job_Remove_Local',
			'backfill'     => 'SwiftOffload\\Jobs\\Job_Backfill',
			'rewrite'      => 'SwiftOffload\\Jobs\\Job_Rewrite',
		);

		if ( ! isset( $processors[ $job_type ] ) ) {
			return null;
		}

		$class_name = $processors[ $job_type ];

		if ( ! class_exists( $class_name ) ) {
			return null;
		}

		return new $class_name();
	}

	/**
	 * Pause job
	 *
	 * @param int $job_id Job ID
	 * @return bool
	 */
	public function pause_job( $job_id ) {
		return $this->update_job_status( $job_id, 'paused' );
	}

	/**
	 * Resume job
	 *
	 * @param int $job_id Job ID
	 * @return bool
	 */
	public function resume_job( $job_id ) {
		return $this->update_job_status( $job_id, 'pending' );
	}

	/**
	 * Cancel job
	 *
	 * @param int $job_id Job ID
	 * @return bool
	 */
	public function cancel_job( $job_id ) {
		return $this->update_job_status( $job_id, 'cancelled' );
	}

	/**
	 * Get job progress
	 *
	 * @param int $job_id Job ID
	 * @return array
	 */
	public function get_job_progress( $job_id ) {
		$job = $this->get_job( $job_id );

		if ( ! $job ) {
			return array();
		}

		$total     = (int) $job['total_items'];
		$processed = (int) $job['processed_items'];
		$succeeded = (int) $job['succeeded_items'];
		$failed    = (int) $job['failed_items'];

		$percentage = $total > 0 ? round( ( $processed / $total ) * 100, 2 ) : 0;

		// Estimate time remaining
		$eta = null;
		if ( $processed > 0 && $job['started_at'] ) {
			$start_time = strtotime( $job['started_at'] );
			$elapsed    = time() - $start_time;
			$rate       = $processed / $elapsed; // items per second

			if ( $rate > 0 ) {
				$remaining_items = $total - $processed;
				$eta             = $remaining_items / $rate; // seconds
			}
		}

		return array(
			'job_id'          => $job_id,
			'job_type'        => $job['job_type'],
			'status'          => $job['status'],
			'total_items'     => $total,
			'processed_items' => $processed,
			'succeeded_items' => $succeeded,
			'failed_items'    => $failed,
			'percentage'      => $percentage,
			'eta_seconds'     => $eta,
			'started_at'      => $job['started_at'],
			'updated_at'      => $job['updated_at'],
		);
	}

	/**
	 * Get recent jobs
	 *
	 * @param int $limit Number of jobs to return
	 * @return array
	 */
	public function get_recent_jobs( $limit = 10 ) {
		global $wpdb;

		$table = $wpdb->prefix . 'swift_offload_jobs';

		return $wpdb->get_results(
			$wpdb->prepare(
				"SELECT * FROM {$table} ORDER BY created_at DESC LIMIT %d",
				$limit
			),
			ARRAY_A
		);
	}

	/**
	 * Clean up old completed jobs
	 *
	 * @param int $days_old Number of days old
	 * @return int Number of deleted jobs
	 */
	public function cleanup_old_jobs( $days_old = 30 ) {
		global $wpdb;

		$table = $wpdb->prefix . 'swift_offload_jobs';

		$cutoff_date = date( 'Y-m-d H:i:s', strtotime( "-{$days_old} days" ) );

		return $wpdb->query(
			$wpdb->prepare(
				"DELETE FROM {$table} WHERE status IN ('completed', 'failed', 'cancelled') AND completed_at < %s",
				$cutoff_date
			)
		);
	}

	/**
	 * Get PHP memory limit in bytes
	 *
	 * @return int
	 */
	private function get_memory_limit() {
		$memory_limit = ini_get( 'memory_limit' );

		if ( preg_match( '/^(\d+)(.)$/', $memory_limit, $matches ) ) {
			$number = (int) $matches[1];
			$unit   = strtoupper( $matches[2] );

			switch ( $unit ) {
				case 'G':
					return $number * 1024 * 1024 * 1024;
				case 'M':
					return $number * 1024 * 1024;
				case 'K':
					return $number * 1024;
				default:
					return $number;
			}
		}

		return 128 * 1024 * 1024; // Default 128MB
	}
}
