<?php
/**
 * URL Rewrite Job Class
 *
 * @package SwiftOffload\Jobs
 */

namespace SwiftOffload\Jobs;

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * URL Rewrite Job handles rewriting URLs in post content
 */
class Job_Rewrite {

	/**
	 * Process URL rewrite job
	 *
	 * @param array $job Job data
	 * @return array
	 */
	public function process( $job ) {
		$job_data  = maybe_unserialize( $job['job_data'] );
		$processed = (int) $job['processed_items'];
		$succeeded = (int) $job['succeeded_items'];
		$failed    = (int) $job['failed_items'];

		// Get parameters
		$post_types = $job_data['post_types'] ?? array( 'post', 'page' );
		$batch_size = $job_data['batch_size'] ?? 20;
		$dry_run    = $job_data['dry_run'] ?? false;

		// Get posts to process
		$post_ids = $this->get_posts_with_media_urls( $post_types );
		$total    = count( $post_ids );

		// Update total if not set
		if ( (int) $job['total_items'] === 0 ) {
			global $wpdb;
			$table = $wpdb->prefix . 'swift_offload_jobs';
			$wpdb->update( $table, array( 'total_items' => $total ), array( 'id' => $job['id'] ), array( '%d' ), array( '%d' ) );
		}

		// Get batch to process
		$batch = array_slice( $post_ids, $processed, $batch_size );

		if ( empty( $batch ) ) {
			return array(
				'completed'       => true,
				'processed_items' => $processed,
				'succeeded_items' => $succeeded,
				'failed_items'    => $failed,
			);
		}

		// Process batch
		foreach ( $batch as $post_id ) {
			$result = $this->rewrite_post_urls( $post_id, $dry_run );

			if ( $result['success'] ) {
				++$succeeded;
				if ( $result['changes_made'] > 0 ) {
					swift_offload_log( "Rewrote {$result['changes_made']} URLs in post {$post_id}", 'info' );
				}
			} else {
				++$failed;
				swift_offload_log( "Failed to rewrite URLs in post {$post_id}: " . $result['error'], 'error' );
			}

			++$processed;

			// Update progress periodically
			if ( $processed % 10 === 0 ) {
				$this->update_progress( $job['id'], $total, $processed, $succeeded, $failed );
			}
		}

		$completed = $processed >= $total;

		return array(
			'completed'       => $completed,
			'processed_items' => $processed,
			'succeeded_items' => $succeeded,
			'failed_items'    => $failed,
			'total_items'     => $total,
		);
	}

	/**
	 * Get posts containing media URLs
	 *
	 * @param array $post_types Post types to search
	 * @return array
	 */
	private function get_posts_with_media_urls( $post_types ) {
		global $wpdb;

		$upload_url  = wp_upload_dir()['baseurl'];
		$upload_host = wp_parse_url( $upload_url, PHP_URL_HOST );

		$post_types_placeholders = implode( ',', array_fill( 0, count( $post_types ), '%s' ) );

		$sql = "SELECT DISTINCT ID FROM {$wpdb->posts} 
                WHERE post_type IN ({$post_types_placeholders})
                AND post_status = 'publish'
                AND (post_content LIKE %s OR post_content LIKE %s)
                ORDER BY post_date DESC";

		$values = array_merge(
			$post_types,
			array( '%' . $wpdb->esc_like( $upload_host ) . '%', '%' . $wpdb->esc_like( '/wp-content/uploads/' ) . '%' )
		);

		return $wpdb->get_col( $wpdb->prepare( $sql, $values ) );
	}

	/**
	 * Rewrite URLs in a single post
	 *
	 * @param int  $post_id Post ID
	 * @param bool $dry_run Whether this is a dry run
	 * @return array
	 */
	private function rewrite_post_urls( $post_id, $dry_run = false ) {
		$post = get_post( $post_id );

		if ( ! $post ) {
			return array(
				'success'      => false,
				'error'        => 'Post not found',
				'changes_made' => 0,
			);
		}

		$content          = $post->post_content;
		$original_content = $content;
		$changes_made     = 0;

		// Get all offloaded attachments
		global $wpdb;
		$items_table     = $wpdb->prefix . 'swift_offload_items';
		$offloaded_items = $wpdb->get_results(
			"SELECT attachment_id, bucket, object_key FROM {$items_table}",
			ARRAY_A
		);

		// Build URL mapping
		$url_mapping = array();
		foreach ( $offloaded_items as $item ) {
			$attachment_id = $item['attachment_id'];
			$old_url       = wp_get_attachment_url( $attachment_id );
			$new_url       = swift_offload_get_attachment_s3_url( $attachment_id );

			if ( $old_url && $new_url && $old_url !== $new_url ) {
				$url_mapping[ $old_url ] = $new_url;

				// Also map image sizes
				$metadata = wp_get_attachment_metadata( $attachment_id );
				if ( $metadata && ! empty( $metadata['sizes'] ) ) {
					foreach ( $metadata['sizes'] as $size => $size_data ) {
						$size_old_url = wp_get_attachment_image_url( $attachment_id, $size );
						$size_new_url = swift_offload_get_attachment_s3_url( $attachment_id, $size );

						if ( $size_old_url && $size_new_url && $size_old_url !== $size_new_url ) {
							$url_mapping[ $size_old_url ] = $size_new_url;
						}
					}
				}
			}
		}

		// Replace URLs in content
		foreach ( $url_mapping as $old_url => $new_url ) {
			$old_count = substr_count( $content, $old_url );
			if ( $old_count > 0 ) {
				$content       = str_replace( $old_url, $new_url, $content );
				$changes_made += $old_count;
			}
		}

		// Save changes if not dry run and changes were made
		if ( ! $dry_run && $changes_made > 0 && $content !== $original_content ) {
			$update_result = wp_update_post(
				array(
					'ID'           => $post_id,
					'post_content' => $content,
				)
			);

			if ( is_wp_error( $update_result ) ) {
				return array(
					'success'      => false,
					'error'        => $update_result->get_error_message(),
					'changes_made' => 0,
				);
			}
		}

		return array(
			'success'      => true,
			'changes_made' => $changes_made,
			'dry_run'      => $dry_run,
		);
	}

	/**
	 * Update job progress
	 *
	 * @param int $job_id Job ID
	 * @param int $total Total items
	 * @param int $processed Processed items
	 * @param int $succeeded Succeeded items
	 * @param int $failed Failed items
	 */
	private function update_progress( $job_id, $total, $processed, $succeeded, $failed ) {
		global $wpdb;

		$table = $wpdb->prefix . 'swift_offload_jobs';

		$wpdb->update(
			$table,
			array(
				'total_items'     => $total,
				'processed_items' => $processed,
				'succeeded_items' => $succeeded,
				'failed_items'    => $failed,
				'updated_at'      => current_time( 'mysql' ),
			),
			array( 'id' => $job_id ),
			array( '%d', '%d', '%d', '%d', '%s' ),
			array( '%d' )
		);
	}
}
