<?php
/**
 * Remove Local Files Job Class
 *
 * @package SwiftOffload\Jobs
 */

namespace SwiftOffload\Jobs;

use SwiftOffload\Plugin;

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Remove Local Files Job removes local files after successful S3 offload
 */
class Job_Remove_Local {

	/**
	 * Process remove local files job
	 *
	 * @param array $job Job data
	 * @return array
	 */
	public function process( $job ) {
		$job_data  = maybe_unserialize( $job['job_data'] );
		$processed = (int) $job['processed_items'];
		$succeeded = (int) $job['succeeded_items'];
		$failed    = (int) $job['failed_items'];

		// Get attachment IDs to process
		$attachment_ids = $job_data['attachment_ids'] ?? array();
		$batch_size     = $job_data['batch_size'] ?? 10;
		$keep_count     = $job_data['keep_count'] ?? 0; // Keep X most recent files locally

		if ( empty( $attachment_ids ) ) {
			// Get all offloaded attachments
			$attachment_ids = $this->get_offloaded_attachments( $job_data );
		}

		$total = count( $attachment_ids );

		// Update total if not set
		if ( (int) $job['total_items'] === 0 ) {
			global $wpdb;
			$table = $wpdb->prefix . 'swift_offload_jobs';
			$wpdb->update( $table, array( 'total_items' => $total ), array( 'id' => $job['id'] ), array( '%d' ), array( '%d' ) );
		}

		// Apply keep count filter
		if ( $keep_count > 0 ) {
			// Sort by date and keep most recent
			$attachment_ids = $this->filter_by_keep_count( $attachment_ids, $keep_count );
			$total          = count( $attachment_ids );
		}

		// Get batch to process
		$batch = array_slice( $attachment_ids, $processed, $batch_size );

		if ( empty( $batch ) ) {
			return array(
				'completed'       => true,
				'processed_items' => $processed,
				'succeeded_items' => $succeeded,
				'failed_items'    => $failed,
			);
		}

		// Get media hooks instance
		$media_hooks = Plugin::get_instance()->get_media_hooks();

		// Process batch
		foreach ( $batch as $attachment_id ) {
			// Verify attachment is still offloaded
			if ( ! swift_offload_is_attachment_offloaded( $attachment_id ) ) {
				++$failed;
				++$processed;
				continue;
			}

			// Verify S3 object exists before removing local
			if ( ! $this->verify_s3_object_exists( $attachment_id ) ) {
				swift_offload_log( "S3 object verification failed for attachment {$attachment_id}, skipping local removal", 'warning' );
				++$failed;
				++$processed;
				continue;
			}

			$success = $media_hooks->remove_local_file( $attachment_id );

			if ( $success ) {
				++$succeeded;
				swift_offload_log( "Removed local file for attachment {$attachment_id}", 'info' );
			} else {
				++$failed;
				swift_offload_log( "Failed to remove local file for attachment {$attachment_id}", 'error' );
			}

			++$processed;

			// Update progress periodically
			if ( $processed % 5 === 0 ) {
				$this->update_progress( $job['id'], $total, $processed, $succeeded, $failed );
			}
		}

		$completed = $processed >= $total;

		return array(
			'completed'       => $completed,
			'processed_items' => $processed,
			'succeeded_items' => $succeeded,
			'failed_items'    => $failed,
			'total_items'     => $total,
		);
	}

	/**
	 * Get offloaded attachments
	 *
	 * @param array $criteria Search criteria
	 * @return array
	 */
	private function get_offloaded_attachments( $criteria = array() ) {
		global $wpdb;

		$items_table = $wpdb->prefix . 'swift_offload_items';
		$posts_table = $wpdb->posts;

		$where_conditions = array( 'ci.attachment_id IS NOT NULL' );
		$join_conditions  = array( "JOIN {$posts_table} p ON ci.attachment_id = p.ID" );

		// Filter by date range
		if ( ! empty( $criteria['date_from'] ) ) {
			$where_conditions[] = 'ci.offloaded_at >= %s';
		}

		if ( ! empty( $criteria['date_to'] ) ) {
			$where_conditions[] = 'ci.offloaded_at <= %s';
		}

		// Filter by provider
		if ( ! empty( $criteria['provider'] ) ) {
			$where_conditions[] = 'ci.provider = %s';
		}

		$sql = "SELECT ci.attachment_id FROM {$items_table} ci " .
				implode( ' ', $join_conditions ) .
				' WHERE ' . implode( ' AND ', $where_conditions ) .
				' ORDER BY ci.offloaded_at DESC';

		$values = array();
		if ( ! empty( $criteria['date_from'] ) ) {
			$values[] = $criteria['date_from'];
		}
		if ( ! empty( $criteria['date_to'] ) ) {
			$values[] = $criteria['date_to'];
		}
		if ( ! empty( $criteria['provider'] ) ) {
			$values[] = $criteria['provider'];
		}

		if ( ! empty( $values ) ) {
			$sql = $wpdb->prepare( $sql, $values );
		}

		return $wpdb->get_col( $sql );
	}

	/**
	 * Filter attachments by keep count
	 *
	 * @param array $attachment_ids Attachment IDs
	 * @param int   $keep_count Number of files to keep locally
	 * @return array
	 */
	private function filter_by_keep_count( $attachment_ids, $keep_count ) {
		global $wpdb;

		if ( $keep_count <= 0 || count( $attachment_ids ) <= $keep_count ) {
			return array();
		}

		// Get attachments ordered by date (newest first)
		$id_placeholders = implode( ', ', array_fill( 0, count( $attachment_ids ), '%d' ) );

		// phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare
		$sql         = $wpdb->prepare( "SELECT ID FROM {$wpdb->posts} WHERE ID IN ({$id_placeholders}) AND post_type = 'attachment' ORDER BY post_date DESC", $attachment_ids );
		$ordered_ids = $wpdb->get_col( $sql );

		// Return all except the most recent X files
		return array_slice( $ordered_ids, $keep_count );
	}

	/**
	 * Verify S3 object exists before removing local file
	 *
	 * @param int $attachment_id Attachment ID
	 * @return bool
	 */
	private function verify_s3_object_exists( $attachment_id ) {
		$s3_data = swift_offload_get_attachment_s3_data( $attachment_id );

		if ( ! $s3_data ) {
			return false;
		}

		$storage_registry = Plugin::get_instance()->get_storage_registry();
		$provider         = $storage_registry->get_active_provider();

		if ( ! $provider ) {
			return false;
		}

		// Check main object
		if ( ! $provider->object_exists( $s3_data['object_key'] ) ) {
			return false;
		}

		// Check image sizes if they exist
		if ( ! empty( $s3_data['size_data'] ) ) {
			$size_data = maybe_unserialize( $s3_data['size_data'] );

			foreach ( $size_data as $size => $size_info ) {
				if ( ! empty( $size_info['object_key'] ) && ! $provider->object_exists( $size_info['object_key'] ) ) {
					swift_offload_log( "Missing S3 object for attachment {$attachment_id} size {$size}", 'warning' );
					return false;
				}
			}
		}

		return true;
	}

	/**
	 * Update job progress
	 *
	 * @param int $job_id Job ID
	 * @param int $total Total items
	 * @param int $processed Processed items
	 * @param int $succeeded Succeeded items
	 * @param int $failed Failed items
	 */
	private function update_progress( $job_id, $total, $processed, $succeeded, $failed ) {
		global $wpdb;

		$table = $wpdb->prefix . 'swift_offload_jobs';

		$wpdb->update(
			$table,
			array(
				'total_items'     => $total,
				'processed_items' => $processed,
				'succeeded_items' => $succeeded,
				'failed_items'    => $failed,
				'updated_at'      => current_time( 'mysql' ),
			),
			array( 'id' => $job_id ),
			array( '%d', '%d', '%d', '%d', '%s' ),
			array( '%d' )
		);
	}
}
