<?php
/**
 * Offload Job Class
 *
 * @package SwiftOffload\Jobs
 */

namespace SwiftOffload\Jobs;

use SwiftOffload\Plugin;

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Offload Job processes bulk media offload operations
 */
class Job_Offload {

	/**
	 * Process offload job
	 *
	 * @param array $job Job data
	 * @return array
	 */
	public function process( $job ) {
		$job_data  = maybe_unserialize( $job['job_data'] );
		$processed = (int) $job['processed_items'];
		$succeeded = (int) $job['succeeded_items'];
		$failed    = (int) $job['failed_items'];

		// Get attachment IDs to process
		$attachment_ids = $job_data['attachment_ids'] ?? array();
		$batch_size     = $job_data['batch_size'] ?? 10;

		if ( empty( $attachment_ids ) ) {
			// If no specific IDs, get all unoffloaded attachments
			$attachment_ids = $this->get_unoffloaded_attachments( $job_data );
		}

		$total = count( $attachment_ids );

		// Update total if not set
		if ( (int) $job['total_items'] === 0 ) {
			global $wpdb;
			$table = $wpdb->prefix . 'swift_offload_jobs';
			$wpdb->update( $table, array( 'total_items' => $total ), array( 'id' => $job['id'] ), array( '%d' ), array( '%d' ) );
		}

		// Get batch to process
		$batch = array_slice( $attachment_ids, $processed, $batch_size );

		if ( empty( $batch ) ) {
			return array(
				'completed'       => true,
				'processed_items' => $processed,
				'succeeded_items' => $succeeded,
				'failed_items'    => $failed,
			);
		}

		// Get media hooks instance
		$media_hooks = Plugin::get_instance()->get_media_hooks();

		// Process batch
		foreach ( $batch as $attachment_id ) {
			$metadata = wp_get_attachment_metadata( $attachment_id );
			$success  = $media_hooks->offload_attachment( $attachment_id, $metadata );

			if ( $success ) {
				++$succeeded;
			} else {
				++$failed;
			}

			++$processed;

			// Update progress periodically
			if ( $processed % 5 === 0 ) {
				$this->update_progress( $job['id'], $total, $processed, $succeeded, $failed );
			}
		}

		$completed = $processed >= $total;

		return array(
			'completed'       => $completed,
			'processed_items' => $processed,
			'succeeded_items' => $succeeded,
			'failed_items'    => $failed,
			'total_items'     => $total,
		);
	}

	/**
	 * Get unoffloaded attachments
	 *
	 * @param array $criteria Search criteria
	 * @return array
	 */
	private function get_unoffloaded_attachments( $criteria = array() ) {
		global $wpdb;

		$where_conditions = array( "post_type = 'attachment'" );
		$join_conditions  = array();

		// Join with swift_offload_items to exclude already offloaded
		$items_table        = $wpdb->prefix . 'swift_offload_items';
		$join_conditions[]  = "LEFT JOIN {$items_table} ci ON p.ID = ci.attachment_id";
		$where_conditions[] = 'ci.attachment_id IS NULL';

		// Filter by mime type
		if ( ! empty( $criteria['mime_types'] ) ) {
			$mime_placeholders  = implode( ',', array_fill( 0, count( $criteria['mime_types'] ), '%s' ) );
			$where_conditions[] = "post_mime_type IN ({$mime_placeholders})";
		}

		// Filter by date range
		if ( ! empty( $criteria['date_from'] ) ) {
			$where_conditions[] = 'post_date >= %s';
		}

		if ( ! empty( $criteria['date_to'] ) ) {
			$where_conditions[] = 'post_date <= %s';
		}

		// Filter by file size
		if ( ! empty( $criteria['min_size'] ) || ! empty( $criteria['max_size'] ) ) {
			$join_conditions[] = "LEFT JOIN {$wpdb->postmeta} pm_size ON p.ID = pm_size.post_id AND pm_size.meta_key = '_wp_attachment_metadata'";
		}

		$sql = "SELECT p.ID FROM {$wpdb->posts} p " .
				implode( ' ', $join_conditions ) .
				' WHERE ' . implode( ' AND ', $where_conditions ) .
				' ORDER BY p.post_date DESC';

		$values = array();
		if ( ! empty( $criteria['mime_types'] ) ) {
			$values = array_merge( $values, $criteria['mime_types'] );
		}
		if ( ! empty( $criteria['date_from'] ) ) {
			$values[] = $criteria['date_from'];
		}
		if ( ! empty( $criteria['date_to'] ) ) {
			$values[] = $criteria['date_to'];
		}

		if ( ! empty( $values ) ) {
			$sql = $wpdb->prepare( $sql, $values );
		}

		return $wpdb->get_col( $sql );
	}

	/**
	 * Update job progress
	 *
	 * @param int $job_id Job ID
	 * @param int $total Total items
	 * @param int $processed Processed items
	 * @param int $succeeded Succeeded items
	 * @param int $failed Failed items
	 */
	private function update_progress( $job_id, $total, $processed, $succeeded, $failed ) {
		global $wpdb;

		$table = $wpdb->prefix . 'swift_offload_jobs';

		$wpdb->update(
			$table,
			array(
				'total_items'     => $total,
				'processed_items' => $processed,
				'succeeded_items' => $succeeded,
				'failed_items'    => $failed,
				'updated_at'      => current_time( 'mysql' ),
			),
			array( 'id' => $job_id ),
			array( '%d', '%d', '%d', '%d', '%s' ),
			array( '%d' )
		);
	}
}
