<?php
/**
 * Backfill Job Class
 *
 * @package SwiftOffload\Jobs
 */

namespace SwiftOffload\Jobs;

use SwiftOffload\Plugin;

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Backfill Job handles offloading existing media library files
 */
class Job_Backfill {

	/**
	 * Process backfill job
	 *
	 * @param array $job Job data
	 * @return array
	 */
	public function process( $job ) {
		$job_data  = maybe_unserialize( $job['job_data'] );
		$processed = (int) $job['processed_items'];
		$succeeded = (int) $job['succeeded_items'];
		$failed    = (int) $job['failed_items'];

		// Get criteria for backfill
		$criteria   = $job_data['criteria'] ?? array();
		$batch_size = $job_data['batch_size'] ?? 10;

		// Get attachments to backfill
		$attachment_ids = $this->get_backfill_attachments( $criteria );
		$total          = count( $attachment_ids );

		// Update total if not set
		if ( (int) $job['total_items'] === 0 ) {
			global $wpdb;
			$table = $wpdb->prefix . 'swift_offload_jobs';
			$wpdb->update( $table, array( 'total_items' => $total ), array( 'id' => $job['id'] ), array( '%d' ), array( '%d' ) );
		}

		// Get batch to process
		$batch = array_slice( $attachment_ids, $processed, $batch_size );

		if ( empty( $batch ) ) {
			return array(
				'completed'       => true,
				'processed_items' => $processed,
				'succeeded_items' => $succeeded,
				'failed_items'    => $failed,
			);
		}

		// Get media hooks instance
		$media_hooks = Plugin::get_instance()->get_media_hooks();

		// Process batch
		foreach ( $batch as $attachment_id ) {
			// Skip if already offloaded
			if ( swift_offload_is_attachment_offloaded( $attachment_id ) ) {
				++$processed;
				++$succeeded;
				continue;
			}

			$metadata = wp_get_attachment_metadata( $attachment_id );
			$success  = $media_hooks->offload_attachment( $attachment_id, $metadata );

			if ( $success ) {
				++$succeeded;
				swift_offload_log( "Backfilled attachment {$attachment_id}", 'info' );
			} else {
				++$failed;
				swift_offload_log( "Failed to backfill attachment {$attachment_id}", 'error' );
			}

			++$processed;

			// Update progress periodically
			if ( $processed % 5 === 0 ) {
				$this->update_progress( $job['id'], $total, $processed, $succeeded, $failed );
			}
		}

		$completed = $processed >= $total;

		return array(
			'completed'       => $completed,
			'processed_items' => $processed,
			'succeeded_items' => $succeeded,
			'failed_items'    => $failed,
			'total_items'     => $total,
		);
	}

	/**
	 * Get attachments for backfill based on criteria
	 *
	 * @param array $criteria Search criteria
	 * @return array
	 */
	private function get_backfill_attachments( $criteria = array() ) {
		global $wpdb;

		$where_conditions = array( "p.post_type = 'attachment'" );
		$join_conditions  = array();

		// Join with swift_offload_items to exclude already offloaded
		$items_table        = $wpdb->prefix . 'swift_offload_items';
		$join_conditions[]  = "LEFT JOIN {$items_table} ci ON p.ID = ci.attachment_id";
		$where_conditions[] = 'ci.attachment_id IS NULL';

		// Filter by mime type
		if ( ! empty( $criteria['mime_types'] ) ) {
			$mime_placeholders  = implode( ',', array_fill( 0, count( $criteria['mime_types'] ), '%s' ) );
			$where_conditions[] = "p.post_mime_type IN ({$mime_placeholders})";
		}

		// Filter by date range
		if ( ! empty( $criteria['date_from'] ) ) {
			$where_conditions[] = 'p.post_date >= %s';
		}

		if ( ! empty( $criteria['date_to'] ) ) {
			$where_conditions[] = 'p.post_date <= %s';
		}

		// Filter by author
		if ( ! empty( $criteria['author_ids'] ) ) {
			$author_placeholders = implode( ',', array_fill( 0, count( $criteria['author_ids'] ), '%d' ) );
			$where_conditions[]  = "p.post_author IN ({$author_placeholders})";
		}

		$sql = "SELECT p.ID FROM {$wpdb->posts} p " .
				implode( ' ', $join_conditions ) .
				' WHERE ' . implode( ' AND ', $where_conditions ) .
				' ORDER BY p.post_date ' . ( $criteria['order'] ?? 'ASC' );

		$values = array();
		if ( ! empty( $criteria['mime_types'] ) ) {
			$values = array_merge( $values, $criteria['mime_types'] );
		}
		if ( ! empty( $criteria['date_from'] ) ) {
			$values[] = $criteria['date_from'];
		}
		if ( ! empty( $criteria['date_to'] ) ) {
			$values[] = $criteria['date_to'];
		}
		if ( ! empty( $criteria['author_ids'] ) ) {
			$values = array_merge( $values, $criteria['author_ids'] );
		}

		if ( ! empty( $values ) ) {
			$sql = $wpdb->prepare( $sql, $values );
		}

		return $wpdb->get_col( $sql );
	}

	/**
	 * Update job progress
	 *
	 * @param int $job_id Job ID
	 * @param int $total Total items
	 * @param int $processed Processed items
	 * @param int $succeeded Succeeded items
	 * @param int $failed Failed items
	 */
	private function update_progress( $job_id, $total, $processed, $succeeded, $failed ) {
		global $wpdb;

		$table = $wpdb->prefix . 'swift_offload_jobs';

		$wpdb->update(
			$table,
			array(
				'total_items'     => $total,
				'processed_items' => $processed,
				'succeeded_items' => $succeeded,
				'failed_items'    => $failed,
				'updated_at'      => current_time( 'mysql' ),
			),
			array( 'id' => $job_id ),
			array( '%d', '%d', '%d', '%d', '%s' ),
			array( '%d' )
		);
	}
}
