<?php
/**
 * Storage Registry Class
 *
 * @package SwiftOffload
 */

namespace SwiftOffload;

use SwiftOffload\Storage\Storage_Interface;

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Storage Registry manages storage providers
 */
class Storage_Registry {

	/**
	 * Registered storage providers
	 *
	 * @var array
	 */
	private $providers = array();

	/**
	 * Active storage provider instance
	 *
	 * @var Storage_Interface
	 */
	private $active_provider;

	/**
	 * Register a storage provider
	 *
	 * @param string $id Provider ID
	 * @param string $class_name Provider class name
	 * @return bool
	 */
	public function register_provider( $id, $class_name ) {
		if ( ! class_exists( $class_name ) ) {
			return false;
		}

		$this->providers[ $id ] = $class_name;

		return true;
	}

	/**
	 * Get registered providers
	 *
	 * @return array
	 */
	public function get_providers() {
		return $this->providers;
	}

	/**
	 * Get provider class name
	 *
	 * @param string $id Provider ID
	 * @return string|false
	 */
	public function get_provider_class( $id ) {
		return isset( $this->providers[ $id ] ) ? $this->providers[ $id ] : false;
	}

	/**
	 * Get active storage provider
	 *
	 * @return Storage_Interface|null
	 */
	public function get_active_provider() {
		if ( null !== $this->active_provider ) {
			return $this->active_provider;
		}

		$storage_config = swift_offload_get_storage_config();
		$provider_id    = $storage_config['provider'] ?? 's3';

		if ( ! isset( $this->providers[ $provider_id ] ) ) {
			return null;
		}

		$class_name = $this->providers[ $provider_id ];
		$provider   = new $class_name();

		if ( ! $provider instanceof Storage_Interface ) {
			return null;
		}

		// Initialize provider with configuration
		if ( $provider->init( $storage_config ) ) {
			$this->active_provider = $provider;
			return $this->active_provider;
		}

		return null;
	}

	/**
	 * Create provider instance
	 *
	 * @param string $id Provider ID
	 * @param array  $config Provider configuration
	 * @return Storage_Interface|null
	 */
	public function create_provider( $id, $config = array() ) {
		if ( ! isset( $this->providers[ $id ] ) ) {
			return null;
		}

		$class_name = $this->providers[ $id ];
		$provider   = new $class_name();

		if ( ! $provider instanceof Storage_Interface ) {
			return null;
		}

		if ( empty( $config ) ) {
			$config = swift_offload_get_storage_config();
		}

		if ( $provider->init( $config ) ) {
			return $provider;
		}

		return null;
	}

	/**
	 * Test provider connection
	 *
	 * @param string $id Provider ID
	 * @param array  $config Provider configuration
	 * @return array
	 */
	public function test_provider( $id, $config ) {
		$provider = $this->create_provider( $id, $config );

		if ( ! $provider ) {
			return array(
				'success' => false,
				'error'   => __( 'Failed to initialize storage provider', 'swift-offload' ),
			);
		}

		$result = $provider->test_connection();

		return array(
			'success' => $result->is_success(),
			'error'   => $result->get_error(),
			'data'    => $result->to_array(),
		);
	}

	/**
	 * Get provider info
	 *
	 * @param string $id Provider ID
	 * @return array
	 */
	public function get_provider_info( $id ) {
		$providers_info = array(
			's3' => array(
				'name'            => __( 'Amazon S3', 'swift-offload' ),
				'description'     => __( 'Amazon Simple Storage Service', 'swift-offload' ),
				'icon'            => 'aws-s3',
				'fields'          => array( 'access_key', 'secret_key', 'region', 'bucket' ),
				'optional_fields' => array( 'endpoint', 'use_path_style' ),
			),
		);

		return $providers_info[ $id ] ?? array();
	}

	/**
	 * Reset active provider (force reload)
	 */
	public function reset_active_provider() {
		$this->active_provider = null;
	}
}
