<?php
/**
 * Main Plugin Class
 *
 * @package SwiftOffload
 */

namespace SwiftOffload;

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Main Plugin class
 */
class Plugin {

	/**
	 * Plugin instance
	 *
	 * @var Plugin
	 */
	private static $instance = null;

	/**
	 * Database version
	 */
	const DB_VERSION = '1.0.0';

	/**
	 * Storage registry instance
	 *
	 * @var Storage_Registry
	 */
	private $storage_registry;

	/**
	 * Media hooks instance
	 *
	 * @var Media_Hooks
	 */
	private $media_hooks;

	/**
	 * Assets manager instance
	 *
	 * @var Assets
	 */
	private $assets;

	/**
	 * Get plugin instance
	 *
	 * @return Plugin
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		$this->init_hooks();
		$this->init_components();
	}

	/**
	 * Initialize WordPress hooks
	 */
	private function init_hooks() {
		add_action( 'init', array( $this, 'init' ) );
		add_action( 'rest_api_init', array( $this, 'init_rest_api' ) );
		add_action( 'wp_loaded', array( $this, 'wp_loaded' ) );

		// Schedule background job processing
		add_action( 'swift_offload_process_jobs', array( $this, 'process_background_jobs' ) );

		if ( ! wp_next_scheduled( 'swift_offload_process_jobs' ) ) {
			wp_schedule_event( time(), 'hourly', 'swift_offload_process_jobs' );
		}
	}

	/**
	 * Initialize plugin components
	 */
	private function init_components() {
		// Initialize storage registry
		$this->storage_registry = new Storage_Registry();

		// Initialize media hooks
		$this->media_hooks = new Media_Hooks();

		// Initialize assets manager
		$this->assets = new Assets();
	}

	/**
	 * Initialize plugin
	 */
	public function init() {
		// Register storage providers
		$this->storage_registry->register_provider( 's3', 'SwiftOffload\\Storage\\Storage_S3' );

		// Load more components after init
		do_action( 'swift_offload_init', $this );
	}

	/**
	 * Initialize REST API
	 */
	public function init_rest_api() {
		$rest_controller = new Rest\Rest_Controller();
		$rest_controller->register_routes();
	}

	/**
	 * WordPress loaded hook
	 */
	public function wp_loaded() {
		// Plugin fully loaded
		do_action( 'swift_offload_loaded', $this );
	}

	/**
	 * Process background jobs
	 */
	public function process_background_jobs() {
		if ( class_exists( 'SwiftOffload\\Jobs\\Job_Runner' ) ) {
			$job_runner = new Jobs\Job_Runner();
			$job_runner->process_pending_jobs();
		}
	}

	/**
	 * Get storage registry
	 *
	 * @return Storage_Registry
	 */
	public function get_storage_registry() {
		return $this->storage_registry;
	}

	/**
	 * Get media hooks instance
	 *
	 * @return Media_Hooks
	 */
	public function get_media_hooks() {
		return $this->media_hooks;
	}

	/**
	 * Get assets manager
	 *
	 * @return Assets
	 */
	public function get_assets() {
		return $this->assets;
	}

	/**
	 * Plugin activation
	 */
	public static function activate() {
		// Create database tables
		self::create_tables();

		// Set default options
		self::set_default_options();

		// Flush rewrite rules
		flush_rewrite_rules();

		// Update DB version
		update_option( 'swift_offload_db_version', self::DB_VERSION );
		update_option( 'swift_offload_activation_time', time() );
	}

	/**
	 * Plugin deactivation
	 */
	public static function deactivate() {
		// Clear scheduled hooks
		wp_clear_scheduled_hook( 'swift_offload_process_jobs' );
		wp_clear_scheduled_hook( 'swift_offload_cleanup_logs' );

		// Flush rewrite rules
		flush_rewrite_rules();
	}

	/**
	 * Create database tables
	 */
	private static function create_tables() {
		global $wpdb;

		$charset_collate = $wpdb->get_charset_collate();

		// Items table for tracking offloaded media
		$items_table = $wpdb->prefix . 'swift_offload_items';
		$items_sql   = "CREATE TABLE $items_table (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            attachment_id bigint(20) unsigned NOT NULL,
            provider varchar(50) NOT NULL DEFAULT 's3',
            bucket varchar(255) NOT NULL,
            object_key varchar(1000) NOT NULL,
            region varchar(50) DEFAULT NULL,
            size_data longtext DEFAULT NULL,
            meta_data longtext DEFAULT NULL,
            offloaded_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY attachment_provider (attachment_id, provider),
            KEY bucket_key (bucket, object_key(191)),
            KEY offloaded_at (offloaded_at)
        ) $charset_collate;";

		// Jobs table for background processing
		$jobs_table = $wpdb->prefix . 'swift_offload_jobs';
		$jobs_sql   = "CREATE TABLE $jobs_table (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            job_type varchar(50) NOT NULL,
            status varchar(20) NOT NULL DEFAULT 'pending',
            total_items int(11) unsigned DEFAULT 0,
            processed_items int(11) unsigned DEFAULT 0,
            succeeded_items int(11) unsigned DEFAULT 0,
            failed_items int(11) unsigned DEFAULT 0,
            job_data longtext DEFAULT NULL,
            error_data longtext DEFAULT NULL,
            started_at datetime DEFAULT NULL,
            completed_at datetime DEFAULT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY status_type (status, job_type),
            KEY created_at (created_at)
        ) $charset_collate;";

		// Settings table for configuration
		$settings_table = $wpdb->prefix . 'swift_offload_settings';
		$settings_sql   = "CREATE TABLE $settings_table (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            setting_key varchar(255) NOT NULL,
            setting_value longtext NOT NULL,
            autoload varchar(3) NOT NULL DEFAULT 'yes',
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY setting_key (setting_key),
            KEY autoload (autoload)
        ) $charset_collate;";

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';

		dbDelta( $items_sql );
		dbDelta( $jobs_sql );
		dbDelta( $settings_sql );
	}

	/**
	 * Set default plugin options
	 */
	private static function set_default_options() {
		$defaults = array(
			'swift_offload_settings'       => array(
				'offload_on_upload'  => true,
				'remove_local_files' => false,
				'object_key_format'  => '{year}/{month}/{filename}',
				'storage_class'      => 'STANDARD',
				'object_acl'         => 'private',
			),
			'swift_offload_storage_config' => array(
				'provider'       => 's3',
				'access_key'     => '',
				'secret_key'     => '',
				'region'         => 'us-east-1',
				'bucket'         => '',
				'endpoint'       => '',
				'use_path_style' => false,
			),
			'swift_offload_cdn_config'     => array(
				'enabled'             => false,
				'distribution_domain' => '',
				'custom_domain'       => '',
				'signed_urls'         => false,
				'query_string_auth'   => false,
			),
		);

		foreach ( $defaults as $option_name => $default_value ) {
			if ( ! get_option( $option_name ) ) {
				add_option( $option_name, $default_value );
			}
		}
	}
}
