<?php
/**
 * Migration Manager Class
 *
 * @package SwiftOffload
 */

namespace SwiftOffload;

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Migration Manager handles importing from other S3 plugins
 */
class Migration_Manager {

	/**
	 * Supported source plugins
	 *
	 * @var array
	 */
	private $supported_plugins = array(
		'amazon-s3-and-cloudfront-pro' => array(
			'name'        => 'WP Offload Media (amazon-s3-and-cloudfront-pro)',
			'option_keys' => array(
				'tantan_wordpress_s3',
				'as3cf_settings',
			),
			'meta_keys'   => array(
				'amazonS3_info',
				'as3cf_object_info',
			),
		),
		'next3-offload'                => array(
			'name'        => 'Next3 Offload',
			'option_keys' => array(
				'next3_offload_settings',
				'next3_aws_config',
			),
			'meta_keys'   => array(
				'_next3_offload_info',
				'_next3_s3_key',
			),
		),
	);

	/**
	 * Detect available plugins for migration
	 *
	 * @return array
	 */
	public function detect_available_migrations() {
		$available = array();

		foreach ( $this->supported_plugins as $plugin_id => $plugin_info ) {
			$has_settings     = false;
			$has_meta         = false;
			$attachment_count = 0;

			// Check for settings
			foreach ( $plugin_info['option_keys'] as $option_key ) {
				if ( get_option( $option_key ) ) {
					$has_settings = true;
					break;
				}
			}

			// Check for attachment meta
			if ( $has_settings ) {
				global $wpdb;
				foreach ( $plugin_info['meta_keys'] as $meta_key ) {
					$count = $wpdb->get_var(
						$wpdb->prepare(
							"SELECT COUNT(DISTINCT post_id) FROM {$wpdb->postmeta} WHERE meta_key = %s",
							$meta_key
						)
					);

					if ( $count > 0 ) {
						$has_meta         = true;
						$attachment_count = max( $attachment_count, $count );
					}
				}
			}

			if ( $has_settings || $has_meta ) {
				$available[ $plugin_id ] = array(
					'name'             => $plugin_info['name'],
					'has_settings'     => $has_settings,
					'has_meta'         => $has_meta,
					'attachment_count' => $attachment_count,
				);
			}
		}

		return $available;
	}

	/**
	 * Preview migration data
	 *
	 * @param string $plugin_id Plugin ID to migrate from
	 * @return array
	 */
	public function preview_migration( $plugin_id ) {
		if ( ! isset( $this->supported_plugins[ $plugin_id ] ) ) {
			return array(
				'success' => false,
				'error'   => __( 'Unsupported plugin for migration', 'swift-offload' ),
			);
		}

		$plugin_info = $this->supported_plugins[ $plugin_id ];
		$preview     = array(
			'plugin_name'       => $plugin_info['name'],
			'settings'          => array(),
			'attachments'       => array(),
			'total_attachments' => 0,
		);

		// Preview settings migration
		foreach ( $plugin_info['option_keys'] as $option_key ) {
			$option_value = get_option( $option_key );
			if ( $option_value ) {
				$mapped_settings = $this->map_settings( $plugin_id, $option_key, $option_value );
				if ( $mapped_settings ) {
					$preview['settings'][ $option_key ] = $mapped_settings;
				}
			}
		}

		// Preview attachment meta migration
		global $wpdb;
		foreach ( $plugin_info['meta_keys'] as $meta_key ) {
			$meta_results = $wpdb->get_results(
				$wpdb->prepare(
					"SELECT post_id, meta_value FROM {$wpdb->postmeta} WHERE meta_key = %s LIMIT 10",
					$meta_key
				),
				ARRAY_A
			);

			if ( $meta_results ) {
				foreach ( $meta_results as $meta_row ) {
					$mapped_meta = $this->map_attachment_meta( $plugin_id, $meta_key, $meta_row['meta_value'] );
					if ( $mapped_meta ) {
						$preview['attachments'][] = array(
							'attachment_id' => $meta_row['post_id'],
							'original_meta' => $meta_row['meta_value'],
							'mapped_meta'   => $mapped_meta,
						);
					}
				}

				// Get total count
				$total_count                  = $wpdb->get_var(
					$wpdb->prepare(
						"SELECT COUNT(*) FROM {$wpdb->postmeta} WHERE meta_key = %s",
						$meta_key
					)
				);
				$preview['total_attachments'] = max( $preview['total_attachments'], $total_count );
			}
		}

		return array(
			'success' => true,
			'preview' => $preview,
		);
	}

	/**
	 * Execute migration
	 *
	 * @param string $plugin_id Plugin ID to migrate from
	 * @param array  $options Migration options
	 * @return array
	 */
	public function execute_migration( $plugin_id, $options = array() ) {
		if ( ! isset( $this->supported_plugins[ $plugin_id ] ) ) {
			return array(
				'success' => false,
				'error'   => __( 'Unsupported plugin for migration', 'swift-offload' ),
			);
		}

		$plugin_info = $this->supported_plugins[ $plugin_id ];
		$results     = array(
			'settings_migrated'    => 0,
			'attachments_migrated' => 0,
			'errors'               => array(),
		);

		// Migrate settings
		if ( ! empty( $options['migrate_settings'] ) ) {
			foreach ( $plugin_info['option_keys'] as $option_key ) {
				$option_value = get_option( $option_key );
				if ( $option_value ) {
					$mapped_settings = $this->map_settings( $plugin_id, $option_key, $option_value );
					if ( $mapped_settings ) {
						$this->save_migrated_settings( $mapped_settings );
						++$results['settings_migrated'];
					}
				}
			}
		}

		// Migrate attachment meta
		if ( ! empty( $options['migrate_attachments'] ) ) {
			foreach ( $plugin_info['meta_keys'] as $meta_key ) {
				$migration_result                 = $this->migrate_attachment_meta( $plugin_id, $meta_key, $options );
				$results['attachments_migrated'] += $migration_result['migrated'];
				$results['errors']                = array_merge( $results['errors'], $migration_result['errors'] );
			}
		}

		return array(
			'success' => true,
			'results' => $results,
		);
	}

	/**
	 * Map settings from source plugin to Swift Offload format
	 *
	 * @param string $plugin_id Plugin ID
	 * @param string $option_key Option key
	 * @param mixed  $option_value Option value
	 * @return array|null
	 */
	private function map_settings( $plugin_id, $option_key, $option_value ) {
		switch ( $plugin_id ) {
			case 'amazon-s3-and-cloudfront-pro':
				return $this->map_as3cf_settings( $option_key, $option_value );
			case 'next3-offload':
				return $this->map_next3_settings( $option_key, $option_value );
		}

		return null;
	}

	/**
	 * Map amazon-s3-and-cloudfront-pro settings
	 *
	 * @param string $option_key Option key
	 * @param mixed  $option_value Option value
	 * @return array|null
	 */
	private function map_as3cf_settings( $option_key, $option_value ) {
		if ( ! is_array( $option_value ) ) {
			return null;
		}

		$mapped = array();

		// Map storage config
		if ( isset( $option_value['access-key-id'] ) ) {
			$mapped['storage_config'] = array(
				'provider'   => 's3',
				'access_key' => $option_value['access-key-id'],
				'secret_key' => $option_value['secret-access-key'] ?? '',
				'bucket'     => $option_value['bucket'] ?? '',
				'region'     => $option_value['region'] ?? 'us-east-1',
			);
		}

		// Map general settings
		$mapped['settings'] = array(
			'offload_on_upload'  => ! empty( $option_value['copy-to-s3'] ),
			'remove_local_files' => ! empty( $option_value['serve-from-s3'] ),
			'object_key_format'  => $option_value['object-prefix'] ?? '{year}/{month}/{filename}',
		);

		// Map CDN config
		if ( ! empty( $option_value['cloudfront'] ) ) {
			$mapped['cdn_config'] = array(
				'enabled'             => true,
				'distribution_domain' => $option_value['domain'] ?? '',
			);
		}

		return $mapped;
	}

	/**
	 * Map next3-offload settings
	 *
	 * @param string $option_key Option key
	 * @param mixed  $option_value Option value
	 * @return array|null
	 */
	private function map_next3_settings( $option_key, $option_value ) {
		if ( ! is_array( $option_value ) ) {
			return null;
		}

		$mapped = array();

		// Map storage config
		if ( isset( $option_value['aws_access_key'] ) ) {
			$mapped['storage_config'] = array(
				'provider'   => 's3',
				'access_key' => $option_value['aws_access_key'],
				'secret_key' => $option_value['aws_secret_key'] ?? '',
				'bucket'     => $option_value['aws_bucket'] ?? '',
				'region'     => $option_value['aws_region'] ?? 'us-east-1',
			);
		}

		// Map general settings
		$mapped['settings'] = array(
			'offload_on_upload'  => ! empty( $option_value['auto_offload'] ),
			'remove_local_files' => ! empty( $option_value['remove_local'] ),
		);

		return $mapped;
	}

	/**
	 * Map attachment meta from source plugin
	 *
	 * @param string $plugin_id Plugin ID
	 * @param string $meta_key Meta key
	 * @param mixed  $meta_value Meta value
	 * @return array|null
	 */
	private function map_attachment_meta( $plugin_id, $meta_key, $meta_value ) {
		switch ( $plugin_id ) {
			case 'amazon-s3-and-cloudfront-pro':
				return $this->map_as3cf_attachment_meta( $meta_key, $meta_value );
			case 'next3-offload':
				return $this->map_next3_attachment_meta( $meta_key, $meta_value );
		}

		return null;
	}

	/**
	 * Map amazon-s3-and-cloudfront-pro attachment meta
	 *
	 * @param string $meta_key Meta key
	 * @param mixed  $meta_value Meta value
	 * @return array|null
	 */
	private function map_as3cf_attachment_meta( $meta_key, $meta_value ) {
		$meta_data = maybe_unserialize( $meta_value );

		if ( ! is_array( $meta_data ) ) {
			return null;
		}

		return array(
			'provider'   => 's3',
			'bucket'     => $meta_data['bucket'] ?? '',
			'object_key' => $meta_data['key'] ?? '',
			'region'     => $meta_data['region'] ?? '',
			'size_data'  => array(),
			'meta_data'  => $meta_data,
		);
	}

	/**
	 * Map next3-offload attachment meta
	 *
	 * @param string $meta_key Meta key
	 * @param mixed  $meta_value Meta value
	 * @return array|null
	 */
	private function map_next3_attachment_meta( $meta_key, $meta_value ) {
		$meta_data = maybe_unserialize( $meta_value );

		if ( ! is_array( $meta_data ) ) {
			return null;
		}

		return array(
			'provider'   => 's3',
			'bucket'     => $meta_data['bucket'] ?? '',
			'object_key' => $meta_data['key'] ?? $meta_data['object_key'] ?? '',
			'region'     => $meta_data['region'] ?? '',
			'size_data'  => $meta_data['sizes'] ?? array(),
			'meta_data'  => $meta_data,
		);
	}

	/**
	 * Save migrated settings
	 *
	 * @param array $mapped_settings Mapped settings
	 */
	private function save_migrated_settings( $mapped_settings ) {
		foreach ( $mapped_settings as $setting_type => $settings ) {
			$option_name      = 'swift_offload_' . $setting_type;
			$current_settings = get_option( $option_name, array() );
			$merged_settings  = array_merge( $current_settings, $settings );
			update_option( $option_name, $merged_settings );
		}
	}

	/**
	 * Migrate attachment meta for a specific meta key
	 *
	 * @param string $plugin_id Plugin ID
	 * @param string $meta_key Meta key
	 * @param array  $options Migration options
	 * @return array
	 */
	private function migrate_attachment_meta( $plugin_id, $meta_key, $options ) {
		global $wpdb;

		$batch_size = $options['batch_size'] ?? 50;
		$offset     = $options['offset'] ?? 0;
		$migrated   = 0;
		$errors     = array();

		// Get attachment meta in batches
		$meta_results = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT post_id, meta_value FROM {$wpdb->postmeta} 
                 WHERE meta_key = %s 
                 ORDER BY post_id ASC 
                 LIMIT %d OFFSET %d",
				$meta_key,
				$batch_size,
				$offset
			),
			ARRAY_A
		);

		$items_table = $wpdb->prefix . 'swift_offload_items';

		foreach ( $meta_results as $meta_row ) {
			$attachment_id = $meta_row['post_id'];

			// Skip if already migrated
			$existing = $wpdb->get_var(
				$wpdb->prepare(
					"SELECT id FROM {$items_table} WHERE attachment_id = %d",
					$attachment_id
				)
			);

			if ( $existing ) {
				continue;
			}

			$mapped_meta = $this->map_attachment_meta( $plugin_id, $meta_key, $meta_row['meta_value'] );

			if ( $mapped_meta ) {
				$insert_data = array(
					'attachment_id' => $attachment_id,
					'provider'      => $mapped_meta['provider'],
					'bucket'        => $mapped_meta['bucket'],
					'object_key'    => $mapped_meta['object_key'],
					'region'        => $mapped_meta['region'],
					'size_data'     => maybe_serialize( $mapped_meta['size_data'] ),
					'meta_data'     => maybe_serialize( $mapped_meta['meta_data'] ),
					'offloaded_at'  => current_time( 'mysql' ),
				);

				$result = $wpdb->insert(
					$items_table,
					$insert_data,
					array(
						'%d',
						'%s',
						'%s',
						'%s',
						'%s',
						'%s',
						'%s',
						'%s',
					)
				);

				if ( $result ) {
					++$migrated;
				} else {
					/* translators: %d: attachment ID */
					$errors[] = sprintf( __( 'Failed to migrate attachment %d', 'swift-offload' ), $attachment_id );
				}
			}
		}

		return array(
			'migrated' => $migrated,
			'errors'   => $errors,
		);
	}

	/**
	 * Get migration progress
	 *
	 * @param string $plugin_id Plugin ID
	 * @return array
	 */
	public function get_migration_progress( $plugin_id ) {
		if ( ! isset( $this->supported_plugins[ $plugin_id ] ) ) {
			return array();
		}

		global $wpdb;
		$plugin_info = $this->supported_plugins[ $plugin_id ];

		$total_meta     = 0;
		$migrated_count = 0;

		foreach ( $plugin_info['meta_keys'] as $meta_key ) {
			$total       = $wpdb->get_var(
				$wpdb->prepare(
					"SELECT COUNT(*) FROM {$wpdb->postmeta} WHERE meta_key = %s",
					$meta_key
				)
			);
			$total_meta += $total;
		}

		// Count migrated attachments
		$items_table    = $wpdb->prefix . 'swift_offload_items';
		$migrated_count = $wpdb->get_var( "SELECT COUNT(*) FROM {$items_table}" );

		return array(
			'total_meta'     => $total_meta,
			'migrated_count' => $migrated_count,
			'percentage'     => $total_meta > 0 ? round( ( $migrated_count / $total_meta ) * 100, 2 ) : 0,
		);
	}

	/**
	 * Clean up after migration
	 *
	 * @param string $plugin_id Plugin ID
	 * @param array  $options Cleanup options
	 * @return array
	 */
	public function cleanup_after_migration( $plugin_id, $options = array() ) {
		if ( ! isset( $this->supported_plugins[ $plugin_id ] ) ) {
			return array(
				'success' => false,
				'error'   => __( 'Unsupported plugin for cleanup', 'swift-offload' ),
			);
		}

		$cleaned = array();

		// Only clean up if explicitly requested
		if ( ! empty( $options['remove_source_data'] ) ) {
			$plugin_info = $this->supported_plugins[ $plugin_id ];

			// Remove source plugin options
			if ( ! empty( $options['remove_settings'] ) ) {
				foreach ( $plugin_info['option_keys'] as $option_key ) {
					if ( delete_option( $option_key ) ) {
						$cleaned[] = $option_key;
					}
				}
			}

			// Remove source plugin meta (WARNING: This is destructive)
			if ( ! empty( $options['remove_meta'] ) ) {
				global $wpdb;
				foreach ( $plugin_info['meta_keys'] as $meta_key ) {
					$deleted = $wpdb->query(
						$wpdb->prepare(
							"DELETE FROM {$wpdb->postmeta} WHERE meta_key = %s",
							$meta_key
						)
					);
					if ( $deleted ) {
						$cleaned[] = "{$meta_key} ({$deleted} records)";
					}
				}
			}
		}

		/* translators: %d: number of items cleaned */
		return array(
			'success' => true,
			'cleaned' => $cleaned,
			'message' => sprintf( __( 'Cleaned up %d items', 'swift-offload' ), count( $cleaned ) ),
		);
	}
}
