<?php
/**
 * Media Hooks Class
 *
 * @package SwiftOffload
 */

namespace SwiftOffload;

use SwiftOffload\Storage\Upload_Result;

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Media Hooks handles WordPress media events
 */
class Media_Hooks {

	/**
	 * Constructor
	 */
	public function __construct() {
		$this->init_hooks();
	}

	/**
	 * Initialize WordPress hooks
	 */
	private function init_hooks() {
		// Upload hooks
		add_filter( 'wp_handle_upload', array( $this, 'handle_upload' ), 10, 2 );
		add_action( 'add_attachment', array( $this, 'handle_add_attachment' ) );
		add_action( 'wp_generate_attachment_metadata', array( $this, 'handle_generate_metadata' ), 10, 2 );

		// Delete hooks
		add_action( 'delete_attachment', array( $this, 'handle_delete_attachment' ) );

		// URL filtering hooks
		add_filter( 'wp_get_attachment_url', array( $this, 'filter_attachment_url' ), 10, 2 );
		add_filter( 'wp_calculate_image_srcset', array( $this, 'filter_image_srcset' ), 10, 5 );

		// Attachment metadata hooks
		add_filter( 'wp_get_attachment_metadata', array( $this, 'filter_attachment_metadata' ), 10, 2 );

		// Image editor hooks
		add_action( 'wp_image_editor_after_change', array( $this, 'handle_image_edit' ), 10, 2 );
	}

	/**
	 * Handle file upload
	 *
	 * @param array $upload Upload data
	 * @param array $context Upload context
	 * @return array
	 */
	public function handle_upload( $upload, $context ) {
		// Only process if offload on upload is enabled
		if ( ! swift_offload_get_setting( 'offload_on_upload', true ) ) {
			return $upload;
		}

		// Skip if upload failed
		if ( ! empty( $upload['error'] ) ) {
			return $upload;
		}

		// Skip if this is not a media upload
		if ( empty( $context['action'] ) || $context['action'] !== 'wp_handle_upload' ) {
			return $upload;
		}

		// Add flag to indicate this should be offloaded
		$upload['swift_offload_offload'] = true;

		return $upload;
	}

	/**
	 * Handle attachment addition
	 *
	 * @param int $attachment_id Attachment ID
	 */
	public function handle_add_attachment( $attachment_id ) {
		// Check if we should offload this attachment
		$should_offload = $this->should_offload_attachment( $attachment_id );

		if ( ! $should_offload ) {
			return;
		}

		// Schedule immediate offload
		wp_schedule_single_event( time(), 'swift_offload_offload_attachment', array( $attachment_id ) );
	}

	/**
	 * Handle attachment metadata generation
	 *
	 * @param array $metadata Attachment metadata
	 * @param int   $attachment_id Attachment ID
	 * @return array
	 */
	public function handle_generate_metadata( $metadata, $attachment_id ) {
		// If attachment was marked for offload, process it now
		$upload_info = get_post_meta( $attachment_id, '_wp_attachment_backup_sizes', true );

		if ( is_array( $upload_info ) && isset( $upload_info['swift_offload_offload'] ) ) {
			$this->offload_attachment( $attachment_id, $metadata );
		}

		return $metadata;
	}

	/**
	 * Handle attachment deletion
	 *
	 * @param int $attachment_id Attachment ID
	 */
	public function handle_delete_attachment( $attachment_id ) {
		$s3_data = swift_offload_get_attachment_s3_data( $attachment_id );

		if ( ! $s3_data ) {
			return;
		}

		// Get storage provider
		$storage_registry = Plugin::get_instance()->get_storage_registry();
		$provider         = $storage_registry->get_active_provider();

		if ( ! $provider ) {
			return;
		}

		// Delete main object
		$provider->delete_object( $s3_data['object_key'] );

		// Delete image sizes if they exist
		if ( ! empty( $s3_data['size_data'] ) ) {
			$size_data = maybe_unserialize( $s3_data['size_data'] );
			foreach ( $size_data as $size => $size_info ) {
				if ( ! empty( $size_info['object_key'] ) ) {
					$provider->delete_object( $size_info['object_key'] );
				}
			}
		}

		// Remove database record
		global $wpdb;
		$table = $wpdb->prefix . 'swift_offload_items';
		$wpdb->delete( $table, array( 'attachment_id' => $attachment_id ), array( '%d' ) );

		swift_offload_log( "Deleted S3 objects for attachment {$attachment_id}", 'info' );
	}

	/**
	 * Filter attachment URL
	 *
	 * @param string $url Attachment URL
	 * @param int    $attachment_id Attachment ID
	 * @return string
	 */
	public function filter_attachment_url( $url, $attachment_id ) {
		$s3_url = swift_offload_get_attachment_s3_url( $attachment_id );

		if ( $s3_url ) {
			return $s3_url;
		}

		return $url;
	}

	/**
	 * Filter image srcset
	 *
	 * @param array  $sources Image sources
	 * @param array  $size_array Image size data
	 * @param string $image_src Image source URL
	 * @param array  $image_meta Image metadata
	 * @param int    $attachment_id Attachment ID
	 * @return array
	 */
	public function filter_image_srcset( $sources, $size_array, $image_src, $image_meta, $attachment_id ) {
		$s3_data = swift_offload_get_attachment_s3_data( $attachment_id );

		if ( ! $s3_data || empty( $s3_data['size_data'] ) ) {
			return $sources;
		}

		$size_data = maybe_unserialize( $s3_data['size_data'] );

		foreach ( $sources as $width => &$source ) {
			// Find matching size in our S3 data
			foreach ( $size_data as $size_name => $size_info ) {
				if ( isset( $size_info['width'] ) && $size_info['width'] == $width ) {
					$source['url'] = swift_offload_get_attachment_s3_url( $attachment_id, $size_name );
					break;
				}
			}
		}

		return $sources;
	}

	/**
	 * Filter attachment metadata
	 *
	 * @param array $metadata Attachment metadata
	 * @param int   $attachment_id Attachment ID
	 * @return array
	 */
	public function filter_attachment_metadata( $metadata, $attachment_id ) {
		// Add S3 status to metadata if needed
		if ( swift_offload_is_attachment_offloaded( $attachment_id ) ) {
			$metadata['swift_offload_offloaded'] = true;
		}

		return $metadata;
	}

	/**
	 * Handle image editing
	 *
	 * @param \WP_Image_Editor $editor Image editor instance
	 * @param array            $changes Changes made
	 */
	public function handle_image_edit( $editor, $changes ) {
		// Re-offload edited images if they were previously offloaded
		$attachment_id = $this->get_attachment_id_from_path( $editor->get_path() );

		if ( $attachment_id && swift_offload_is_attachment_offloaded( $attachment_id ) ) {
			// Schedule re-offload
			wp_schedule_single_event( time() + 10, 'swift_offload_reoffload_attachment', array( $attachment_id ) );
		}
	}

	/**
	 * Offload attachment to S3
	 *
	 * @param int   $attachment_id Attachment ID
	 * @param array $metadata Optional metadata
	 * @return bool
	 */
	public function offload_attachment( $attachment_id, $metadata = null ) {
		if ( swift_offload_is_attachment_offloaded( $attachment_id ) ) {
			return true; // Already offloaded
		}

		$storage_registry = Plugin::get_instance()->get_storage_registry();
		$provider         = $storage_registry->get_active_provider();

		if ( ! $provider ) {
			swift_offload_log( "No storage provider available for attachment {$attachment_id}", 'error' );
			return false;
		}

		$local_path = get_attached_file( $attachment_id );
		if ( ! $local_path || ! file_exists( $local_path ) ) {
			swift_offload_log( "Local file not found for attachment {$attachment_id}", 'error' );
			return false;
		}

		$filename   = basename( $local_path );
		$object_key = swift_offload_generate_object_key( $attachment_id, $filename );

		// Upload main file
		$upload_options = array(
			'content_type'  => get_post_mime_type( $attachment_id ),
			'acl'           => swift_offload_get_setting( 'object_acl', 'private' ),
			'storage_class' => swift_offload_get_setting( 'storage_class', 'STANDARD' ),
		);

		$result = $provider->upload_file( $local_path, $object_key, $upload_options );

		if ( ! $result->is_success() ) {
			swift_offload_log( "Failed to upload attachment {$attachment_id}: " . $result->get_error(), 'error' );
			return false;
		}

		// Upload image sizes if this is an image
		$size_data = array();
		if ( $metadata && ! empty( $metadata['sizes'] ) ) {
			$base_dir = dirname( $local_path );

			foreach ( $metadata['sizes'] as $size_name => $size_info ) {
				$size_file = $base_dir . '/' . $size_info['file'];

				if ( file_exists( $size_file ) ) {
					$size_object_key = dirname( $object_key ) . '/' . $size_info['file'];
					$size_result     = $provider->upload_file( $size_file, $size_object_key, $upload_options );

					if ( $size_result->is_success() ) {
						$size_data[ $size_name ] = array(
							'object_key' => $size_object_key,
							'width'      => $size_info['width'],
							'height'     => $size_info['height'],
							'file'       => $size_info['file'],
						);
					}
				}
			}
		}

		// Save to database
		$this->save_attachment_s3_data(
			$attachment_id,
			array(
				'provider'   => 's3',
				'bucket'     => swift_offload_get_storage_config( 'bucket' ),
				'object_key' => $object_key,
				'region'     => swift_offload_get_storage_config( 'region' ),
				'size_data'  => $size_data,
				'meta_data'  => $result->to_array(),
			)
		);

		// Remove local files if configured
		if ( swift_offload_get_setting( 'remove_local_files', false ) ) {
			wp_schedule_single_event( time() + 60, 'swift_offload_remove_local_file', array( $attachment_id ) );
		}

		swift_offload_log( "Successfully offloaded attachment {$attachment_id} to S3", 'info' );

		return true;
	}

	/**
	 * Check if attachment should be offloaded
	 *
	 * @param int $attachment_id Attachment ID
	 * @return bool
	 */
	private function should_offload_attachment( $attachment_id ) {
		// Check if offload is enabled
		if ( ! swift_offload_get_setting( 'offload_on_upload', true ) ) {
			return false;
		}

		$mime_type = get_post_mime_type( $attachment_id );
		$file_path = get_attached_file( $attachment_id );
		$file_size = file_exists( $file_path ) ? filesize( $file_path ) : 0;

		// Check mime type rules
		$allowed_mime_types = swift_offload_get_setting( 'allowed_mime_types', array() );
		if ( ! empty( $allowed_mime_types ) && ! in_array( $mime_type, $allowed_mime_types ) ) {
			return false;
		}

		// Check file size limits
		$min_file_size = swift_offload_get_setting( 'min_file_size', 0 );
		$max_file_size = swift_offload_get_setting( 'max_file_size', 0 );

		if ( $min_file_size > 0 && $file_size < $min_file_size ) {
			return false;
		}

		if ( $max_file_size > 0 && $file_size > $max_file_size ) {
			return false;
		}

		// Check upload date
		$upload_date       = get_the_date( 'Y-m-d', $attachment_id );
		$offload_from_date = swift_offload_get_setting( 'offload_from_date', '' );

		if ( ! empty( $offload_from_date ) && $upload_date < $offload_from_date ) {
			return false;
		}

		return apply_filters( 'swift_offload_should_offload_attachment', true, $attachment_id );
	}

	/**
	 * Save attachment S3 data to database
	 *
	 * @param int   $attachment_id Attachment ID
	 * @param array $data S3 data
	 * @return bool
	 */
	private function save_attachment_s3_data( $attachment_id, $data ) {
		global $wpdb;

		$table = $wpdb->prefix . 'swift_offload_items';

		$insert_data = array(
			'attachment_id' => $attachment_id,
			'provider'      => $data['provider'],
			'bucket'        => $data['bucket'],
			'object_key'    => $data['object_key'],
			'region'        => $data['region'],
			'size_data'     => maybe_serialize( $data['size_data'] ),
			'meta_data'     => maybe_serialize( $data['meta_data'] ),
			'offloaded_at'  => current_time( 'mysql' ),
		);

		$result = $wpdb->insert(
			$table,
			$insert_data,
			array(
				'%d',
				'%s',
				'%s',
				'%s',
				'%s',
				'%s',
				'%s',
				'%s',
			)
		);

		return $result !== false;
	}

	/**
	 * Get attachment ID from file path
	 *
	 * @param string $file_path File path
	 * @return int|false
	 */
	private function get_attachment_id_from_path( $file_path ) {
		global $wpdb;

		$sql           = "SELECT post_id FROM {$wpdb->postmeta} WHERE meta_key = '_wp_attached_file' AND meta_value = %s";
		$relative_path = str_replace( wp_upload_dir()['basedir'] . '/', '', $file_path );

		return $wpdb->get_var( $wpdb->prepare( $sql, $relative_path ) );
	}

	/**
	 * Remove local file after successful offload
	 *
	 * @param int $attachment_id Attachment ID
	 * @return bool
	 */
	public function remove_local_file( $attachment_id ) {
		if ( ! swift_offload_is_attachment_offloaded( $attachment_id ) ) {
			return false;
		}

		$local_path = get_attached_file( $attachment_id );
		$metadata   = wp_get_attachment_metadata( $attachment_id );

		if ( ! $local_path || ! file_exists( $local_path ) ) {
			return true; // Already removed
		}

		// Remove main file
		$removed = wp_delete_file( $local_path );

		// Remove image sizes
		if ( $metadata && ! empty( $metadata['sizes'] ) ) {
			$base_dir = dirname( $local_path );

			foreach ( $metadata['sizes'] as $size_info ) {
				$size_file = $base_dir . '/' . $size_info['file'];
				if ( file_exists( $size_file ) ) {
					wp_delete_file( $size_file );
				}
			}
		}

		if ( $removed ) {
			swift_offload_log( "Removed local files for attachment {$attachment_id}", 'info' );
		}

		return $removed;
	}

	/**
	 * Bulk offload attachments
	 *
	 * @param array $attachment_ids Attachment IDs
	 * @return array Results
	 */
	public function bulk_offload( $attachment_ids ) {
		$results = array(
			'success' => 0,
			'failed'  => 0,
			'skipped' => 0,
			'errors'  => array(),
		);

		foreach ( $attachment_ids as $attachment_id ) {
			if ( swift_offload_is_attachment_offloaded( $attachment_id ) ) {
				++$results['skipped'];
				continue;
			}

			$metadata = wp_get_attachment_metadata( $attachment_id );
			$success  = $this->offload_attachment( $attachment_id, $metadata );

			if ( $success ) {
				++$results['success'];
			} else {
				++$results['failed'];
				$results['errors'][] = $attachment_id;
			}

			// Prevent timeouts and memory issues
			if ( function_exists( 'wp_suspend_cache_invalidation' ) ) {
				wp_suspend_cache_invalidation( true );
			}

			// Clear object cache periodically
			if ( ( $results['success'] + $results['failed'] ) % 50 === 0 ) {
				wp_cache_flush();
			}
		}

		return $results;
	}

	/**
	 * Regenerate S3 URLs for attachment
	 *
	 * @param int $attachment_id Attachment ID
	 * @return bool
	 */
	public function regenerate_urls( $attachment_id ) {
		if ( ! swift_offload_is_attachment_offloaded( $attachment_id ) ) {
			return false;
		}

		// Clear any cached URLs
		clean_attachment_cache( $attachment_id );

		// Trigger URL regeneration
		wp_get_attachment_url( $attachment_id );

		return true;
	}

	/**
	 * Reprocess attachment after image regeneration
	 *
	 * @param int $attachment_id Attachment ID
	 * @return bool
	 */
	public function reprocess_attachment( $attachment_id ) {
		if ( ! swift_offload_is_attachment_offloaded( $attachment_id ) ) {
			return false;
		}

		// Get updated metadata
		$metadata = wp_get_attachment_metadata( $attachment_id );

		if ( ! $metadata ) {
			return false;
		}

		// Re-offload with new sizes
		return $this->offload_attachment( $attachment_id, $metadata );
	}
}
