<?php
/**
 * Assets Manager Class
 *
 * @package SwiftOffload
 */

namespace SwiftOffload;

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Assets manager class
 */
class Assets {

	/**
	 * Constructor
	 */
	public function __construct() {
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_scripts' ) );
		add_action( 'wp_enqueue_scripts', array( $this, 'frontend_scripts' ) );
	}

	/**
	 * Enqueue admin scripts and styles
	 */
	public function admin_scripts( $hook ) {
		// Only load on Swift Offload admin pages
		if ( strpos( $hook, 'swift-offload' ) === false && $hook !== 'settings_page_swift-offload' ) {
			return;
		}

		// Enqueue admin CSS
		wp_enqueue_style(
			'swift-offload-admin',
			SWIFT_OFFLOAD_PLUGIN_URL . 'assets/admin.css',
			array(),
			SWIFT_OFFLOAD_VERSION
		);

		// Enqueue React app for modern admin interface
		$this->enqueue_react_app();
	}

	/**
	 * Enqueue frontend scripts (if needed)
	 */
	public function frontend_scripts() {
		// Frontend scripts would go here if needed
		// Currently no frontend scripts required
	}

	/**
	 * Enqueue React admin application
	 */
	private function enqueue_react_app() {
		$app_js  = SWIFT_OFFLOAD_PLUGIN_URL . 'app/dist/main.js';
		$app_css = SWIFT_OFFLOAD_PLUGIN_URL . 'app/dist/main.css';

		// Check if built files exist
		if ( file_exists( SWIFT_OFFLOAD_PLUGIN_DIR . 'app/dist/main.js' ) ) {
			wp_enqueue_script(
				'swift-offload-app',
				$app_js,
				array( 'wp-element', 'wp-api-fetch' ),
				SWIFT_OFFLOAD_VERSION,
				true
			);

			// Localize script with necessary data
			wp_localize_script(
				'swift-offload-app',
				'swiftOffloadAdmin',
				array(
					'nonce'       => wp_create_nonce( 'wp_rest' ),
					'restUrl'     => rest_url( 'swift-offload/v1/' ),
					'adminUrl'    => admin_url( 'options-general.php?page=swift-offload' ),
					'pluginUrl'   => SWIFT_OFFLOAD_PLUGIN_URL,
					'version'     => SWIFT_OFFLOAD_VERSION,
					'isMultisite' => is_multisite(),
					'currentUser' => array(
						'id'               => get_current_user_id(),
						'canManageOptions' => current_user_can( 'manage_options' ),
					),
				)
			);
		}

		if ( file_exists( SWIFT_OFFLOAD_PLUGIN_DIR . 'app/dist/main.css' ) ) {
			wp_enqueue_style(
				'swift-offload-app',
				$app_css,
				array(),
				SWIFT_OFFLOAD_VERSION
			);
		}
	}

	/**
	 * Get admin page URL
	 *
	 * @param string $page Page slug
	 * @param array  $args Additional arguments
	 * @return string
	 */
	public static function get_admin_url( $page = '', $args = array() ) {
		$base_url = admin_url( 'options-general.php?page=swift-offload' );

		if ( ! empty( $page ) ) {
			$args['tab'] = $page;
		}

		if ( ! empty( $args ) ) {
			$base_url = add_query_arg( $args, $base_url );
		}

		return $base_url;
	}

	/**
	 * Get asset URL
	 *
	 * @param string $asset Asset path relative to assets directory
	 * @return string
	 */
	public static function get_asset_url( $asset ) {
		return SWIFT_OFFLOAD_PLUGIN_URL . 'assets/' . ltrim( $asset, '/' );
	}
}
