<?php
/**
 * Admin Class
 *
 * @package SwiftOffload
 */

namespace SwiftOffload;

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Admin class handles WordPress admin functionality
 */
class Admin {

	/**
	 * Admin instance
	 *
	 * @var Admin
	 */
	private static $instance = null;

	/**
	 * Get admin instance
	 *
	 * @return Admin
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
		add_action( 'admin_init', array( $this, 'admin_init' ) );
		add_filter( 'plugin_action_links_' . SWIFT_OFFLOAD_PLUGIN_BASENAME, array( $this, 'add_action_links' ) );
	}

	/**
	 * Admin initialization
	 */
	public function admin_init() {
		// Register settings with sanitization callbacks
		register_setting(
			'swift_offload_settings',
			'swift_offload_settings',
			array(
				'type'              => 'array',
				'sanitize_callback' => array( $this, 'sanitize_settings' ),
			)
		);
		register_setting(
			'swift_offload_storage_config',
			'swift_offload_storage_config',
			array(
				'type'              => 'array',
				'sanitize_callback' => array( $this, 'sanitize_storage_config' ),
			)
		);
		register_setting(
			'swift_offload_cdn_config',
			'swift_offload_cdn_config',
			array(
				'type'              => 'array',
				'sanitize_callback' => array( $this, 'sanitize_cdn_config' ),
			)
		);
	}

	/**
	 * Sanitize general settings
	 *
	 * @param array $input Raw input.
	 * @return array Sanitized input.
	 */
	public function sanitize_settings( $input ) {
		if ( ! is_array( $input ) ) {
			return array();
		}

		$sanitized = array();

		foreach ( $input as $key => $value ) {
			if ( is_bool( $value ) ) {
				$sanitized[ sanitize_key( $key ) ] = (bool) $value;
			} elseif ( is_numeric( $value ) ) {
				$sanitized[ sanitize_key( $key ) ] = intval( $value );
			} else {
				$sanitized[ sanitize_key( $key ) ] = sanitize_text_field( $value );
			}
		}

		return $sanitized;
	}

	/**
	 * Sanitize storage configuration
	 *
	 * @param array $input Raw input.
	 * @return array Sanitized input.
	 */
	public function sanitize_storage_config( $input ) {
		if ( ! is_array( $input ) ) {
			return array();
		}

		$sanitized = array();

		$allowed_keys = array(
			'provider',
			'bucket',
			'region',
			'access_key',
			'secret_key',
			'endpoint',
			'path_prefix',
			'use_path_style',
		);

		foreach ( $allowed_keys as $key ) {
			if ( isset( $input[ $key ] ) ) {
				if ( 'use_path_style' === $key ) {
					$sanitized[ $key ] = (bool) $input[ $key ];
				} else {
					$sanitized[ $key ] = sanitize_text_field( $input[ $key ] );
				}
			}
		}

		return $sanitized;
	}

	/**
	 * Sanitize CDN configuration
	 *
	 * @param array $input Raw input.
	 * @return array Sanitized input.
	 */
	public function sanitize_cdn_config( $input ) {
		if ( ! is_array( $input ) ) {
			return array();
		}

		$sanitized = array();

		$allowed_keys = array(
			'enabled',
			'domain',
			'distribution_id',
			'key_pair_id',
			'private_key',
			'signed_urls',
			'signed_url_expiry',
		);

		foreach ( $allowed_keys as $key ) {
			if ( isset( $input[ $key ] ) ) {
				if ( in_array( $key, array( 'enabled', 'signed_urls' ), true ) ) {
					$sanitized[ $key ] = (bool) $input[ $key ];
				} elseif ( 'signed_url_expiry' === $key ) {
					$sanitized[ $key ] = absint( $input[ $key ] );
				} elseif ( 'domain' === $key ) {
					$sanitized[ $key ] = esc_url_raw( $input[ $key ] );
				} elseif ( 'private_key' === $key ) {
					// Private key needs special handling - preserve newlines
					$sanitized[ $key ] = sanitize_textarea_field( $input[ $key ] );
				} else {
					$sanitized[ $key ] = sanitize_text_field( $input[ $key ] );
				}
			}
		}

		return $sanitized;
	}

	/**
	 * Add admin menu
	 */
	public function add_admin_menu() {
		// Main menu page
		add_options_page(
			__( 'Swift Offload Settings', 'swift-offload' ),
			__( 'Swift Offload', 'swift-offload' ),
			'manage_options',
			'swift-offload',
			array( $this, 'render_admin_page' )
		);

		// Add media library column
		add_filter( 'manage_media_columns', array( $this, 'add_media_column' ) );
		add_action( 'manage_media_custom_column', array( $this, 'render_media_column' ), 10, 2 );
	}

	/**
	 * Render main admin page
	 */
	public function render_admin_page() {
		// Check user capabilities
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'swift-offload' ) );
		}

		echo '<div id="swift-offload-admin-app"></div>';
	}

	/**
	 * Add media library column
	 */
	public function add_media_column( $columns ) {
		$columns['swift_offload_status'] = __( 'Swift Offload', 'swift-offload' );
		return $columns;
	}

	/**
	 * Render media library column
	 */
	public function render_media_column( $column_name, $attachment_id ) {
		if ( 'swift_offload_status' !== $column_name ) {
			return;
		}

		$item = $this->get_attachment_item( $attachment_id );

		if ( $item ) {
			echo '<span class="swift-offload-offloaded" title="' . esc_attr__( 'Offloaded to S3', 'swift-offload' ) . '">✓</span>';
			echo '<br><small>' . esc_html( $item['bucket'] ) . '</small>';
		} else {
			echo '<span class="swift-offload-local" title="' . esc_attr__( 'Stored locally', 'swift-offload' ) . '">—</span>';
		}
	}

	/**
	 * Get attachment item data
	 */
	private function get_attachment_item( $attachment_id ) {
		global $wpdb;

		$table = $wpdb->prefix . 'swift_offload_items';

		return $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM {$table} WHERE attachment_id = %d",
				$attachment_id
			),
			ARRAY_A
		);
	}

	/**
	 * Add plugin action links
	 */
	public function add_action_links( $links ) {
		$settings_link = sprintf(
			'<a href="%s">%s</a>',
			admin_url( 'options-general.php?page=swift-offload' ),
			__( 'Settings', 'swift-offload' )
		);

		array_unshift( $links, $settings_link );

		return $links;
	}

	/**
	 * Get current tab
	 */
	public function get_current_tab() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- No form submission, just reading URL tab.
		return isset( $_GET['tab'] ) ? sanitize_text_field( wp_unslash( $_GET['tab'] ) ) : 'dashboard';
	}

	/**
	 * Check if we're on a Swift Offload admin page
	 */
	public function is_swift_offload_admin_page() {
		$screen = get_current_screen();
		return $screen && strpos( $screen->id, 'swift-offload' ) !== false;
	}
}
