<?php
/**
 * CloudFront CDN Provider
 *
 * @package SwiftOffload\CDN
 */

namespace SwiftOffload\CDN;

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * CloudFront CDN Class
 */
class CDN_CloudFront {

	/**
	 * Distribution domain
	 *
	 * @var string
	 */
	private $distribution_domain;

	/**
	 * Custom domain
	 *
	 * @var string
	 */
	private $custom_domain;

	/**
	 * Private key path for signed URLs
	 *
	 * @var string
	 */
	private $private_key_path;

	/**
	 * Key pair ID for signed URLs
	 *
	 * @var string
	 */
	private $key_pair_id;

	/**
	 * Whether to use signed URLs
	 *
	 * @var bool
	 */
	private $use_signed_urls = false;

	/**
	 * Constructor
	 *
	 * @param array $config CDN configuration
	 */
	public function __construct( $config = array() ) {
		if ( empty( $config ) ) {
			$config = swift_offload_get_cdn_config();
		}

		$this->distribution_domain = $config['distribution_domain'] ?? '';
		$this->custom_domain       = $config['custom_domain'] ?? '';
		$this->private_key_path    = $config['private_key_path'] ?? '';
		$this->key_pair_id         = $config['key_pair_id'] ?? '';
		$this->use_signed_urls     = ! empty( $config['signed_urls'] );
	}

	/**
	 * Get CDN URL for object
	 *
	 * @param string $object_key Object key
	 * @param array  $options URL options
	 * @return string
	 */
	public function get_url( $object_key, $options = array() ) {
		$domain = $this->get_domain();

		if ( empty( $domain ) ) {
			return '';
		}

		$url = 'https://' . $domain . '/' . ltrim( $object_key, '/' );

		// Add query parameters if specified
		if ( ! empty( $options['query_params'] ) ) {
			$url = add_query_arg( $options['query_params'], $url );
		}

		// Generate signed URL if required
		if ( $this->use_signed_urls && $this->can_sign_urls() ) {
			return $this->create_signed_url( $url, $options );
		}

		return $url;
	}

	/**
	 * Create signed URL
	 *
	 * @param string $url Base URL
	 * @param array  $options Signing options
	 * @return string
	 */
	public function create_signed_url( $url, $options = array() ) {
		if ( ! $this->can_sign_urls() ) {
			return $url;
		}

		$expires = time() + ( $options['expires'] ?? 3600 );
		$policy  = $this->create_policy( $url, $expires, $options );

		if ( ! $policy ) {
			return $url;
		}

		$signature = $this->sign_policy( $policy );

		if ( ! $signature ) {
			return $url;
		}

		$query_params = array(
			'Expires'     => $expires,
			'Signature'   => $signature,
			'Key-Pair-Id' => $this->key_pair_id,
		);

		return add_query_arg( $query_params, $url );
	}

	/**
	 * Create policy for signed URL
	 *
	 * @param string $url URL to sign
	 * @param int    $expires Expiration timestamp
	 * @param array  $options Additional options
	 * @return string|false
	 */
	private function create_policy( $url, $expires, $options = array() ) {
		$conditions = array(
			'DateLessThan' => array(
				'AWS:EpochTime' => $expires,
			),
		);

		// Add IP address restriction if specified
		if ( ! empty( $options['ip_address'] ) ) {
			$conditions['IpAddress'] = array(
				'AWS:SourceIp' => $options['ip_address'],
			);
		}

		$policy = array(
			'Statement' => array(
				array(
					'Resource'  => $url,
					'Condition' => $conditions,
				),
			),
		);

		return json_encode( $policy, JSON_UNESCAPED_SLASHES );
	}

	/**
	 * Sign policy with private key
	 *
	 * @param string $policy Policy JSON
	 * @return string|false
	 */
	private function sign_policy( $policy ) {
		if ( ! file_exists( $this->private_key_path ) ) {
			return false;
		}

		$private_key = file_get_contents( $this->private_key_path );

		if ( ! $private_key ) {
			return false;
		}

		$encoded_policy = base64_encode( $policy );
		$signature      = '';

		if ( openssl_sign( $encoded_policy, $signature, $private_key, OPENSSL_ALGO_SHA1 ) ) {
			return base64_encode( $signature );
		}

		return false;
	}

	/**
	 * Get CDN domain
	 *
	 * @return string
	 */
	public function get_domain() {
		// Prefer custom domain over distribution domain
		if ( ! empty( $this->custom_domain ) ) {
			return $this->custom_domain;
		}

		return $this->distribution_domain;
	}

	/**
	 * Check if we can sign URLs
	 *
	 * @return bool
	 */
	public function can_sign_urls() {
		return ! empty( $this->private_key_path ) &&
				! empty( $this->key_pair_id ) &&
				file_exists( $this->private_key_path );
	}

	/**
	 * Test CDN connectivity
	 *
	 * @return array
	 */
	public function test_connection() {
		$domain = $this->get_domain();

		if ( empty( $domain ) ) {
			return array(
				'success' => false,
				'error'   => __( 'No CDN domain configured', 'swift-offload' ),
			);
		}

		// Test with a simple request
		$test_url = 'https://' . $domain;
		$response = wp_remote_head(
			$test_url,
			array(
				'timeout'    => 30,
				'user-agent' => 'SwiftOffload/' . SWIFT_OFFLOAD_VERSION,
			)
		);

		if ( is_wp_error( $response ) ) {
			return array(
				'success' => false,
				'error'   => $response->get_error_message(),
			);
		}

		$response_code = wp_remote_retrieve_response_code( $response );

		// Accept various response codes as valid
		$valid_codes = array( 200, 403, 404 ); // 403/404 are OK for CDN endpoints

		if ( in_array( $response_code, $valid_codes, true ) ) {
			return array(
				'success'       => true,
				'message'       => __( 'CDN connection successful', 'swift-offload' ),
				'response_code' => $response_code,
			);
		}

		/* translators: %d: HTTP response status code */
		return array(
			'success' => false,
			'error'   => sprintf( __( 'CDN returned status code: %d', 'swift-offload' ), $response_code ),
		);
	}

	/**
	 * Invalidate CDN cache for objects
	 *
	 * @param array $object_keys Object keys to invalidate
	 * @return array
	 */
	public function invalidate_cache( $object_keys ) {
		// CloudFront invalidation would require AWS SDK
		// This is a placeholder for the invalidation functionality

		return array(
			'success' => false,
			'error'   => __( 'Cache invalidation requires AWS SDK integration', 'swift-offload' ),
			'message' => __( 'Please use AWS console or CLI to invalidate cache', 'swift-offload' ),
		);
	}

	/**
	 * Get cache statistics (if available)
	 *
	 * @return array
	 */
	public function get_cache_stats() {
		return array(
			'hit_rate'  => null,
			'bandwidth' => null,
			'requests'  => null,
			'message'   => __( 'Statistics require CloudFront API integration', 'swift-offload' ),
		);
	}

	/**
	 * Validate CDN configuration
	 *
	 * @param array $config Configuration to validate
	 * @return array
	 */
	public static function validate_config( $config ) {
		$errors = array();

		// Check distribution domain format
		if ( ! empty( $config['distribution_domain'] ) ) {
			if ( ! preg_match( '/^[a-z0-9\-\.]+\.cloudfront\.net$/i', $config['distribution_domain'] ) ) {
				$errors[] = __( 'Distribution domain must be a valid CloudFront domain (*.cloudfront.net)', 'swift-offload' );
			}
		}

		// Check custom domain format
		if ( ! empty( $config['custom_domain'] ) ) {
			if ( ! filter_var( 'https://' . $config['custom_domain'], FILTER_VALIDATE_URL ) ) {
				$errors[] = __( 'Custom domain must be a valid domain name', 'swift-offload' );
			}
		}

		// Check signed URL requirements
		if ( ! empty( $config['signed_urls'] ) ) {
			if ( empty( $config['key_pair_id'] ) ) {
				$errors[] = __( 'Key Pair ID is required for signed URLs', 'swift-offload' );
			}
			if ( empty( $config['private_key_path'] ) || ! file_exists( $config['private_key_path'] ) ) {
				$errors[] = __( 'Valid private key file path is required for signed URLs', 'swift-offload' );
			}
		}

		return array(
			'valid'  => empty( $errors ),
			'errors' => $errors,
		);
	}
}
