<?php
/**
 * Main Swift Offload class for legacy functionality.
 *
 * @package SwiftOffload
 */

namespace SwiftOffload;

use SwiftOffload\Libs\Assets;
use SwiftOffload\Libs\Helper;

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Swift_Offload Class
 *
 * @since 1.0.0
 */
if ( ! class_exists( '\SwiftOffload\Swift_Offload' ) ) {

	/**
	 * Class: Swift_Offload
	 */
	final class Swift_Offload {

		/**
		 * Plugin version.
		 *
		 * @var string
		 */
		const VERSION = SWIFT_OFFLOAD_VERSION;

		/**
		 * Singleton instance.
		 *
		 * @var Swift_Offload|null
		 */
		private static $instance = null;

		/**
		 * Constructor method.
		 *
		 * @author Swift Offload Team <support@swiftplugins.pro>
		 */
		public function __construct() {
			$this->includes();
			add_action( 'plugins_loaded', array( $this, 'swift_offload_plugins_loaded' ), 999 );
			add_filter( 'admin_body_class', array( $this, 'swift_offload_body_class' ) );
		}

		/**
		 * Plugins loaded method.
		 *
		 * @author Swift Offload Team <support@swiftplugins.pro>
		 */
		public function swift_offload_plugins_loaded() {
			$this->swift_offload_activate();
		}

		/**
		 * Version Key.
		 *
		 * @author Swift Offload Team <support@swiftplugins.pro>
		 */
		public static function plugin_version_key() {
			return Helper::swift_offload_slug_cleanup() . '_version';
		}

		/**
		 * Activation Hook.
		 *
		 * @author Swift Offload Team <support@swiftplugins.pro>
		 */
		public static function swift_offload_activate() {
			$current_swift_offload_version = get_option( self::plugin_version_key(), null );

			if ( get_option( 'swift_offload_activation_time' ) === false ) {
				update_option( 'swift_offload_activation_time', strtotime( 'now' ) );
			}

			if ( is_null( $current_swift_offload_version ) ) {
				update_option( self::plugin_version_key(), self::VERSION );
			}

			$allowed = get_option( Helper::swift_offload_slug_cleanup() . '_allow_tracking', 'no' );

			if ( 'yes' !== $allowed ) {
				return;
			}

			$hook_name = Helper::swift_offload_slug_cleanup() . '_tracker_send_event';
			if ( ! wp_next_scheduled( $hook_name ) ) {
				wp_schedule_event( time(), 'weekly', $hook_name );
			}
		}


		/**
		 * Add Body Class.
		 *
		 * @param string $classes Body classes.
		 *
		 * @author Swift Offload Team <support@swiftplugins.pro>
		 */
		public function swift_offload_body_class( $classes ) {
			$classes .= ' swift_offload ';
			return $classes;
		}


		/**
		 * Include methods.
		 *
		 * @author Swift Offload Team <support@swiftplugins.pro>
		 */
		public function includes() {
			new Assets();
		}


		/**
		 * Initialization.
		 *
		 * @author Swift Offload Team <support@swiftplugins.pro>
		 */
		public function swift_offload_init() {
			$this->swift_offload_load_textdomain();
		}


		/**
		 * Text Domain.
		 *
		 * @author Swift Offload Team <support@swiftplugins.pro>
		 */
		public function swift_offload_load_textdomain() {
			// Text domain loading is handled automatically by WordPress 4.6+
			// for plugins hosted on WordPress.org.
		}

		/**
		 * Returns the singleton instance of the class.
		 *
		 * @return Swift_Offload
		 */
		public static function get_instance() {
			if ( ! isset( self::$instance ) && ! ( self::$instance instanceof Swift_Offload ) ) {
				self::$instance = new Swift_Offload();
				self::$instance->swift_offload_init();
			}

			return self::$instance;
		}
	}

	// Get instance of Swift_Offload Class.
	Swift_Offload::get_instance();
}
