/**
 * Swift Offload - Plupload Interceptor
 *
 * Intercepts the WordPress plupload uploader to capture original file data
 * before upload. This allows the image optimizer to access the original
 * uncompressed file for client-side optimization.
 *
 * Must load early (in header) before other plugins initialize their uploaders.
 */
(function() {
	'use strict';

	// Fix mixed content - upgrade http to https for images
	if (window.location.protocol === 'https:') {
		document.addEventListener('DOMContentLoaded', function() {
			document.querySelectorAll('img[src^="http://"]').forEach(function(img) {
				img.src = img.src.replace('http://', 'https://');
			});
			document.querySelectorAll('img[srcset]').forEach(function(img) {
				if (img.srcset) {
					img.srcset = img.srcset.replace(/http:\/\//g, 'https://');
				}
			});
		});
	}

	window.swiftOffloadOriginalFiles = window.swiftOffloadOriginalFiles || {};

	function captureFileData(file, nativeFile) {
		var ext = nativeFile.name.toLowerCase().split('.').pop();
		if (['jpg', 'jpeg', 'png'].indexOf(ext) === -1) return;
		window.swiftOffloadOriginalFiles[file.id] = {
			name: nativeFile.name,
			type: nativeFile.type,
			size: nativeFile.size,
			fileRef: nativeFile
		};
	}

	function interceptPlupload() {
		if (typeof plupload === 'undefined' || !plupload.Uploader) return false;
		if (plupload._swiftOffloadIntercepted) return true;

		var OriginalUploader = plupload.Uploader;
		plupload.Uploader = function(settings) {
			var uploader = new OriginalUploader(settings);
			uploader.bind('FilesAdded', function(up, files) {
				for (var i = 0; i < files.length; i++) {
					var file = files[i];
					var nativeFile = file.getNative ? file.getNative() : null;
					if (nativeFile && !window.swiftOffloadOriginalFiles[file.id]) {
						captureFileData(file, nativeFile);
					}
				}
			}, 0);
			return uploader;
		};

		for (var key in OriginalUploader) {
			if (OriginalUploader.hasOwnProperty(key)) {
				plupload.Uploader[key] = OriginalUploader[key];
			}
		}
		plupload.Uploader.prototype = OriginalUploader.prototype;
		plupload._swiftOffloadIntercepted = true;
		return true;
	}

	if (!interceptPlupload()) {
		var checkInterval = setInterval(function() {
			if (interceptPlupload()) clearInterval(checkInterval);
		}, 10);
		setTimeout(function() { clearInterval(checkInterval); }, 10000);
	}
})();
