(function($) {
  var SwiftOffload_ImageOptimizer = {
    settings: {},
    i18n: {},
    isSupported: false,
    pendingOptimizations: {},
    worker: null,
    workerReady: false,
    useWorker: false,
    ffmpeg: null,
    ffmpegLoading: false,
    ffmpegReady: false,
    getOriginalFileData: function(fileId) {
      return window.swiftOffloadOriginalFiles ? window.swiftOffloadOriginalFiles[fileId] : null;
    },
    init: function() {
      if (typeof swiftOffloadImageOptimizer === "undefined") return;
      this.settings = swiftOffloadImageOptimizer.settings || {};
      this.i18n = swiftOffloadImageOptimizer.i18n || {};
      this.isSupported = this.checkBrowserSupport();
      this.useWorker = this.checkWorkerSupport();
      this.initAdminPanel();
      this.initMediaLibraryDisplay();
      if (!this.isSupported || this.settings.enabled !== "yes") return;
      if (this.useWorker) this.initWorker();
      this.hookIntoPlupload();
    },
    checkBrowserSupport: function() {
      var canvas = document.createElement("canvas");
      return !!(canvas.getContext && canvas.getContext("2d")) && typeof FileReader !== "undefined" && typeof Blob !== "undefined";
    },
    checkWorkerSupport: function() {
      return typeof Worker !== "undefined" && typeof OffscreenCanvas !== "undefined" && typeof WebAssembly !== "undefined";
    },
    initWorker: function() {
      var self = this;
      try {
        var workerUrl = swiftOffloadImageOptimizer.workerUrl || swiftOffloadImageOptimizer.pluginUrl + "dev/js/image-optimizer/compression-worker.js";
        this.worker = new Worker(workerUrl);
        this.worker.onmessage = function(e) {
          var data = e.data;
          switch (data.type) {
            case "ready":
              self.workerReady = true;
              break;
            case "result":
              if (self.pendingWorkerCallbacks && self.pendingWorkerCallbacks[data.result.fileId]) {
                self.pendingWorkerCallbacks[data.result.fileId](data.result);
                delete self.pendingWorkerCallbacks[data.result.fileId];
              }
              break;
            case "batchResult":
              if (self.pendingBatchCallback) {
                self.pendingBatchCallback(data.results);
                self.pendingBatchCallback = null;
              }
              break;
          }
        };
        this.worker.onerror = function() {
          self.useWorker = false;
        };
        this.worker.postMessage({ type: "init" });
        this.pendingWorkerCallbacks = {};
      } catch (e) {
        this.useWorker = false;
      }
    },
    hookIntoPlupload: function() {
      var self = this;
      if (typeof wp === "undefined" || typeof wp.Uploader === "undefined") {
        $(document).ready(function() {
          setTimeout(function() {
            self.hookIntoPlupload();
          }, 500);
        });
        return;
      }
      var originalInit = wp.Uploader.prototype.init;
      wp.Uploader.prototype.init = function() {
        if (originalInit) originalInit.apply(this, arguments);
        var uploader = this.uploader;
        if (!uploader) return;
        uploader.bind("FilesAdded", function(up, files) {
          self.handleFilesAdded(up, files);
        });
        uploader.bind("FileUploaded", function(up, file, response) {
          self.handleFileUploaded(file, response);
        });
      };
    },
    handleFilesAdded: function(uploader, files) {
      var self = this;
      var imagesToProcess = [];
      var videosToProcess = [];
      var audiosToProcess = [];
      for (var i = 0; i < files.length; i++) {
        var file = files[i];
        var originalData = self.getOriginalFileData(file.id);
        var fileName, fileType, originalFile;
        if (originalData && originalData.fileRef) {
          fileName = originalData.name;
          fileType = originalData.type;
          originalFile = originalData.fileRef;
        } else {
          var nativeFile = file.getNative ? file.getNative() : null;
          fileName = nativeFile ? nativeFile.name : file.name;
          fileType = nativeFile ? nativeFile.type : file.type;
          originalFile = nativeFile;
        }
        if (originalFile) {
          file._swiftOffloadOriginalFile = originalFile;
          file._swiftOffloadOriginalName = fileName;
          file._swiftOffloadOriginalType = fileType;
          file._swiftOffloadOriginalSize = originalData ? originalData.size : originalFile.size;
        }
        var ext = fileName.toLowerCase().split(".").pop();
        var mimeType = fileType.toLowerCase();
        var isImage = ["jpg", "jpeg", "png"].indexOf(ext) !== -1 || ["image/jpeg", "image/jpg", "image/png"].indexOf(mimeType) !== -1;
        var isVideo = ["mp4", "webm", "mov", "avi", "mkv", "m4v"].indexOf(ext) !== -1 || mimeType.indexOf("video/") === 0;
        var isAudio = ["mp3", "wav", "ogg", "flac", "aac", "m4a"].indexOf(ext) !== -1 || mimeType.indexOf("audio/") === 0;
        if (!file._swiftOffloadProcessed) {
          if (isImage) {
            imagesToProcess.push(file);
          } else if (isVideo && self.settings.optimize_videos === "yes") {
            videosToProcess.push(file);
          } else if (isAudio && self.settings.optimize_audio === "yes") {
            audiosToProcess.push(file);
          }
        }
      }
      var totalToProcess = imagesToProcess.length + videosToProcess.length + audiosToProcess.length;
      if (totalToProcess === 0) return;
      uploader.stop();
      if (imagesToProcess.length > 0) {
        if (imagesToProcess.length > 1) {
          self.showToast("Processing " + imagesToProcess.length + " images...", "processing");
        }
        if (self.useWorker && self.workerReady && imagesToProcess.length > 1) {
          self.processBatchWithWorker(uploader, imagesToProcess, videosToProcess, audiosToProcess);
        } else {
          self.processFilesSequentially(uploader, imagesToProcess, videosToProcess, audiosToProcess);
        }
      } else if (videosToProcess.length > 0 || audiosToProcess.length > 0) {
        self.processMediaFilesSequentially(uploader, videosToProcess, audiosToProcess);
      }
    },
    processFilesSequentially: function(uploader, imagesToProcess, videosToProcess, audiosToProcess) {
      var self = this;
      var processedCount = 0;
      var totalToProcess = imagesToProcess.length;
      videosToProcess = videosToProcess || [];
      audiosToProcess = audiosToProcess || [];
      imagesToProcess.forEach(function(file) {
        self.showFileStatus(file, "Optimizing...");
        self.optimizeFile(file, function(optimizedFile, stats) {
          processedCount++;
          if (optimizedFile && stats) {
            self.replaceFileInUploader(uploader, file, optimizedFile);
            self.showFileStatus(file, "Saved " + self.formatBytes(stats.saved) + " (" + stats.percentage + "%)", "success");
            self.trackStats(stats.originalSize, stats.optimizedSize);
            self.pendingOptimizations[file.id] = stats;
          } else {
            file._swiftOffloadProcessed = true;
            self.showFileStatus(file, "Kept original", "info");
          }
          if (processedCount >= totalToProcess) {
            if (totalToProcess > 1) {
              self.showToast("Batch optimization complete!", "success");
            }
            if (videosToProcess.length > 0 || audiosToProcess.length > 0) {
              self.processMediaFilesSequentially(uploader, videosToProcess, audiosToProcess);
            } else {
              setTimeout(function() {
                uploader.start();
              }, 100);
            }
          }
        });
      });
    },
    /**
     * Process video and audio files sequentially
     */
    processMediaFilesSequentially: function(uploader, videosToProcess, audiosToProcess) {
      var self = this;
      var allMedia = videosToProcess.concat(audiosToProcess);
      var totalMedia = allMedia.length;
      if (totalMedia === 0) {
        setTimeout(function() {
          uploader.start();
        }, 100);
        return;
      }
      if (totalMedia > 0) {
        self.showToast("Processing " + totalMedia + " media file(s)...", "processing");
      }
      function processNextMedia(index) {
        if (index >= totalMedia) {
          self.showToast("Media optimization complete!", "success");
          setTimeout(function() {
            uploader.start();
          }, 100);
          return;
        }
        var file = allMedia[index];
        var isVideo = videosToProcess.indexOf(file) !== -1;
        if (isVideo) {
          self.processVideo(file, function(optimizedFile, stats) {
            handleMediaResult(file, optimizedFile, stats, index);
          });
        } else {
          self.processAudio(file, function(optimizedFile, stats) {
            handleMediaResult(file, optimizedFile, stats, index);
          });
        }
      }
      function handleMediaResult(file, optimizedFile, stats, index) {
        if (optimizedFile && stats) {
          self.replaceFileInUploader(uploader, file, optimizedFile);
          self.showFileStatus(file, "Saved " + self.formatBytes(stats.saved) + " (" + stats.percentage + "%)", "success");
          self.trackStats(stats.originalSize, stats.optimizedSize);
          self.pendingOptimizations[file.id] = stats;
        } else {
          file._swiftOffloadProcessed = true;
          self.showFileStatus(file, "Kept original", "info");
        }
        processNextMedia(index + 1);
      }
      processNextMedia(0);
    },
    processBatchWithWorker: function(uploader, imagesToProcess, videosToProcess, audiosToProcess) {
      var self = this;
      var filesData = [];
      var fileMap = {};
      var loadedCount = 0;
      videosToProcess = videosToProcess || [];
      audiosToProcess = audiosToProcess || [];
      imagesToProcess.forEach(function(file) {
        var originalData = self.getOriginalFileData(file.id);
        var originalName = originalData ? originalData.name : file.name;
        var originalSize = originalData ? originalData.size : 0;
        var originalFileRef = originalData ? originalData.fileRef : null;
        var nativeFile = originalFileRef || (file.getNative ? file.getNative() : null);
        fileMap[file.id] = file;
        self.showFileStatus(file, "Loading...");
        if (!nativeFile) {
          loadedCount++;
          if (loadedCount >= imagesToProcess.length) {
            self.sendBatchToWorker(uploader, filesData, fileMap, imagesToProcess, videosToProcess, audiosToProcess);
          }
          return;
        }
        self.loadImageForWorker(nativeFile, function(imageData, width, height) {
          if (imageData) {
            filesData.push({
              fileId: file.id,
              fileName: originalName || nativeFile.name,
              imageData,
              width,
              height,
              originalSize: originalSize || nativeFile.size
            });
          }
          loadedCount++;
          if (loadedCount >= imagesToProcess.length) {
            self.sendBatchToWorker(uploader, filesData, fileMap, imagesToProcess, videosToProcess, audiosToProcess);
          }
        });
      });
    },
    loadImageForWorker: function(nativeFile, callback) {
      var reader = new FileReader();
      reader.onload = function(e) {
        var img = new Image();
        img.onload = function() {
          var canvas = document.createElement("canvas");
          var ctx = canvas.getContext("2d");
          var width = img.naturalWidth || img.width;
          var height = img.naturalHeight || img.height;
          canvas.width = width;
          canvas.height = height;
          ctx.drawImage(img, 0, 0);
          var imageData = ctx.getImageData(0, 0, width, height);
          callback(imageData.data.buffer, width, height);
        };
        img.onerror = function() {
          callback(null);
        };
        img.src = e.target.result;
      };
      reader.onerror = function() {
        callback(null);
      };
      reader.readAsDataURL(nativeFile);
    },
    sendBatchToWorker: function(uploader, filesData, fileMap, imagesToProcess, videosToProcess, audiosToProcess) {
      var self = this;
      videosToProcess = videosToProcess || [];
      audiosToProcess = audiosToProcess || [];
      if (filesData.length === 0) {
        if (videosToProcess.length > 0 || audiosToProcess.length > 0) {
          self.processMediaFilesSequentially(uploader, videosToProcess, audiosToProcess);
        } else {
          uploader.start();
        }
        return;
      }
      imagesToProcess.forEach(function(file) {
        self.showFileStatus(file, "Compressing...");
      });
      self.pendingBatchCallback = function(results) {
        var totalSaved = 0;
        var optimizedCount = 0;
        results.forEach(function(result) {
          var file = fileMap[result.fileId];
          if (!file) return;
          if (result.success) {
            var blob = new Blob([result.optimizedData], { type: result.outputType });
            var optimizedFile = new File([blob], result.fileName, {
              type: result.outputType,
              lastModified: Date.now()
            });
            self.replaceFileInUploader(uploader, file, optimizedFile);
            self.showFileStatus(file, "Saved " + self.formatBytes(result.stats.saved) + " (" + result.stats.percentage + "%)", "success");
            self.trackStats(result.stats.originalSize, result.stats.optimizedSize);
            self.pendingOptimizations[file.id] = result.stats;
            totalSaved += result.stats.saved;
            optimizedCount++;
          } else {
            file._swiftOffloadProcessed = true;
            self.showFileStatus(file, "Kept original", "info");
          }
        });
        if (optimizedCount > 0) {
          self.showToast("Optimized " + optimizedCount + " images, saved " + self.formatBytes(totalSaved) + "!", "success");
        }
        if (videosToProcess.length > 0 || audiosToProcess.length > 0) {
          self.processMediaFilesSequentially(uploader, videosToProcess, audiosToProcess);
        } else {
          setTimeout(function() {
            uploader.start();
          }, 100);
        }
      };
      self.worker.postMessage({
        type: "batch",
        data: { images: filesData, settings: self.settings }
      });
    },
    isImageFile: function(file) {
      var type = file.type || "";
      return ["image/jpeg", "image/jpg", "image/png"].indexOf(type.toLowerCase()) !== -1;
    },
    isVideoFile: function(file) {
      var type = file.type || "";
      return ["video/mp4", "video/webm", "video/quicktime", "video/avi", "video/mov"].indexOf(type.toLowerCase()) !== -1 || type.startsWith("video/");
    },
    isAudioFile: function(file) {
      var type = file.type || "";
      return ["audio/mpeg", "audio/mp3", "audio/wav", "audio/ogg", "audio/flac", "audio/aac"].indexOf(type.toLowerCase()) !== -1 || type.startsWith("audio/");
    },
    optimizeFile: function(pluploadFile, callback) {
      var self = this;
      var originalData = self.getOriginalFileData(pluploadFile.id);
      if (originalData && originalData.loading) {
        var checkCount = 0;
        var checkInterval = setInterval(function() {
          checkCount++;
          if (!originalData.loading || checkCount > 100) {
            clearInterval(checkInterval);
            self.processFileWithData(pluploadFile, originalData, callback);
          }
        }, 50);
        return;
      }
      self.processFileWithData(pluploadFile, originalData, callback);
    },
    processFileWithData: function(pluploadFile, originalData, callback) {
      var self = this;
      var originalFileRef = originalData ? originalData.fileRef : null;
      var nativeFile = originalFileRef || pluploadFile._swiftOffloadOriginalFile || (pluploadFile.getNative ? pluploadFile.getNative() : null);
      if (!nativeFile) {
        callback(null, null);
        return;
      }
      var reader = new FileReader();
      reader.onload = function(e) {
        self.loadAndProcessImage(e.target.result, pluploadFile, originalData, callback);
      };
      reader.onerror = function() {
        callback(null, null);
      };
      reader.readAsDataURL(nativeFile);
    },
    loadAndProcessImage: function(dataUrl, pluploadFile, originalData, callback) {
      var self = this;
      var img = new Image();
      img.onload = function() {
        try {
          var originalInfo = {
            name: originalData ? originalData.name : pluploadFile.name,
            type: originalData ? originalData.type : "image/jpeg",
            size: originalData ? originalData.size : 0
          };
          var result = self.processWithCanvas(img, null, originalInfo);
          callback(result.file, result.stats);
        } catch (err) {
          callback(null, null);
        }
      };
      img.onerror = function() {
        callback(null, null);
      };
      img.src = dataUrl;
    },
    processWithCanvas: function(img, originalFile, originalInfo) {
      var self = this;
      var startTime = performance.now();
      var canvas = document.createElement("canvas");
      var ctx = canvas.getContext("2d");
      var fileInfo = originalInfo || (originalFile ? {
        name: originalFile.name,
        type: originalFile.type,
        size: originalFile.size
      } : { name: "image.jpg", type: "image/jpeg", size: 0 });
      var width = img.naturalWidth || img.width;
      var height = img.naturalHeight || img.height;
      var maxDim = parseInt(this.settings.max_image_dimension) || 2560;
      var quality = (parseInt(this.settings.webp_quality) || 80) / 100;
      if (this.settings.resize_enabled === "yes" && (width > maxDim || height > maxDim)) {
        var ratio = Math.min(maxDim / width, maxDim / height);
        width = Math.round(width * ratio);
        height = Math.round(height * ratio);
      }
      canvas.width = width;
      canvas.height = height;
      ctx.drawImage(img, 0, 0, width, height);
      var outputType = "image/webp";
      var extension = ".webp";
      var outputFormat = "webp";
      if (this.settings.convert_webp !== "yes" && fileInfo.type === "image/jpeg") {
        outputType = "image/jpeg";
        extension = ".jpg";
        outputFormat = "jpeg";
      }
      var dataUrl = canvas.toDataURL(outputType, quality);
      var blob = this.dataURLtoBlob(dataUrl);
      var originalSize = fileInfo.size;
      var optimizedSize = blob.size;
      if (optimizedSize >= originalSize) {
        return { file: null, stats: null };
      }
      var newName = fileInfo.name.replace(/\.[^.]+$/, extension);
      var optimizedFile = new File([blob], newName, {
        type: outputType,
        lastModified: Date.now()
      });
      var lqipData = null;
      if (self.settings.generate_lqip === "yes") {
        lqipData = self.generateLQIP(canvas, width, height);
      }
      var processingTime = Math.round(performance.now() - startTime);
      var originalFormat = fileInfo.type.replace("image/", "");
      if (originalFormat === "jpeg") originalFormat = "jpg";
      return {
        file: optimizedFile,
        stats: {
          originalSize,
          optimizedSize,
          saved: originalSize - optimizedSize,
          percentage: Math.round((1 - optimizedSize / originalSize) * 100 * 100) / 100,
          processingTime,
          originalFormat,
          outputFormat,
          driver: "Browser",
          quality: Math.round(quality * 100),
          lqipData
        }
      };
    },
    generateLQIP: function(sourceCanvas, width, height) {
      var lqipQuality = (parseInt(this.settings.lqip_quality) || 20) / 100;
      var lqipBlur = parseInt(this.settings.lqip_blur) || 10;
      var ratio = Math.min(32 / width, 32 / height);
      var lqipWidth = Math.round(width * ratio);
      var lqipHeight = Math.round(height * ratio);
      var lqipCanvas = document.createElement("canvas");
      lqipCanvas.width = lqipWidth;
      lqipCanvas.height = lqipHeight;
      var lqipCtx = lqipCanvas.getContext("2d");
      lqipCtx.filter = "blur(" + lqipBlur / 10 + "px)";
      lqipCtx.drawImage(sourceCanvas, 0, 0, lqipWidth, lqipHeight);
      return lqipCanvas.toDataURL("image/webp", lqipQuality);
    },
    /**
     * Load FFmpeg.wasm for video/audio processing
     */
    loadFFmpeg: function() {
      var self = this;
      if (self.ffmpegReady) {
        return Promise.resolve(self.ffmpeg);
      }
      if (self.ffmpegLoading) {
        return new Promise(function(resolve) {
          var checkInterval = setInterval(function() {
            if (self.ffmpegReady) {
              clearInterval(checkInterval);
              resolve(self.ffmpeg);
            }
          }, 100);
        });
      }
      self.ffmpegLoading = true;
      return new Promise(function(resolve, reject) {
        var script = document.createElement("script");
        script.src = "https://unpkg.com/@ffmpeg/ffmpeg@0.12.6/dist/umd/ffmpeg.min.js";
        script.onload = function() {
          var coreScript = document.createElement("script");
          coreScript.src = "https://unpkg.com/@ffmpeg/core@0.12.4/dist/umd/ffmpeg-core.js";
          coreScript.onload = function() {
            try {
              self.ffmpeg = new FFmpeg.FFmpeg();
              self.ffmpeg.load({
                coreURL: "https://unpkg.com/@ffmpeg/core@0.12.4/dist/umd/ffmpeg-core.js",
                wasmURL: "https://unpkg.com/@ffmpeg/core@0.12.4/dist/umd/ffmpeg-core.wasm"
              }).then(function() {
                self.ffmpegReady = true;
                self.ffmpegLoading = false;
                resolve(self.ffmpeg);
              }).catch(function(err) {
                self.ffmpegLoading = false;
                reject(err);
              });
            } catch (e) {
              self.ffmpegLoading = false;
              reject(e);
            }
          };
          coreScript.onerror = function() {
            self.ffmpegLoading = false;
            reject(new Error("Failed to load FFmpeg core"));
          };
          document.head.appendChild(coreScript);
        };
        script.onerror = function() {
          self.ffmpegLoading = false;
          reject(new Error("Failed to load FFmpeg"));
        };
        document.head.appendChild(script);
      });
    },
    /**
     * Process video file with FFmpeg.wasm
     */
    processVideo: function(file, callback) {
      var self = this;
      var startTime = performance.now();
      var originalSize = file._swiftOffloadOriginalSize || file.size;
      var originalName = file._swiftOffloadOriginalName || file.name;
      self.showFileStatus(file, "Loading FFmpeg...");
      self.loadFFmpeg().then(function(ffmpeg) {
        self.showFileStatus(file, "Processing video...");
        var reader = new FileReader();
        reader.onload = function(e) {
          var inputData = new Uint8Array(e.target.result);
          var inputName = "input" + self.getFileExtension(originalName);
          var outputName = "output.webm";
          var quality = self.settings.video_quality || "medium";
          var maxRes = parseInt(self.settings.video_max_resolution) || 1080;
          var crfMap = { "low": 40, "medium": 33, "high": 26 };
          var crf = crfMap[quality] || 33;
          var scaleFilter = "scale='min(" + maxRes + ",iw)':'min(" + maxRes + ",ih)':force_original_aspect_ratio=decrease";
          ffmpeg.writeFile(inputName, inputData).then(function() {
            return ffmpeg.exec([
              "-i",
              inputName,
              "-c:v",
              "libvpx-vp9",
              "-crf",
              crf.toString(),
              "-b:v",
              "0",
              "-c:a",
              "libopus",
              "-b:a",
              "128k",
              "-vf",
              scaleFilter,
              "-y",
              outputName
            ]);
          }).then(function() {
            return ffmpeg.readFile(outputName);
          }).then(function(outputData) {
            var optimizedSize = outputData.length;
            var processingTime = Math.round(performance.now() - startTime);
            ffmpeg.deleteFile(inputName).catch(function() {
            });
            ffmpeg.deleteFile(outputName).catch(function() {
            });
            if (optimizedSize >= originalSize) {
              callback(null, null);
              return;
            }
            var blob = new Blob([outputData], { type: "video/webm" });
            var newName = originalName.replace(/\.[^.]+$/, ".webm");
            var optimizedFile = new File([blob], newName, {
              type: "video/webm",
              lastModified: Date.now()
            });
            callback(optimizedFile, {
              originalSize,
              optimizedSize,
              saved: originalSize - optimizedSize,
              percentage: Math.round((1 - optimizedSize / originalSize) * 100 * 100) / 100,
              processingTime,
              originalFormat: self.getFileExtension(originalName).replace(".", ""),
              outputFormat: "webm",
              driver: "FFmpeg.wasm",
              quality,
              mediaType: "video"
            });
          }).catch(function(err) {
            console.error("Video processing error:", err);
            callback(null, null);
          });
        };
        reader.onerror = function() {
          callback(null, null);
        };
        reader.readAsArrayBuffer(file._swiftOffloadOriginalFile || file.getNative());
      }).catch(function(err) {
        console.error("FFmpeg load error:", err);
        self.showFileStatus(file, "FFmpeg not supported", "error");
        callback(null, null);
      });
    },
    /**
     * Process audio file with FFmpeg.wasm
     */
    processAudio: function(file, callback) {
      var self = this;
      var startTime = performance.now();
      var originalSize = file._swiftOffloadOriginalSize || file.size;
      var originalName = file._swiftOffloadOriginalName || file.name;
      self.showFileStatus(file, "Loading FFmpeg...");
      self.loadFFmpeg().then(function(ffmpeg) {
        self.showFileStatus(file, "Processing audio...");
        var reader = new FileReader();
        reader.onload = function(e) {
          var inputData = new Uint8Array(e.target.result);
          var inputName = "input" + self.getFileExtension(originalName);
          var outputName = "output.mp3";
          var bitrate = (parseInt(self.settings.audio_bitrate) || 128) + "k";
          ffmpeg.writeFile(inputName, inputData).then(function() {
            return ffmpeg.exec([
              "-i",
              inputName,
              "-c:a",
              "libmp3lame",
              "-b:a",
              bitrate,
              "-y",
              outputName
            ]);
          }).then(function() {
            return ffmpeg.readFile(outputName);
          }).then(function(outputData) {
            var optimizedSize = outputData.length;
            var processingTime = Math.round(performance.now() - startTime);
            ffmpeg.deleteFile(inputName).catch(function() {
            });
            ffmpeg.deleteFile(outputName).catch(function() {
            });
            if (optimizedSize >= originalSize) {
              callback(null, null);
              return;
            }
            var blob = new Blob([outputData], { type: "audio/mpeg" });
            var newName = originalName.replace(/\.[^.]+$/, ".mp3");
            var optimizedFile = new File([blob], newName, {
              type: "audio/mpeg",
              lastModified: Date.now()
            });
            callback(optimizedFile, {
              originalSize,
              optimizedSize,
              saved: originalSize - optimizedSize,
              percentage: Math.round((1 - optimizedSize / originalSize) * 100 * 100) / 100,
              processingTime,
              originalFormat: self.getFileExtension(originalName).replace(".", ""),
              outputFormat: "mp3",
              driver: "FFmpeg.wasm",
              quality: bitrate,
              mediaType: "audio"
            });
          }).catch(function(err) {
            console.error("Audio processing error:", err);
            callback(null, null);
          });
        };
        reader.onerror = function() {
          callback(null, null);
        };
        reader.readAsArrayBuffer(file._swiftOffloadOriginalFile || file.getNative());
      }).catch(function(err) {
        console.error("FFmpeg load error:", err);
        self.showFileStatus(file, "FFmpeg not supported", "error");
        callback(null, null);
      });
    },
    /**
     * Get file extension including the dot
     */
    getFileExtension: function(filename) {
      var match = filename.match(/\.[^.]+$/);
      return match ? match[0].toLowerCase() : "";
    },
    dataURLtoBlob: function(dataUrl) {
      var parts = dataUrl.split(",");
      var mime = parts[0].match(/:(.*?);/)[1];
      var binary = atob(parts[1]);
      var array = new Uint8Array(binary.length);
      for (var i = 0; i < binary.length; i++) {
        array[i] = binary.charCodeAt(i);
      }
      return new Blob([array], { type: mime });
    },
    replaceFileInUploader: function(uploader, oldFile, newFile) {
      oldFile._swiftOffloadProcessed = true;
      oldFile.name = newFile.name;
      oldFile.size = newFile.size;
      oldFile.type = newFile.type;
      oldFile.getNative = function() {
        return newFile;
      };
      if (oldFile.native) oldFile.native = newFile;
    },
    showFileStatus: function(file, message, type) {
      type = type || "processing";
      var $item = $(".media-item, .attachment").filter(function() {
        var $this = $(this);
        return $this.find(".filename").text().indexOf(file.name) !== -1 || $this.data("id") === file.id;
      });
      var statusClass = "swift-offload-opt-status";
      var $status = $item.find("." + statusClass);
      if ($status.length === 0) {
        $status = $('<div class="' + statusClass + '"></div>');
        $item.find(".filename, .attachment-info").first().after($status);
      }
      var colors = {
        processing: "#6366f1",
        success: "#10b981",
        error: "#ef4444",
        info: "#6b7280"
      };
      $status.css({
        "font-size": "12px",
        "padding": "4px 8px",
        "margin": "4px 0",
        "border-radius": "4px",
        "background": colors[type] || colors.processing,
        "color": "#fff"
      }).text(message);
      this.showToast(message, type);
    },
    showToast: function(message, type) {
      if (typeof window.swiftOffloadShowToast === "function") {
        window.swiftOffloadShowToast(message, type);
        return;
      }
      var $container = $(".swift-offload-toaster-container");
      if ($container.length === 0) {
        $container = $('<div class="swift-offload-toaster-container"></div>');
        $("body").append($container);
      }
      $container.find(".swift-offload-toaster").remove();
      var icons = {
        processing: "⟳",
        success: "✓",
        error: "✕",
        info: "i"
      };
      var $toast = $('<div class="swift-offload-toaster ' + (type || "success") + '"><span class="swift-offload-toaster-icon">' + (icons[type] || icons.success) + '</span><span class="swift-offload-toaster-content">' + message + '</span><button class="swift-offload-toaster-close"></button><div class="swift-offload-toaster-progress"></div></div>');
      $container.append($toast);
      $toast.find(".swift-offload-toaster-close").on("click", function() {
        $toast.addClass("swift-offload-toaster-exit");
        setTimeout(function() {
          $toast.remove();
        }, 300);
      });
      if (type !== "processing") {
        setTimeout(function() {
          $toast.addClass("swift-offload-toaster-exit");
          setTimeout(function() {
            $toast.remove();
          }, 300);
        }, 3e3);
      }
    },
    trackStats: function(originalSize, optimizedSize) {
      $.ajax({
        url: swiftOffloadImageOptimizer.ajaxUrl,
        type: "POST",
        data: {
          action: "swift_offload_update_optimizer_stats",
          nonce: swiftOffloadImageOptimizer.nonce,
          original_size: originalSize,
          optimized_size: optimizedSize
        }
      });
    },
    handleFileUploaded: function(file, response) {
      var self = this;
      var stats = this.pendingOptimizations[file.id];
      if (!stats) return;
      var attachmentId = null;
      try {
        var responseData;
        if (typeof response.response === "string") {
          responseData = JSON.parse(response.response);
        } else {
          responseData = response.response || response;
        }
        if (responseData && responseData.data && responseData.data.id) {
          attachmentId = responseData.data.id;
        } else if (responseData && responseData.id) {
          attachmentId = responseData.id;
        } else if (responseData && responseData.attachment && responseData.attachment.id) {
          attachmentId = responseData.attachment.id;
        }
      } catch (e) {
      }
      if (!attachmentId) return;
      $.ajax({
        url: swiftOffloadImageOptimizer.ajaxUrl,
        type: "POST",
        data: {
          action: "swift_offload_save_attachment_optimization",
          nonce: swiftOffloadImageOptimizer.nonce,
          attachment_id: attachmentId,
          original_size: stats.originalSize,
          optimized_size: stats.optimizedSize,
          processing_time: stats.processingTime || 0,
          original_format: stats.originalFormat || "",
          output_format: stats.outputFormat || "webp",
          driver: stats.driver || "Browser",
          quality: stats.quality || 80,
          lqip_data: stats.lqipData || ""
        },
        success: function() {
          self.updateOptimizationDisplay(attachmentId, stats);
        }
      });
      delete this.pendingOptimizations[file.id];
    },
    formatBytes: function(bytes) {
      if (bytes === 0) return "0 B";
      var k = 1024;
      var sizes = ["B", "KB", "MB", "GB"];
      var i = Math.floor(Math.log(bytes) / Math.log(k));
      return parseFloat((bytes / Math.pow(k, i)).toFixed(1)) + " " + sizes[i];
    },
    initMediaLibraryDisplay: function() {
      var self = this;
      self.observeMediaLibrary();
    },
    observeMediaLibrary: function() {
      var self = this;
      self.customizeAllOriginalLinks();
      var observer = new MutationObserver(function(mutations) {
        self.customizeAllOriginalLinks();
      });
      observer.observe(document.body, {
        childList: true,
        subtree: true
      });
    },
    customizeAllOriginalLinks: function() {
      $("p, span, div").each(function() {
        var $el = $(this);
        if ($el.data("swift-offload-original-processed")) return;
        var directText = $el.contents().filter(function() {
          return this.nodeType === 3;
        }).text();
        if (directText.indexOf("Original image") === -1) return;
        var $link = $el.find("a").first();
        if (!$link.length) return;
        $el.data("swift-offload-original-processed", true);
        $link.text("View Original").attr("target", "_blank");
        var $editImage = $el.siblings().filter(function() {
          return $(this).text().trim() === "Edit Image" || $(this).hasClass("edit-attachment");
        }).first();
        if ($editImage.length) {
          $el.remove();
          $editImage.after('<span class="swift-offload-separator"> | </span>');
          $editImage.next(".swift-offload-separator").after($link);
        }
      });
    },
    buildOptimizationCardHTML: function(data) {
      var timeDisplay = "";
      if (data.processingTime) {
        timeDisplay = data.processingTime >= 1e3 ? (data.processingTime / 1e3).toFixed(1) + " sec" : data.processingTime + " ms";
      }
      var outputFormat = (data.outputFormat || "webp").toUpperCase();
      var html = '<div class="swift-offload-media-optimization-info swift-offload-optimized-card" style="display: inline-block; clear: both; margin: 20px 0 16px 0; padding: 20px; background: #f0fdf4; border: 1px solid #bbf7d0; border-radius: 12px; font-family: -apple-system, BlinkMacSystemFont, sans-serif;">';
      html += '<div style="display: flex; align-items: center; gap: 8px; margin-bottom: 20px; color: #166534; font-weight: 600; font-size: 13px; white-space: nowrap;">';
      html += '<svg width="18" height="18" viewBox="0 0 24 24" fill="none" style="flex-shrink: 0;"><circle cx="12" cy="12" r="11" fill="#22c55e"/><path d="M8 12l3 3 5-6" stroke="#fff" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"/></svg>';
      html += "<span>Optimized by Swift Offload</span>";
      html += "</div>";
      html += '<div style="font-size: 26px; font-weight: 700; color: #16a34a; margin-bottom: 20px;">Saved ' + data.percentage + "%</div>";
      html += '<div style="font-size: 14px; color: #374151; margin-bottom: 20px; line-height: 1.8;">';
      html += "<div>Original: <strong>" + data.originalFormatted + "</strong></div>";
      html += '<div>Optimized: <strong style="color: #16a34a;">' + data.optimizedFormatted + "</strong></div>";
      html += '<div>Saved: <strong style="color: #16a34a;">-' + data.savedFormatted + "</strong></div>";
      html += "</div>";
      html += '<div style="font-size: 13px; color: #6b7280; line-height: 1.8;">';
      html += '<div>🖼️ Converted to <strong style="color: #111;">' + outputFormat + "</strong></div>";
      if (timeDisplay) {
        html += '<div>⚡ Done in <strong style="color: #111;">' + timeDisplay + "</strong></div>";
      }
      html += "</div>";
      html += "</div>";
      return html;
    },
    addOptimizationInfoToSidebar: function(view) {
      var self = this;
      var model = view.model;
      if (!model || !model.get) return;
      var optData = model.get("swiftOffloadOptimization");
      var $el = view.$el;
      $el.find(".swift-offload-media-optimization-info").remove();
      if (optData && optData.optimized) {
        var html = self.buildOptimizationCardHTML(optData);
        var $details = $el.find(".attachment-info, .details").first();
        if ($details.length) $details.append(html);
      }
    },
    updateOptimizationDisplay: function(attachmentId, stats) {
      var self = this;
      var percentage = stats.percentage.toFixed(2);
      var data = {
        percentage,
        originalFormatted: self.formatBytes(stats.originalSize),
        optimizedFormatted: self.formatBytes(stats.optimizedSize),
        savedFormatted: self.formatBytes(stats.saved),
        processingTime: stats.processingTime,
        outputFormat: stats.outputFormat
      };
      var html = self.buildOptimizationCardHTML(data);
      $(".swift-offload-media-optimization-info, .swift-offload-optimization-card, .swift-offload-optimized-card").remove();
      setTimeout(function() {
        var $details = $(".attachment-details .details, .attachment-info").first();
        if ($details.length) $details.append(html);
      }, 100);
    },
    initAdminPanel: function() {
      var self = this;
      $(document).on("submit", "#swift-offload-image-optimizer-form", function(e) {
        e.preventDefault();
        self.saveSettings($(this));
      });
      $(document).on("click", ".swift-offload-image-optimizer-save", function(e) {
        e.preventDefault();
        self.saveSettings($("#swift-offload-image-optimizer-form"));
      });
      $(document).on("click", "#swift-offload-reset-optimizer-stats", function(e) {
        e.preventDefault();
        self.resetStats();
      });
      $(document).on("input", "#swift-offload-webp-quality", function() {
        $("#swift-offload-webp-quality-value").text($(this).val() + "%");
        self.updateSliderFill($(this));
      });
      $(document).on("click", "#swift-offload-apply-recommended", function(e) {
        e.preventDefault();
        self.applyRecommended();
      });
      $(document).on("click", "#swift-offload-bulk-optimize-all", function(e) {
        e.preventDefault();
        self.openBulkOptimizeModal();
      });
      $(document).on("click", ".swift-offload-vertical-tab-btn", function(e) {
        e.preventDefault();
        var tabId = $(this).data("tab");
        $(".swift-offload-vertical-tab-btn").removeClass("active");
        $(this).addClass("active");
        $(".swift-offload-vertical-tab-pane").removeClass("active");
        $("#swift-offload-tab-" + tabId).addClass("active");
        if (tabId === "image") {
          $("#swift-offload-bulk-optimize-all").show();
        } else {
          $("#swift-offload-bulk-optimize-all").hide();
        }
      });
      $(document).on("change", 'input[name="conversion_mode"]', function() {
        var mode = $(this).val();
        $(".swift-offload-radio-pill").removeClass("active");
        $(this).closest(".swift-offload-radio-pill").addClass("active");
        if (mode === "general") {
          $(".swift-offload-conversion-general").show();
          $(".swift-offload-conversion-custom").hide();
        } else {
          $(".swift-offload-conversion-general").hide();
          $(".swift-offload-conversion-custom").show();
        }
      });
      $(document).on("click", "#swift-offload-add-conversion-rule", function(e) {
        e.preventDefault();
        self.addConversionRule();
      });
      $(document).on("click", ".swift-offload-repeater-remove", function(e) {
        e.preventDefault();
        $(this).closest(".swift-offload-repeater-item").fadeOut(200, function() {
          $(this).remove();
          self.reindexRepeaterItems();
        });
      });
      $(document).on("change", ".swift-offload-repeater-item select", function() {
        self.validateConversionRule($(this).closest(".swift-offload-repeater-item"));
      });
      self.initBulkActions();
      $(document).on("input", "#swift-offload-lqip-quality", function() {
        $("#swift-offload-lqip-quality-value").text($(this).val() + "%");
        self.updateSliderFill($(this));
      });
      $(document).on("input", "#swift-offload-lqip-blur", function() {
        $("#swift-offload-lqip-blur-value").text($(this).val() + "px");
        self.updateSliderFill($(this));
      });
      self.initSliders();
    },
    /**
     * Initialize all range sliders with proper fill
     */
    initSliders: function() {
      var self = this;
      $('.swift-offload-slider-control input[type="range"]').each(function() {
        self.updateSliderFill($(this));
      });
    },
    /**
     * Update slider fill based on current value
     */
    updateSliderFill: function($slider) {
      var min = parseFloat($slider.attr("min")) || 0;
      var max = parseFloat($slider.attr("max")) || 100;
      var val = parseFloat($slider.val()) || 0;
      var percentage = (val - min) / (max - min) * 100;
      $slider.css("--value", percentage + "%");
    },
    addConversionRule: function() {
      var $container = $("#swift-offload-conversion-rules");
      var index = $container.find(".swift-offload-repeater-item").length;
      var sourceFormats = ["heic", "gif", "avif", "webp", "jpeg", "png"];
      var targetFormats = ["avif", "webp", "jpeg", "png"];
      var sourceOptions = sourceFormats.map(function(f) {
        return '<option value="' + f + '">' + f.toUpperCase() + "</option>";
      }).join("");
      var targetOptions = targetFormats.map(function(f) {
        return '<option value="' + f + '">' + f.toUpperCase() + "</option>";
      }).join("");
      var html = '<div class="swift-offload-repeater-item" data-index="' + index + '"><div class="swift-offload-repeater-fields"><div class="swift-offload-repeater-field"><label>Source</label><select name="conversion_rules[' + index + '][source]" class="swift-offload-select-sm">' + sourceOptions + '</select></div><div class="swift-offload-repeater-arrow"><svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><line x1="5" y1="12" x2="19" y2="12"/><polyline points="12 5 19 12 12 19"/></svg></div><div class="swift-offload-repeater-field"><label>Target</label><select name="conversion_rules[' + index + '][target]" class="swift-offload-select-sm">' + targetOptions + '</select></div><div class="swift-offload-repeater-field swift-offload-repeater-quality"><label>Quality</label><input type="number" name="conversion_rules[' + index + '][quality]" value="80" min="1" max="100" class="swift-offload-input-sm"></div></div><button type="button" class="swift-offload-repeater-remove" title="Remove"><svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><line x1="18" y1="6" x2="6" y2="18"/><line x1="6" y1="6" x2="18" y2="18"/></svg></button></div>';
      $container.append(html);
    },
    reindexRepeaterItems: function() {
      $("#swift-offload-conversion-rules .swift-offload-repeater-item").each(function(index) {
        $(this).attr("data-index", index);
        $(this).find("select, input").each(function() {
          var name = $(this).attr("name");
          if (name) {
            $(this).attr("name", name.replace(/\[\d+\]/, "[" + index + "]"));
          }
        });
      });
    },
    validateConversionRule: function($item) {
      var $sourceSelect = $item.find('select[name*="[source]"]');
      var $targetSelect = $item.find('select[name*="[target]"]');
      var sourceVal = $sourceSelect.val();
      var targetVal = $targetSelect.val();
      var normalizedSource = sourceVal === "jpeg" || sourceVal === "jpg" ? "jpeg" : sourceVal;
      var normalizedTarget = targetVal === "jpeg" || targetVal === "jpg" ? "jpeg" : targetVal;
      if (normalizedSource === normalizedTarget) {
        $targetSelect.css("border-color", "#dc2626");
        $item.find(".swift-offload-format-error").remove();
        $targetSelect.after('<span class="swift-offload-format-error" style="color: #dc2626; font-size: 11px; display: block; margin-top: 4px;">Same format not allowed</span>');
        var formats = ["webp", "avif", "png", "jpeg"];
        for (var i = 0; i < formats.length; i++) {
          if (formats[i] !== normalizedSource) {
            $targetSelect.val(formats[i]);
            break;
          }
        }
        setTimeout(function() {
          $targetSelect.css("border-color", "");
          $item.find(".swift-offload-format-error").fadeOut(300, function() {
            $(this).remove();
          });
        }, 2e3);
      } else {
        $targetSelect.css("border-color", "");
        $item.find(".swift-offload-format-error").remove();
      }
    },
    initBulkActions: function() {
      var self = this;
      if (!$("body").hasClass("upload-php")) return;
      var bulkOption = '<option value="swift_offload_bulk_optimize">Swift Offload Optimize</option>';
      $('select[name="action"] option:first, select[name="action2"] option:first').after(bulkOption);
      $(document).on("click", "#doaction, #doaction2", function(e) {
        var $select = $(this).prev("select");
        if ($select.val() === "swift_offload_bulk_optimize") {
          e.preventDefault();
          self.handleBulkOptimize();
        }
      });
      $(document).on("click", ".swift-offload-convert-btn", function(e) {
        e.preventDefault();
        e.stopPropagation();
        var $popover = $(this).siblings(".swift-offload-convert-popover");
        $(".swift-offload-convert-popover").not($popover).hide();
        $popover.toggle();
      });
      $(document).on("click", function(e) {
        if (!$(e.target).closest(".swift-offload-convert-dropdown").length) {
          $(".swift-offload-convert-popover").hide();
        }
      });
      $(document).on("click", ".swift-offload-convert-option", function(e) {
        e.preventDefault();
        var attachmentId = $(this).data("attachment-id");
        var targetFormat = $(this).data("target-format");
        var $column = $(this).closest(".swift-offload-media-column");
        $(".swift-offload-convert-popover").hide();
        $column.addClass("swift-offload-loading");
        $(this).closest(".swift-offload-convert-dropdown").find(".swift-offload-convert-btn").html('<span class="swift-offload-spinner"></span>');
        self.convertAttachment(attachmentId, targetFormat, $column);
      });
      $(document).on("click", ".swift-offload-optimize-single-btn", function(e) {
        e.preventDefault();
        var $btn = $(this);
        var attachmentId = $btn.data("attachment-id");
        var $column = $btn.closest(".swift-offload-media-column");
        $column.addClass("swift-offload-loading");
        $btn.html('<span class="swift-offload-spinner"></span>');
        self.optimizeSingleAttachment(attachmentId, $column);
      });
    },
    convertAttachment: function(attachmentId, targetFormat, $column) {
      var self = this;
      $.ajax({
        url: swiftOffloadImageOptimizer.ajaxUrl,
        type: "POST",
        data: {
          action: "swift_offload_convert_attachment",
          security: swiftOffloadImageOptimizer.nonce,
          attachment_id: attachmentId,
          target_format: targetFormat
        },
        success: function(response) {
          if (response.success) {
            if (response.data.no_reduction) {
              self.showToast("No size reduction with " + targetFormat.toUpperCase() + " format", "info");
              $column.removeClass("swift-offload-loading");
              $column.find(".swift-offload-convert-btn").html("Convert ▾");
            } else {
              self.showToast("Converted to " + targetFormat.toUpperCase() + "! Saved " + response.data.percentage + "%", "success");
              location.reload();
            }
          } else {
            self.showToast(response.data.message || "Conversion failed", "error");
            $column.removeClass("swift-offload-loading");
            $column.find(".swift-offload-convert-btn").html("Convert ▾");
          }
        },
        error: function() {
          self.showToast("Connection error", "error");
          $column.removeClass("swift-offload-loading");
          $column.find(".swift-offload-convert-btn").html("Convert ▾");
        }
      });
    },
    optimizeSingleAttachment: function(attachmentId, $column) {
      var self = this;
      $.ajax({
        url: swiftOffloadImageOptimizer.ajaxUrl,
        type: "POST",
        data: {
          action: "swift_offload_bulk_optimize_attachment",
          security: swiftOffloadImageOptimizer.nonce,
          attachment_id: attachmentId
        },
        success: function(response) {
          if (response.success) {
            if (response.data.no_reduction) {
              self.showToast("Already optimized - no further reduction possible", "info");
            } else if (response.data.already_optimized) {
              self.showToast("Already optimized", "info");
            } else {
              self.showToast("Optimized! Saved " + response.data.percentage + "%", "success");
            }
            location.reload();
          } else {
            self.showToast(response.data.message || "Optimization failed", "error");
            $column.removeClass("swift-offload-loading");
            $column.find(".swift-offload-optimize-single-btn").html("Optimize");
          }
        },
        error: function() {
          self.showToast("Connection error", "error");
          $column.removeClass("swift-offload-loading");
          $column.find(".swift-offload-optimize-single-btn").html("Optimize");
        }
      });
    },
    handleBulkOptimize: function() {
      var self = this;
      var selectedIds = [];
      $('input[name="media[]"]:checked').each(function() {
        selectedIds.push($(this).val());
      });
      if (selectedIds.length === 0) {
        self.showToast("Please select images to optimize", "warning");
        return;
      }
      self.showBulkProgressModal(selectedIds.length);
      self.processBulkOptimization(selectedIds, 0, {
        total: selectedIds.length,
        processed: 0,
        saved: 0,
        originalSize: 0,
        optimizedSize: 0,
        skipped: 0
      });
    },
    showBulkProgressModal: function(total) {
      var self = this;
      $("#swift-offload-bulk-progress-inline").remove();
      var $bulkActions = $(".tablenav.top .bulkactions");
      var html = '<span id="swift-offload-bulk-progress-inline" style="display: inline-flex; align-items: center; gap: 10px; margin-left: 12px; padding: 6px 14px; background: #f0fdf4; border: 1px solid #86efac; margin-top: -6px; border-radius: 6px; font-size: 13px; vertical-align: middle;"><span style="display: inline-flex; align-items: center; gap: 6px;"><svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="#16a34a" stroke-width="2.5" class="swift-offload-spin-animation"><circle cx="12" cy="12" r="10" stroke-dasharray="60" stroke-dashoffset="15"/></svg><span id="swift-offload-bulk-status" style="color: #16a34a; font-weight: 500; white-space: nowrap;">Optimizing...</span></span><span style="width: 80px; background: #dcfce7; border-radius: 99px; height: 6px; overflow: hidden; display: inline-block; vertical-align: middle;"><span id="swift-offload-bulk-progress-bar" style="display: block; background: #16a34a; height: 100%; width: 0%; transition: width 0.3s ease; border-radius: 99px;"></span></span><span id="swift-offload-bulk-count" style="color: #16a34a; font-weight: 600; white-space: nowrap;">0/' + total + '</span><span style="color: #d1d5db;">|</span><span id="swift-offload-bulk-memory" style="color: #374151; white-space: nowrap;"><span style="color: #6b7280;">Main:</span> <span id="swift-offload-bulk-saved" style="color: #16a34a; font-weight: 600;">0</span>% <span style="color: #9ca3af;">(0 B → 0 B)</span></span><button type="button" id="swift-offload-bulk-cancel" style="margin-left: 4px; padding: 4px 12px; background: #fff; border: 1px solid #e5e7eb; border-radius: 4px; color: #374151; font-size: 12px; font-weight: 500; cursor: pointer; white-space: nowrap;">Refresh</button></span>';
      $bulkActions.append(html);
      if ($("#swift-offload-bulk-spin-style").length === 0) {
        $('<style id="swift-offload-bulk-spin-style">.swift-offload-spin-animation { animation: swift-offload-bulk-spin 1s linear infinite; } @keyframes swift-offload-bulk-spin { from { transform: rotate(0deg); } to { transform: rotate(360deg); } } #swift-offload-bulk-cancel:hover { background: #fee2e2; border-color: #fca5a5; }</style>').appendTo("head");
      }
      $("#swift-offload-bulk-cancel").on("click", function() {
        self.bulkCancelled = true;
        $("#swift-offload-bulk-progress-inline").remove();
        self.showToast("Bulk optimization cancelled", "info");
      });
      self.bulkCancelled = false;
    },
    processBulkOptimization: function(ids, index, stats) {
      var self = this;
      if (self.bulkCancelled || index >= ids.length) {
        self.finishBulkOptimization(stats);
        return;
      }
      var attachmentId = ids[index];
      var progress = Math.round((index + 1) / stats.total * 100);
      $("#swift-offload-bulk-progress-bar").css("width", progress + "%");
      $("#swift-offload-bulk-count").text(index + 1 + "/" + stats.total);
      $("#swift-offload-bulk-status").text("Optimizing image " + (index + 1) + " of " + stats.total + "...");
      $.ajax({
        url: swiftOffloadImageOptimizer.ajaxUrl,
        type: "POST",
        data: {
          action: "swift_offload_bulk_optimize_attachment",
          security: swiftOffloadImageOptimizer.nonce,
          attachment_id: attachmentId
        },
        success: function(response) {
          if (response.success && response.data) {
            stats.processed++;
            if (response.data.already_optimized || response.data.no_reduction) {
              stats.skipped++;
            } else {
              stats.saved += response.data.saved || 0;
              stats.originalSize += response.data.original_size || 0;
              stats.optimizedSize += response.data.optimized_size || 0;
            }
            if (stats.originalSize > 0) {
              var savedPercent = Math.round((1 - stats.optimizedSize / stats.originalSize) * 100);
              var originalFormatted = self.formatBytes(stats.originalSize);
              var optimizedFormatted = self.formatBytes(stats.optimizedSize);
              $("#swift-offload-bulk-saved").text(savedPercent);
              $("#swift-offload-bulk-memory").html('<span style="color: #6b7280;">Main:</span> <span style="color: #16a34a; font-weight: 600;">' + savedPercent + '</span>% <span style="color: #9ca3af;">(' + originalFormatted + " → " + optimizedFormatted + ")</span>");
            }
          }
        },
        complete: function() {
          setTimeout(function() {
            self.processBulkOptimization(ids, index + 1, stats);
          }, 100);
        }
      });
    },
    finishBulkOptimization: function(stats) {
      var self = this;
      var savedPercent = stats.originalSize > 0 ? Math.round((1 - stats.optimizedSize / stats.originalSize) * 100) : 0;
      var savedBytes = stats.originalSize - stats.optimizedSize;
      var optimizedCount = stats.processed - stats.skipped;
      var originalFormatted = self.formatBytes(stats.originalSize);
      var optimizedFormatted = self.formatBytes(stats.optimizedSize);
      var savedFormatted = self.formatBytes(savedBytes);
      $("#swift-offload-bulk-progress-bar").css("width", "100%");
      $("#swift-offload-bulk-progress-inline .swift-offload-spin-animation").hide();
      if (stats.skipped === stats.total) {
        $("#swift-offload-bulk-status").html('<svg width="14" height="14" viewBox="0 0 24 24" fill="none" style="vertical-align: middle;"><circle cx="12" cy="12" r="11" fill="#f59e0b"/><path d="M12 8v4m0 4h.01" stroke="#fff" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"/></svg> Already Optimized');
        $("#swift-offload-bulk-memory").html('<span style="color: #92400e;">All selected images are already optimized</span>');
        self.showToast("All " + stats.total + " images are already optimized", "info");
      } else if (stats.skipped > 0) {
        $("#swift-offload-bulk-status").html('<svg width="14" height="14" viewBox="0 0 24 24" fill="none" style="vertical-align: middle;"><circle cx="12" cy="12" r="11" fill="#16a34a"/><path d="M8 12l3 3 5-6" stroke="#fff" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"/></svg> Complete!');
        $("#swift-offload-bulk-memory").html('<span style="color: #6b7280;">Main:</span> ' + savedPercent + '% <span style="color: #9ca3af;">(' + originalFormatted + " → " + optimizedFormatted + ')</span> <span style="color: #f59e0b;">(' + stats.skipped + " skipped)</span>");
        self.showToast("Optimized " + optimizedCount + " images, " + stats.skipped + " already optimized, saved " + savedFormatted, "success");
      } else {
        $("#swift-offload-bulk-status").html('<svg width="14" height="14" viewBox="0 0 24 24" fill="none" style="vertical-align: middle;"><circle cx="12" cy="12" r="11" fill="#16a34a"/><path d="M8 12l3 3 5-6" stroke="#fff" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"/></svg> Complete!');
        $("#swift-offload-bulk-saved").text(savedPercent);
        $("#swift-offload-bulk-memory").html('<span style="color: #6b7280;">Main:</span> ' + savedPercent + '% <span style="color: #9ca3af;">(' + originalFormatted + " → " + optimizedFormatted + ")</span>");
        self.showToast("Optimized " + stats.processed + " images, saved " + savedFormatted, "success");
      }
      $("#swift-offload-bulk-count").text(stats.processed + "/" + stats.total);
      $("#swift-offload-bulk-cancel").off("click").on("click", function() {
        location.reload();
      });
    },
    /**
     * Format bytes to human readable
     */
    formatBytes: function(bytes) {
      if (bytes === 0) return "0 B";
      var k = 1024;
      var sizes = ["B", "KB", "MB", "GB"];
      var i = Math.floor(Math.log(bytes) / Math.log(k));
      return parseFloat((bytes / Math.pow(k, i)).toFixed(1)) + " " + sizes[i];
    },
    saveSettings: function($form) {
      var self = this;
      var $btn = $(".swift-offload-image-optimizer-save");
      var originalText = $btn.text();
      $btn.text("Saving...").prop("disabled", true);
      var formData = {
        action: "swift_offload_save_image_optimizer_settings",
        security: $form.find('input[name="_wpnonce"]').val(),
        // Image settings
        enabled: $form.find("#swift-offload-optimizer-enabled").is(":checked") ? "yes" : "no",
        webp_quality: $form.find("#swift-offload-webp-quality").val(),
        max_image_dimension: $form.find("#swift-offload-max-dimension").val(),
        resize_enabled: $form.find("#swift-offload-resize-enabled").is(":checked") ? "yes" : "no",
        disable_wp_scaling: $form.find("#swift-offload-disable-wp-scaling").is(":checked") ? "yes" : "no",
        disable_thumbnail_generation: $form.find("#swift-offload-disable-thumbnails").is(":checked") ? "yes" : "no",
        disabled_thumbnail_sizes: [],
        // Conversion settings
        conversion_mode: $form.find('input[name="conversion_mode"]:checked').val() || "general",
        general_target_format: $form.find("#swift-offload-general-target").val() || "webp",
        conversion_rules: [],
        // LQIP settings
        generate_lqip: $form.find("#swift-offload-generate-lqip").is(":checked") ? "yes" : "no",
        lqip_quality: $form.find("#swift-offload-lqip-quality").val() || 20,
        lqip_blur: $form.find("#swift-offload-lqip-blur").val() || 10,
        // Video settings
        optimize_videos: $form.find("#swift-offload-optimize-videos").is(":checked") ? "yes" : "no",
        video_quality: $form.find("#swift-offload-video-quality").val() || "medium",
        video_max_resolution: $form.find("#swift-offload-video-max-res").val() || 1080,
        // Audio settings
        optimize_audio: $form.find("#swift-offload-optimize-audio").is(":checked") ? "yes" : "no",
        audio_bitrate: $form.find("#swift-offload-audio-bitrate").val() || 128
      };
      $form.find('input[name="disabled_thumbnail_sizes[]"]:checked').each(function() {
        formData.disabled_thumbnail_sizes.push($(this).val());
      });
      $form.find("#swift-offload-conversion-rules .swift-offload-repeater-item").each(function(index) {
        formData.conversion_rules.push({
          source: $(this).find('select[name*="[source]"]').val(),
          target: $(this).find('select[name*="[target]"]').val(),
          quality: $(this).find('input[name*="[quality]"]').val()
        });
      });
      $.ajax({
        url: swiftOffloadImageOptimizer.ajaxUrl,
        type: "POST",
        data: formData,
        success: function(res) {
          if (res.success) {
            self.settings = res.data.settings;
            self.showToast("Settings saved!", "success");
          } else {
            self.showToast(res.data.message || "Failed to save", "error");
          }
        },
        error: function() {
          self.showToast("Connection error", "error");
        },
        complete: function() {
          $btn.text(originalText).prop("disabled", false);
        }
      });
    },
    // Store original page content and stop flag
    bulkOptimizeOriginalContent: null,
    bulkOptimizeStopRequested: false,
    openBulkOptimizeModal: function() {
      var self = this;
      self.bulkOptimizeStopRequested = false;
      var $wrapper = $(".swift-offload-optimizer-wrapper");
      self.bulkOptimizeOriginalContent = $wrapper.html();
      var pageHtml = '<div id="swift-offload-bulk-optimize-page"><div class="swift-offload-bulk-header" style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 24px; padding-bottom: 16px; border-bottom: 1px solid #e5e7eb;"><div style="display: flex; align-items: center; gap: 16px;"><button type="button" id="swift-offload-bulk-back" class="swift-offload-btn swift-offload-btn-outline" style="padding: 8px 16px;"><svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M19 12H5M12 19l-7-7 7-7"/></svg> Back to Settings</button><h2 style="margin: 0; font-size: 20px; font-weight: 600; color: #111;">Bulk Image Optimization</h2></div><div style="display: flex; align-items: center; gap: 12px;"><span id="swift-offload-bulk-status" style="padding: 6px 12px; border-radius: 6px; font-size: 12px; font-weight: 600; background: #dbeafe; color: #1d4ed8;">LOADING</span><button type="button" id="swift-offload-bulk-stop" class="swift-offload-btn" style="padding: 8px 16px; background: #dc2626; color: #fff; border: none; display: none; align-items: center; gap: 6px;"><span style="display: inline-block; width: 14px; height: 14px; background: #fff; border-radius: 2px;"></span><span>Stop Processing</span></button></div></div><div id="swift-offload-bulk-content"><div style="text-align: center; padding: 60px 40px;"><div style="width: 48px; height: 48px; border: 3px solid #e5e7eb; border-top-color: #6366f1; border-radius: 50%; animation: swift-offload-spin 1s linear infinite; margin: 0 auto 20px;"></div><p style="color: #6b7280; margin: 0; font-size: 15px;">Loading unoptimized images...</p></div></div></div><style>@keyframes swift-offload-spin { from { transform: rotate(0deg); } to { transform: rotate(360deg); } }</style>';
      $wrapper.html(pageHtml);
      $(document).on("click", "#swift-offload-bulk-back", function() {
        self.bulkOptimizeStopRequested = true;
        $wrapper.html(self.bulkOptimizeOriginalContent);
        self.bulkOptimizeOriginalContent = null;
      });
      $(document).on("click", "#swift-offload-bulk-stop", function() {
        self.bulkOptimizeStopRequested = true;
        $(this).prop("disabled", true).css("opacity", "0.6").html(
          '<span style="display: inline-block; width: 14px; height: 14px; border: 2px solid #fff; border-top-color: transparent; border-radius: 50%; animation: swift-offload-spin 1s linear infinite;"></span><span style="margin-left: 6px;">Stopping...</span>'
        );
        $("#swift-offload-bulk-status").text("STOPPING").css({ background: "#fef3c7", color: "#92400e" });
      });
      $.ajax({
        url: swiftOffloadImageOptimizer.ajaxUrl,
        type: "POST",
        data: {
          action: "swift_offload_get_unoptimized_attachments",
          nonce: swiftOffloadImageOptimizer.nonce
        },
        success: function(res) {
          if (res.success && res.data.attachments && res.data.attachments.length > 0) {
            self.showBulkOptimizeConfirmation(res.data.attachments, res.data.settings);
          } else {
            $("#swift-offload-bulk-content").html(
              '<div style="text-align: center; padding: 60px 40px;"><svg width="64" height="64" viewBox="0 0 24 24" fill="none" stroke="#16a34a" stroke-width="2" style="margin-bottom: 20px;"><circle cx="12" cy="12" r="10"/><path d="M8 12l3 3 5-6"/></svg><h3 style="margin: 0 0 12px; color: #111; font-size: 20px;">All Images Optimized!</h3><p style="margin: 0; color: #6b7280; font-size: 15px;">No unoptimized images found in your media library.</p></div>'
            );
            $("#swift-offload-bulk-status").text("COMPLETE").css({ background: "#dcfce7", color: "#16a34a" });
          }
        },
        error: function() {
          $("#swift-offload-bulk-content").html(
            '<div style="text-align: center; padding: 60px 40px; color: #dc2626;"><svg width="64" height="64" viewBox="0 0 24 24" fill="none" stroke="#dc2626" stroke-width="2" style="margin-bottom: 20px;"><circle cx="12" cy="12" r="10"/><line x1="15" y1="9" x2="9" y2="15"/><line x1="9" y1="9" x2="15" y2="15"/></svg><h3 style="margin: 0 0 12px; font-size: 20px;">Error Loading Attachments</h3><p style="margin: 0; font-size: 15px;">Please try again or check your network connection.</p></div>'
          );
          $("#swift-offload-bulk-status").text("ERROR").css({ background: "#fee2e2", color: "#dc2626" });
        }
      });
    },
    showBulkOptimizeConfirmation: function(attachments, settings) {
      var self = this;
      var total = attachments.length;
      var totalSize = 0;
      attachments.forEach(function(att) {
        totalSize += att.filesize || 0;
      });
      var confirmHtml = '<div style="text-align: center; padding: 40px 20px;"><div style="width: 80px; height: 80px; margin: 0 auto 24px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); border-radius: 50%; display: flex; align-items: center; justify-content: center;"><svg width="40" height="40" viewBox="0 0 24 24" fill="none" stroke="#fff" stroke-width="2"><rect x="3" y="3" width="18" height="18" rx="2" ry="2"/><circle cx="8.5" cy="8.5" r="1.5"/><polyline points="21 15 16 10 5 21"/></svg></div><h3 style="margin: 0 0 8px; font-size: 24px; font-weight: 600; color: #111;">Ready to Optimize</h3><p style="margin: 0 0 32px; color: #6b7280; font-size: 15px;">Found <strong>' + total + ' images</strong> ready for optimization</p><div style="display: inline-flex; gap: 24px; margin-bottom: 32px; padding: 20px 32px; background: #f9fafb; border-radius: 12px;"><div style="text-align: center;"><div style="font-size: 28px; font-weight: 700; color: #374151;">' + total + '</div><div style="font-size: 12px; color: #6b7280;">Images</div></div><div style="width: 1px; background: #e5e7eb;"></div><div style="text-align: center;"><div style="font-size: 28px; font-weight: 700; color: #374151;">' + self.formatBytes(totalSize) + '</div><div style="font-size: 12px; color: #6b7280;">Total Size</div></div><div style="width: 1px; background: #e5e7eb;"></div><div style="text-align: center;"><div style="font-size: 28px; font-weight: 700; color: #374151;">' + (settings.output_format || "WEBP").toUpperCase() + '</div><div style="font-size: 12px; color: #6b7280;">Target Format</div></div></div><div style="margin-bottom: 16px;"><button type="button" id="swift-offload-bulk-start" class="swift-offload-btn swift-offload-btn-primary" style="padding: 14px 48px; font-size: 16px; font-weight: 600; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); border: none; border-radius: 8px; cursor: pointer;"><svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="vertical-align: middle; margin-right: 8px;"><polygon points="5 3 19 12 5 21 5 3"/></svg>Start Optimization</button></div><p style="margin: 0; color: #9ca3af; font-size: 13px;">You can stop the process at any time</p></div>';
      $("#swift-offload-bulk-content").html(confirmHtml);
      $("#swift-offload-bulk-status").text("READY").css({ background: "#dbeafe", color: "#1d4ed8" });
      $(document).off("click", "#swift-offload-bulk-start").on("click", "#swift-offload-bulk-start", function() {
        self.startBulkOptimizeProcess(attachments, settings);
      });
    },
    startBulkOptimizeProcess: function(attachments, settings) {
      var self = this;
      var total = attachments.length;
      var converted = 0;
      var totalOriginalSize = 0;
      var totalOptimizedSize = 0;
      var startTime = Date.now();
      $("#swift-offload-bulk-stop").show().css("display", "inline-flex");
      var maxDim = settings.max_dimension || 2560;
      var resizeEnabled = settings.resize_enabled === "yes";
      var statsHtml = '<div class="swift-offload-bulk-stats" style="display: grid; grid-template-columns: repeat(3, 1fr); gap: 16px; margin-bottom: 16px;"><div style="background: #f9fafb; padding: 16px 20px; border-radius: 8px; border: 1px solid #e5e7eb;"><div style="font-size: 13px; color: #6b7280;">Total Space Saved</div><div id="swift-offload-bm-saved" style="font-size: 22px; font-weight: 700; color: #111; margin-top: 4px;">0 B (0%)</div></div><div style="background: #f9fafb; padding: 16px 20px; border-radius: 8px; border: 1px solid #e5e7eb;"><div style="font-size: 13px; color: #6b7280;">Conversion Time</div><div id="swift-offload-bm-time" style="font-size: 22px; font-weight: 700; color: #111; margin-top: 4px;">0 seconds</div></div><div style="background: #f9fafb; padding: 16px 20px; border-radius: 8px; border: 1px solid #e5e7eb;"><div style="font-size: 13px; color: #6b7280;">Files Converted</div><div style="font-size: 22px; font-weight: 700; color: #111; margin-top: 4px;"><span id="swift-offload-bm-converted">0</span>/' + total + ' <span id="swift-offload-bm-pct" style="color: #6b7280;">(0%)</span></div></div></div><div style="display: grid; grid-template-columns: repeat(5, 1fr); gap: 12px; margin-bottom: 20px;"><div style="background: #f9fafb; padding: 12px 16px; border-radius: 6px; border: 1px solid #e5e7eb;"><span style="font-size: 12px; color: #6b7280;">Source Format:</span><strong style="float: right; color: #111;">' + (settings.source_format || "JPEG/PNG").toUpperCase() + '</strong></div><div style="background: #f9fafb; padding: 12px 16px; border-radius: 6px; border: 1px solid #e5e7eb;"><span style="font-size: 12px; color: #6b7280;">Target Format:</span><strong style="float: right; color: #111;">' + (settings.output_format || "WEBP").toUpperCase() + '</strong></div><div style="background: #f9fafb; padding: 12px 16px; border-radius: 6px; border: 1px solid #e5e7eb;"><span style="font-size: 12px; color: #6b7280;">Library:</span><strong style="float: right; color: #111;">' + (settings.driver || "Imagick") + '</strong></div><div style="background: #f9fafb; padding: 12px 16px; border-radius: 6px; border: 1px solid #e5e7eb;"><span style="font-size: 12px; color: #6b7280;">Mode:</span><strong style="float: right; color: #111;">Lossy (Q: ' + (settings.quality || 80) + ')</strong></div><div style="background: #f9fafb; padding: 12px 16px; border-radius: 6px; border: 1px solid #e5e7eb;"><span style="font-size: 12px; color: #6b7280;">Resize:</span><strong style="float: right; color: #111;">' + (resizeEnabled ? maxDim + "px" : "Off") + '</strong></div></div><h4 style="margin: 0 0 12px; font-size: 15px; font-weight: 600; color: #111;">Converted Files</h4><div style="border: 1px solid #e5e7eb; border-radius: 8px; overflow: hidden; max-height: 400px; overflow-y: auto;"><table style="width: 100%; border-collapse: collapse; font-size: 13px;"><thead style="position: sticky; top: 0; background: #f9fafb;"><tr><th style="padding: 12px 16px; text-align: left; font-weight: 500; color: #374151; border-bottom: 1px solid #e5e7eb;">File</th><th style="padding: 12px 16px; text-align: right; font-weight: 500; color: #374151; border-bottom: 1px solid #e5e7eb;">Original</th><th style="padding: 12px 16px; text-align: right; font-weight: 500; color: #374151; border-bottom: 1px solid #e5e7eb;">Optimized</th><th style="padding: 12px 16px; text-align: right; font-weight: 500; color: #374151; border-bottom: 1px solid #e5e7eb;">Savings</th><th style="padding: 12px 16px; text-align: center; font-weight: 500; color: #374151; border-bottom: 1px solid #e5e7eb;">Status</th></tr></thead><tbody id="swift-offload-bm-results"></tbody></table></div>';
      $("#swift-offload-bulk-content").html(statsHtml);
      $("#swift-offload-bulk-status").text("PROCESSING").css({ background: "#dbeafe", color: "#1d4ed8" });
      function processNext(index) {
        if (self.bulkOptimizeStopRequested) {
          var elapsedTime = ((Date.now() - startTime) / 1e3).toFixed(2);
          $("#swift-offload-bm-time").text(elapsedTime + " seconds");
          $("#swift-offload-bulk-status").text("STOPPED").css({ background: "#fef3c7", color: "#92400e" });
          $("#swift-offload-bulk-stop").hide();
          return;
        }
        if (index >= total) {
          var elapsedTime = ((Date.now() - startTime) / 1e3).toFixed(2);
          var savedBytes = totalOriginalSize - totalOptimizedSize;
          var savedPercent = totalOriginalSize > 0 ? Math.round(savedBytes / totalOriginalSize * 100) : 0;
          var convertedPct = Math.round(converted / total * 100);
          $("#swift-offload-bm-saved").text(self.formatBytes(savedBytes) + " (" + savedPercent + "%)");
          $("#swift-offload-bm-time").text(elapsedTime + " seconds");
          $("#swift-offload-bm-converted").text(converted);
          $("#swift-offload-bm-pct").text("(" + convertedPct + "%)");
          $("#swift-offload-bulk-status").text("SUCCESS").css({ background: "#dcfce7", color: "#16a34a" });
          $("#swift-offload-bulk-stop").hide();
          return;
        }
        var attachment = attachments[index];
        $.ajax({
          url: swiftOffloadImageOptimizer.ajaxUrl,
          type: "POST",
          data: {
            action: "swift_offload_bulk_optimize_attachment",
            security: swiftOffloadImageOptimizer.nonce,
            attachment_id: attachment.id
          },
          success: function(res) {
            var row = "";
            var filename = attachment.filename || "Unknown";
            var displayName = filename.length > 40 ? filename.substring(0, 37) + "..." : filename;
            if (res.success && res.data && res.data.percentage > 0) {
              converted++;
              totalOriginalSize += res.data.original_size || 0;
              totalOptimizedSize += res.data.optimized_size || 0;
              var savedAmt = (res.data.original_size || 0) - (res.data.optimized_size || 0);
              var savedPct = res.data.percentage || 0;
              row = '<tr style="border-top: 1px solid #f3f4f6;"><td style="padding: 10px 16px;" title="' + filename + '">' + displayName + '</td><td style="padding: 10px 16px; text-align: right; color: #6b7280;">' + self.formatBytes(res.data.original_size) + '</td><td style="padding: 10px 16px; text-align: right; color: #374151;">' + self.formatBytes(res.data.optimized_size) + '</td><td style="padding: 10px 16px; text-align: right; color: #16a34a; font-weight: 500;">-' + self.formatBytes(savedAmt) + " (" + savedPct.toFixed(0) + '%)</td><td style="padding: 10px 16px; text-align: center;"><span style="padding: 4px 10px; border-radius: 12px; font-size: 11px; font-weight: 600; background: #dcfce7; color: #16a34a;">CONVERTED</span></td></tr>';
            } else {
              row = '<tr style="border-top: 1px solid #f3f4f6;"><td style="padding: 10px 16px;" title="' + filename + '">' + displayName + '</td><td style="padding: 10px 16px; text-align: right; color: #9ca3af;">' + self.formatBytes(res.data && res.data.original_size || 0) + '</td><td style="padding: 10px 16px; text-align: right; color: #9ca3af;">—</td><td style="padding: 10px 16px; text-align: right; color: #9ca3af;">—</td><td style="padding: 10px 16px; text-align: center;"><span style="padding: 4px 10px; border-radius: 12px; font-size: 11px; font-weight: 600; background: #fef3c7; color: #92400e;">SKIPPED</span></td></tr>';
            }
            $("#swift-offload-bm-results").prepend(row);
            var elapsedTime2 = ((Date.now() - startTime) / 1e3).toFixed(2);
            var savedBytes2 = totalOriginalSize - totalOptimizedSize;
            var savedPercent2 = totalOriginalSize > 0 ? Math.round(savedBytes2 / totalOriginalSize * 100) : 0;
            var convertedPct2 = Math.round(converted / total * 100);
            $("#swift-offload-bm-saved").text(self.formatBytes(savedBytes2) + " (" + savedPercent2 + "%)");
            $("#swift-offload-bm-time").text(elapsedTime2 + " seconds");
            $("#swift-offload-bm-converted").text(converted);
            $("#swift-offload-bm-pct").text("(" + convertedPct2 + "%)");
            setTimeout(function() {
              processNext(index + 1);
            }, 50);
          },
          error: function() {
            $("#swift-offload-bm-results").prepend(
              '<tr style="border-top: 1px solid #f3f4f6;"><td style="padding: 10px 16px;">' + (attachment.filename || "Unknown") + '</td><td colspan="3" style="padding: 10px 16px; color: #dc2626;">Error processing file</td><td style="padding: 10px 16px; text-align: center;"><span style="padding: 4px 10px; border-radius: 12px; font-size: 11px; font-weight: 600; background: #fee2e2; color: #dc2626;">ERROR</span></td></tr>'
            );
            setTimeout(function() {
              processNext(index + 1);
            }, 50);
          }
        });
      }
      processNext(0);
    },
    resetStats: function() {
      var self = this;
      if (!confirm("Reset all optimization statistics?")) return;
      $.ajax({
        url: swiftOffloadImageOptimizer.ajaxUrl,
        type: "POST",
        data: {
          action: "swift_offload_reset_optimizer_stats",
          nonce: swiftOffloadImageOptimizer.nonce
        },
        success: function(res) {
          if (res.success) {
            $("#swift-offload-stat-total-files").text("0");
            $("#swift-offload-stat-total-saved").text("0 B");
            $("#swift-offload-stat-original-size").text("0 B");
            $("#swift-offload-stat-optimized-size").text("0 B");
            self.showToast("Stats reset!", "success");
          }
        }
      });
    },
    applyRecommended: function() {
      $("#swift-offload-optimizer-enabled").prop("checked", true);
      $("#swift-offload-convert-webp").prop("checked", true);
      $("#swift-offload-webp-quality").val(80).trigger("input");
      $("#swift-offload-max-dimension").val(2560);
      $("#swift-offload-resize-enabled").prop("checked", true);
      $("#swift-offload-disable-wp-scaling").prop("checked", false);
      $("#swift-offload-disable-thumbnails").prop("checked", false);
      this.showToast("Recommended settings applied!", "success");
    }
  };
  $(document).ready(function() {
    SwiftOffload_ImageOptimizer.init();
  });
})(jQuery);
