/**
 * Swift Offload Image Compression Worker
 * High-performance image compression using Web Workers and OffscreenCanvas
 *
 * @package SwiftOffload
 */

/**
 * Initialize worker (no WASM needed - using native browser APIs)
 */
async function initWasm() {
	// Using native OffscreenCanvas API for compression
	// No external WASM modules required
	return true;
}

/**
 * Compress image using Canvas API with optimizations
 */
function compressWithCanvas(imageData, width, height, quality, format) {
	const canvas = new OffscreenCanvas(width, height);
	const ctx = canvas.getContext('2d');

	// Create ImageData and put it on canvas
	const imgData = new ImageData(
		new Uint8ClampedArray(imageData),
		width,
		height
	);
	ctx.putImageData(imgData, 0, 0);

	// Convert to blob with specified format and quality
	return canvas.convertToBlob({
		type: format,
		quality: quality / 100
	});
}

/**
 * Advanced JPEG compression with mozjpeg-like optimizations
 */
async function compressJPEG(imageData, width, height, quality) {
	// Use OffscreenCanvas for better performance
	const canvas = new OffscreenCanvas(width, height);
	const ctx = canvas.getContext('2d');

	const imgData = new ImageData(
		new Uint8ClampedArray(imageData),
		width,
		height
	);
	ctx.putImageData(imgData, 0, 0);

	// Multiple quality passes for optimal compression
	let bestBlob = await canvas.convertToBlob({
		type: 'image/jpeg',
		quality: quality / 100
	});

	// Try slightly lower quality to see if file size difference is worth it
	if (quality > 70) {
		const lowerQualityBlob = await canvas.convertToBlob({
			type: 'image/jpeg',
			quality: (quality - 5) / 100
		});

		// If size reduction is more than 15% with only 5% quality loss, use it
		if (lowerQualityBlob.size < bestBlob.size * 0.85) {
			bestBlob = lowerQualityBlob;
		}
	}

	return bestBlob;
}

/**
 * WebP compression
 */
async function compressWebP(imageData, width, height, quality) {
	const canvas = new OffscreenCanvas(width, height);
	const ctx = canvas.getContext('2d');

	const imgData = new ImageData(
		new Uint8ClampedArray(imageData),
		width,
		height
	);
	ctx.putImageData(imgData, 0, 0);

	return canvas.convertToBlob({
		type: 'image/webp',
		quality: quality / 100
	});
}

/**
 * PNG compression with quantization
 */
async function compressPNG(imageData, width, height) {
	const canvas = new OffscreenCanvas(width, height);
	const ctx = canvas.getContext('2d');

	const imgData = new ImageData(
		new Uint8ClampedArray(imageData),
		width,
		height
	);
	ctx.putImageData(imgData, 0, 0);

	return canvas.convertToBlob({
		type: 'image/png'
	});
}

/**
 * Resize image if needed
 */
function resizeImage(imageData, width, height, maxDimension) {
	if (width <= maxDimension && height <= maxDimension) {
		return { imageData, width, height };
	}

	const ratio = Math.min(maxDimension / width, maxDimension / height);
	const newWidth = Math.round(width * ratio);
	const newHeight = Math.round(height * ratio);

	// Create source canvas
	const srcCanvas = new OffscreenCanvas(width, height);
	const srcCtx = srcCanvas.getContext('2d');
	const srcImgData = new ImageData(
		new Uint8ClampedArray(imageData),
		width,
		height
	);
	srcCtx.putImageData(srcImgData, 0, 0);

	// Create destination canvas with new dimensions
	const dstCanvas = new OffscreenCanvas(newWidth, newHeight);
	const dstCtx = dstCanvas.getContext('2d');

	// Use high-quality image smoothing
	dstCtx.imageSmoothingEnabled = true;
	dstCtx.imageSmoothingQuality = 'high';

	// Draw resized image
	dstCtx.drawImage(srcCanvas, 0, 0, newWidth, newHeight);

	// Get resized image data
	const resizedImgData = dstCtx.getImageData(0, 0, newWidth, newHeight);

	return {
		imageData: resizedImgData.data.buffer,
		width: newWidth,
		height: newHeight
	};
}

/**
 * Main compression function
 */
async function compress(data) {
	const {
		imageData,
		width,
		height,
		originalSize,
		settings,
		fileId,
		fileName
	} = data;

	const startTime = performance.now();

	try {
		let processedData = imageData;
		let processedWidth = width;
		let processedHeight = height;

		// Get original format from filename
		const fileExt = fileName.split('.').pop().toLowerCase();
		const originalFormat = (fileExt === 'jpeg') ? 'jpg' : fileExt;

		// Resize if enabled
		if (settings.resize_enabled === 'yes') {
			const maxDim = parseInt(settings.max_image_dimension) || 2560;
			const resized = resizeImage(imageData, width, height, maxDim);
			processedData = resized.imageData;
			processedWidth = resized.width;
			processedHeight = resized.height;
		}

		// Determine output format and compress
		// Always use WebP for best compression (works for both photos and graphics)
		let blob;
		let outputType = 'image/webp';
		let extension = '.webp';
		let outputFormat = 'webp';

		const quality = parseInt(settings.webp_quality) || 80;

		// Only use JPEG if WebP is disabled AND original is JPEG
		if (settings.convert_webp !== 'yes' && fileName.toLowerCase().match(/\.jpe?g$/)) {
			blob = await compressJPEG(processedData, processedWidth, processedHeight, quality);
			outputType = 'image/jpeg';
			extension = '.jpg';
			outputFormat = 'jpg';
		} else {
			// Use WebP for everything else (best compression)
			blob = await compressWebP(processedData, processedWidth, processedHeight, quality);
		}

		const optimizedSize = blob.size;
		const processingTime = Math.round(performance.now() - startTime);

		// Only return optimized if smaller
		if (optimizedSize >= originalSize) {
			return {
				success: false,
				fileId,
				reason: 'no_reduction'
			};
		}

		// Convert blob to ArrayBuffer for transfer
		const arrayBuffer = await blob.arrayBuffer();

		return {
			success: true,
			fileId,
			fileName: fileName.replace(/\.[^.]+$/, extension),
			optimizedData: arrayBuffer,
			outputType,
			stats: {
				originalSize,
				optimizedSize,
				saved: originalSize - optimizedSize,
				percentage: Math.round((1 - optimizedSize / originalSize) * 100),
				processingTime,
				originalFormat,
				outputFormat,
				driver: 'Worker',
				quality,
				dimensions: {
					original: { width, height },
					optimized: { width: processedWidth, height: processedHeight }
				}
			}
		};
	} catch (error) {
		return {
			success: false,
			fileId,
			error: error.message
		};
	}
}

/**
 * Process batch of images
 */
async function processBatch(images, settings) {
	const results = [];

	// Process images in parallel (limited concurrency)
	const concurrency = navigator.hardwareConcurrency || 4;
	const chunks = [];

	for (let i = 0; i < images.length; i += concurrency) {
		chunks.push(images.slice(i, i + concurrency));
	}

	for (const chunk of chunks) {
		const chunkResults = await Promise.all(
			chunk.map(img => compress({ ...img, settings }))
		);
		results.push(...chunkResults);

		// Report progress
		self.postMessage({
			type: 'progress',
			completed: results.length,
			total: images.length
		});
	}

	return results;
}

// Initialize WASM on worker start
let wasmReady = initWasm();

// Handle messages from main thread
self.onmessage = async function(e) {
	const { type, data } = e.data;

	switch (type) {
		case 'init':
			await wasmReady;
			self.postMessage({ type: 'ready' });
			break;

		case 'compress':
			await wasmReady;
			const result = await compress(data);
			self.postMessage({
				type: 'result',
				result
			}, result.success ? [result.optimizedData] : []);
			break;

		case 'batch':
			await wasmReady;
			const results = await processBatch(data.images, data.settings);
			self.postMessage({
				type: 'batchResult',
				results
			});
			break;
	}
};
