/**
 * Swift Offload Admin JavaScript
 */

(function($) {
    'use strict';

    // Admin object
    window.SwiftOffloadAdmin = {

        /**
         * Initialize admin functionality
         */
        init: function() {
            this.bindEvents();
            this.initTooltips();
        },

        /**
         * Bind event handlers
         */
        bindEvents: function() {
            // Test connection button
            $(document).on('click', '.swift-offload-test-connection', this.testConnection);

            // Bulk actions
            $(document).on('click', '.swift-offload-bulk-offload', this.bulkOffload);
            $(document).on('click', '.swift-offload-bulk-remove-local', this.bulkRemoveLocal);

            // Individual attachment actions
            $(document).on('click', '.swift-offload-offload-single', this.offloadSingle);
            $(document).on('click', '.swift-offload-remove-local-single', this.removeLocalSingle);
        },

        /**
         * Initialize tooltips
         */
        initTooltips: function() {
            // Add tooltips for status indicators
            $('.swift-offload-offloaded, .swift-offload-local').each(function() {
                $(this).attr('title', $(this).data('tooltip') || '');
            });
        },

        /**
         * Test storage connection
         */
        testConnection: function(e) {
            e.preventDefault();

            var $button = $(this);
            var $form = $button.closest('form');

            // Show loading state
            $button.addClass('swift-offload-loading').prop('disabled', true);
            $button.find('.swift-offload-spinner').show();

            // Get form data
            var formData = new FormData($form[0]);
            var config = {};

            for (var pair of formData.entries()) {
                config[pair[0]] = pair[1];
            }

            // Make API request
            $.ajax({
                url: window.swiftOffloadAdmin.restUrl + 'test-connection',
                method: 'POST',
                headers: {
                    'X-WP-Nonce': window.swiftOffloadAdmin.nonce
                },
                data: JSON.stringify({
                    provider: 's3',
                    config: config
                }),
                contentType: 'application/json',
                success: function(response) {
                    SwiftOffloadAdmin.showNotice('Connection successful!', 'success');
                },
                error: function(xhr) {
                    var error = xhr.responseJSON ? xhr.responseJSON.message : 'Connection failed';
                    SwiftOffloadAdmin.showNotice(error, 'error');
                },
                complete: function() {
                    $button.removeClass('swift-offload-loading').prop('disabled', false);
                    $button.find('.swift-offload-spinner').hide();
                }
            });
        },

        /**
         * Bulk offload attachments
         */
        bulkOffload: function(e) {
            e.preventDefault();

            var attachmentIds = SwiftOffloadAdmin.getSelectedAttachments();

            if (attachmentIds.length === 0) {
                SwiftOffloadAdmin.showNotice('Please select attachments to offload', 'warning');
                return;
            }

            if (!confirm('Offload ' + attachmentIds.length + ' selected files to S3?')) {
                return;
            }

            SwiftOffloadAdmin.performBulkAction('offload', attachmentIds);
        },

        /**
         * Bulk remove local files
         */
        bulkRemoveLocal: function(e) {
            e.preventDefault();

            var attachmentIds = SwiftOffloadAdmin.getSelectedAttachments();

            if (attachmentIds.length === 0) {
                SwiftOffloadAdmin.showNotice('Please select attachments', 'warning');
                return;
            }

            if (!confirm('Remove local files for ' + attachmentIds.length + ' selected attachments?\n\nWARNING: This cannot be undone!')) {
                return;
            }

            SwiftOffloadAdmin.performBulkAction('remove-local', attachmentIds);
        },

        /**
         * Get selected attachment IDs
         */
        getSelectedAttachments: function() {
            var ids = [];
            $('.wp-list-table input[type="checkbox"]:checked').each(function() {
                var id = $(this).val();
                if (id && id !== 'on') {
                    ids.push(parseInt(id));
                }
            });
            return ids;
        },

        /**
         * Perform bulk action
         */
        performBulkAction: function(action, attachmentIds) {
            var endpoint = action === 'offload' ? 'media/offload' : 'media/remove-local';

            $.ajax({
                url: window.swiftOffloadAdmin.restUrl + endpoint,
                method: 'POST',
                headers: {
                    'X-WP-Nonce': window.swiftOffloadAdmin.nonce
                },
                data: JSON.stringify({
                    attachment_ids: attachmentIds
                }),
                contentType: 'application/json',
                success: function(response) {
                    SwiftOffloadAdmin.showNotice(response.message, 'success');
                    location.reload(); // Refresh to show updated status
                },
                error: function(xhr) {
                    var error = xhr.responseJSON ? xhr.responseJSON.message : 'Action failed';
                    SwiftOffloadAdmin.showNotice(error, 'error');
                }
            });
        },

        /**
         * Show admin notice
         */
        showNotice: function(message, type) {
            var $notice = $('<div class="notice notice-' + type + ' is-dismissible swift-offload-notice">')
                .append('<p>' + message + '</p>')
                .append('<button type="button" class="notice-dismiss"><span class="screen-reader-text">Dismiss</span></button>');

            $('.wrap h1').after($notice);

            // Auto-dismiss after 5 seconds
            setTimeout(function() {
                $notice.fadeOut(function() {
                    $(this).remove();
                });
            }, 5000);
        },

        /**
         * Format file size
         */
        formatFileSize: function(bytes) {
            if (bytes === 0) return '0 Bytes';

            var k = 1024;
            var sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB'];
            var i = Math.floor(Math.log(bytes) / Math.log(k));

            return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
        },

        /**
         * Update progress bar
         */
        updateProgress: function(selector, percentage) {
            var $progressBar = $(selector).find('.swift-offload-progress-bar');
            $progressBar.css('width', percentage + '%');

            var $percentage = $(selector).find('.swift-offload-percentage');
            $percentage.text(Math.round(percentage) + '%');
        }
    };

    // Initialize when DOM is ready
    $(document).ready(function() {
        SwiftOffloadAdmin.init();
    });

})(jQuery);
