<?php

namespace SwiftOffload\Libs;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Image Optimizer Module
 * Browser-based image optimization with WebP conversion
 *
 * @package SwiftOffload
 */
class Image_Optimizer {

	private static $instance = null;
	private $option_key      = 'swift_offload_image_optimizer_settings';
	private $stats_key       = 'swift_offload_image_optimizer_stats';

	/**
	 * Default settings
	 */
	private $defaults = array(
		'enabled'                      => 'yes',
		'convert_webp'                 => 'yes',
		'webp_quality'                 => 80,
		'max_image_dimension'          => 2560,
		'resize_enabled'               => 'yes',
		'disable_wp_scaling'           => 'no',
		'disable_thumbnail_generation' => 'no',
		'disabled_thumbnail_sizes'     => array(),
		'optimize_videos'              => 'yes',
		'video_quality'                => 'medium',
		'video_max_resolution'         => 1080,
		'optimize_audio'               => 'yes',
		'audio_bitrate'                => 128,
		'generate_lqip'                => 'yes',
		'lqip_quality'                 => 20,
		'lqip_blur'                    => 10,
	);

	public function __construct() {
		// Enqueue image optimizer admin styles and plupload interceptor
		add_action( 'admin_enqueue_scripts', array( $this, 'inject_early_plupload_hook' ), 1 );

		// Enqueue main script
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ), 1 );

		// AJAX handlers
		add_action( 'wp_ajax_swift_offload_save_image_optimizer_settings', array( $this, 'ajax_save_settings' ) );
		add_action( 'wp_ajax_swift_offload_update_optimizer_stats', array( $this, 'ajax_update_stats' ) );
		add_action( 'wp_ajax_swift_offload_reset_optimizer_stats', array( $this, 'ajax_reset_stats' ) );
		add_action( 'wp_ajax_swift_offload_save_attachment_optimization', array( $this, 'ajax_save_attachment_optimization' ) );
		add_action( 'wp_ajax_swift_offload_bulk_optimize_attachment', array( $this, 'ajax_bulk_optimize_attachment' ) );
		add_action( 'wp_ajax_swift_offload_convert_attachment', array( $this, 'ajax_convert_attachment' ) );
		add_action( 'wp_ajax_swift_offload_optimize_single_attachment', array( $this, 'ajax_optimize_single_attachment' ) );
		add_action( 'wp_ajax_swift_offload_get_unoptimized_attachments', array( $this, 'ajax_get_unoptimized_attachments' ) );

		// WordPress filters
		add_filter( 'intermediate_image_sizes', array( $this, 'filter_thumbnail_sizes' ) );
		add_filter( 'big_image_size_threshold', array( $this, 'maybe_disable_wp_scaling' ) );

		// Media Library integration
		add_filter( 'attachment_fields_to_edit', array( $this, 'add_optimization_fields' ), 10, 2 );
		add_filter( 'wp_prepare_attachment_for_js', array( $this, 'add_optimization_to_attachment_js' ), 10, 3 );

		// Media Library statistics bar
		add_action( 'admin_notices', array( $this, 'render_media_library_stats_bar' ) );

		// Enqueue list view pointer assets on media library page
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_media_library_pointer' ) );

		// Media Library list view column
		add_filter( 'manage_media_columns', array( $this, 'add_media_column' ) );
		add_action( 'manage_media_custom_column', array( $this, 'render_media_column' ), 10, 2 );

		// Row actions for Restore/Delete Original
		add_filter( 'media_row_actions', array( $this, 'add_media_row_actions' ), 10, 2 );

		// AJAX handlers for restore/delete original
		add_action( 'wp_ajax_swift_offload_restore_original', array( $this, 'ajax_restore_original' ) );
		add_action( 'wp_ajax_swift_offload_delete_original', array( $this, 'ajax_delete_original' ) );

		// AJAX handler for dismissing list view notice
		add_action( 'wp_ajax_swift_offload_dismiss_list_view_notice', array( $this, 'ajax_dismiss_list_view_notice' ) );

		// Attachment edit page optimization card
		add_action( 'attachment_submitbox_misc_actions', array( $this, 'render_attachment_edit_optimization_card' ), 99 );

		// Fix mixed content - force HTTPS for attachment URLs
		add_filter( 'wp_get_attachment_url', array( $this, 'force_https_attachment_url' ) );
		add_filter( 'wp_get_attachment_image_src', array( $this, 'force_https_attachment_src' ) );
		add_filter( 'wp_calculate_image_srcset', array( $this, 'force_https_srcset' ) );

		// Add LQIP data attribute to images on frontend
		add_filter( 'wp_get_attachment_image_attributes', array( $this, 'add_lqip_to_image' ), 10, 3 );
	}

	/**
	 * Force HTTPS for attachment URLs to prevent mixed content
	 */
	public function force_https_attachment_url( $url ) {
		if ( is_ssl() && strpos( $url, 'http://' ) === 0 ) {
			$url = str_replace( 'http://', 'https://', $url );
		}
		return $url;
	}

	/**
	 * Force HTTPS for attachment image src
	 */
	public function force_https_attachment_src( $image ) {
		if ( is_ssl() && is_array( $image ) && ! empty( $image[0] ) ) {
			$image[0] = str_replace( 'http://', 'https://', $image[0] );
		}
		return $image;
	}

	/**
	 * Force HTTPS for srcset URLs
	 */
	public function force_https_srcset( $sources ) {
		if ( is_ssl() && is_array( $sources ) ) {
			foreach ( $sources as &$source ) {
				if ( ! empty( $source['url'] ) ) {
					$source['url'] = str_replace( 'http://', 'https://', $source['url'] );
				}
			}
		}
		return $sources;
	}

	/**
	 * Get a specific setting
	 */
	public function get_setting( $key ) {
		$settings = $this->get_settings();
		return isset( $settings[ $key ] ) ? $settings[ $key ] : ( isset( $this->defaults[ $key ] ) ? $this->defaults[ $key ] : '' );
	}

	/**
	 * Get all settings
	 */
	public function get_settings() {
		return wp_parse_args( get_option( $this->option_key, array() ), $this->defaults );
	}

	/**
	 * Get optimization stats
	 */
	public function get_stats() {
		$defaults = array(
			'total_files'    => 0,
			'total_saved'    => 0,
			'original_size'  => 0,
			'optimized_size' => 0,
		);
		return wp_parse_args( get_option( $this->stats_key, array() ), $defaults );
	}

	/**
	 * Enqueue image optimizer admin styles and plupload interceptor script.
	 * Hooked to admin_enqueue_scripts at priority 1 to load before other plugins.
	 */
	public function inject_early_plupload_hook() {
		$settings = $this->get_settings();

		// Only inject if optimizer is enabled
		if ( $settings['enabled'] !== 'yes' ) {
			return;
		}

		wp_enqueue_style(
			'swift-offload-image-optimizer-admin',
			SWIFT_OFFLOAD_PLUGIN_URL . 'assets/css/image-optimizer-admin.css',
			array(),
			SWIFT_OFFLOAD_VERSION
		);

		wp_enqueue_script(
			'swift-offload-plupload-interceptor',
			SWIFT_OFFLOAD_PLUGIN_URL . 'assets/js/image-optimizer/plupload-interceptor.js',
			array(),
			SWIFT_OFFLOAD_VERSION,
			false // Load in header — needed to run before other plugins.
		);
	}

	/**
	 * Enqueue list view pointer assets on the Media Library page.
	 * Hooked to admin_enqueue_scripts.
	 */
	public function enqueue_media_library_pointer() {
		$screen = get_current_screen();
		if ( ! $screen || $screen->id !== 'upload' ) {
			return;
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reading view mode parameter, no security impact.
		$mode             = isset( $_GET['mode'] ) ? sanitize_text_field( wp_unslash( $_GET['mode'] ) ) : get_user_option( 'media_library_mode', get_current_user_id() );
		$mode             = $mode ? $mode : 'grid';
		$notice_dismissed = get_user_meta( get_current_user_id(), '_swift_offload_list_view_notice_dismissed', true );

		if ( $mode === 'grid' && ! $notice_dismissed ) {
			wp_enqueue_style( 'wp-pointer' );
			wp_enqueue_script( 'wp-pointer' );

			wp_enqueue_script(
				'swift-offload-list-view-pointer',
				SWIFT_OFFLOAD_PLUGIN_URL . 'assets/js/image-optimizer/list-view-pointer.js',
				array( 'jquery', 'wp-pointer' ),
				SWIFT_OFFLOAD_VERSION,
				true
			);

			wp_localize_script(
				'swift-offload-list-view-pointer',
				'swiftOffloadPointer',
				array(
					'switchText'    => __( 'Switch to list view', 'swift-offload' ),
					'bodyText'      => __( 'Get the most out of your optimizing options. Use the List view to quickly optimize your uploaded images with Image Optimizer.', 'swift-offload' ),
					'switchBtnText' => __( 'Switch to List View', 'swift-offload' ),
					'dismissText'   => __( 'Dismiss', 'swift-offload' ),
					'listViewUrl'   => add_query_arg( 'mode', 'list', admin_url( 'upload.php' ) ),
					'nonce'         => wp_create_nonce( 'swift_offload_dismiss_notice' ),
					'ajaxUrl'       => admin_url( 'admin-ajax.php' ),
				)
			);
		}
	}

	/**
	 * Enqueue admin assets
	 */
	public function enqueue_admin_assets( $hook ) {
		// Only load on media-related pages to avoid dashboard hijacking.
		$allowed = array(
			'upload.php',           // Media Library
			'post.php',             // Edit post (for media modal)
			'post-new.php',         // New post (for media modal)
			'media-new.php',        // Add new media
			'media-upload.php',     // Media upload
		);

		// Also allow on Swift Offload pages.
		$is_swift_offload_page = strpos( $hook, 'swift-offload' ) !== false || strpos( $hook, 'swift_offload' ) !== false;

		if ( ! in_array( $hook, $allowed, true ) && ! $is_swift_offload_page ) {
			return;
		}

		$settings = $this->get_settings();

		// Load before media scripts but after jQuery
		wp_enqueue_script(
			'swift-offload-image-optimizer',
			SWIFT_OFFLOAD_PLUGIN_URL . 'assets/js/image-optimizer/image-optimizer.min.js',
			array( 'jquery', 'plupload' ),
			SWIFT_OFFLOAD_VERSION,
			false // Load in header to run before other plugins
		);

		wp_localize_script(
			'swift-offload-image-optimizer',
			'swiftOffloadImageOptimizer',
			array(
				'ajaxUrl'   => admin_url( 'admin-ajax.php' ),
				'nonce'     => wp_create_nonce( 'swift_offload_image_optimizer' ),
				'settings'  => $settings,
				'stats'     => $this->get_stats(),
				'pluginUrl' => SWIFT_OFFLOAD_PLUGIN_URL,
				'workerUrl' => SWIFT_OFFLOAD_PLUGIN_URL . 'assets/js/image-optimizer/compression-worker.js',
				'i18n'      => array(
					'optimizing'      => __( 'Optimizing...', 'swift-offload' ),
					'optimized'       => __( 'Optimized!', 'swift-offload' ),
					'saved'           => __( 'Saved', 'swift-offload' ),
					'error'           => __( 'Optimization failed', 'swift-offload' ),
					'batchProcessing' => __( 'Processing images with WebAssembly...', 'swift-offload' ),
					'batchComplete'   => __( 'Batch optimization complete!', 'swift-offload' ),
				),
			)
		);
	}

	/**
	 * Filter thumbnail sizes
	 */
	public function filter_thumbnail_sizes( $sizes ) {
		// Disable all thumbnails
		if ( $this->get_setting( 'disable_thumbnail_generation' ) === 'yes' ) {
			return array();
		}

		// Disable specific sizes
		$disabled = $this->get_setting( 'disabled_thumbnail_sizes' );
		if ( ! empty( $disabled ) && is_array( $disabled ) ) {
			$sizes = array_diff( $sizes, $disabled );
		}

		return $sizes;
	}

	/**
	 * Maybe disable WordPress scaling
	 */
	public function maybe_disable_wp_scaling( $threshold ) {
		if ( $this->get_setting( 'disable_wp_scaling' ) === 'yes' ) {
			return false;
		}
		return $threshold;
	}

	/**
	 * Sanitize settings
	 */
	public function sanitize_settings( $options ) {
		$sanitized = array();

		$sanitized['enabled']                      = isset( $options['enabled'] ) && $options['enabled'] === 'yes' ? 'yes' : 'no';
		$sanitized['convert_webp']                 = isset( $options['convert_webp'] ) && $options['convert_webp'] === 'yes' ? 'yes' : 'no';
		$sanitized['webp_quality']                 = isset( $options['webp_quality'] ) ? max( 1, min( 100, absint( $options['webp_quality'] ) ) ) : 80;
		$sanitized['max_image_dimension']          = isset( $options['max_image_dimension'] ) ? absint( $options['max_image_dimension'] ) : 2560;
		$sanitized['resize_enabled']               = isset( $options['resize_enabled'] ) && $options['resize_enabled'] === 'yes' ? 'yes' : 'no';
		$sanitized['disable_wp_scaling']           = isset( $options['disable_wp_scaling'] ) && $options['disable_wp_scaling'] === 'yes' ? 'yes' : 'no';
		$sanitized['disable_thumbnail_generation'] = isset( $options['disable_thumbnail_generation'] ) && $options['disable_thumbnail_generation'] === 'yes' ? 'yes' : 'no';
		$sanitized['disabled_thumbnail_sizes']     = isset( $options['disabled_thumbnail_sizes'] ) && is_array( $options['disabled_thumbnail_sizes'] )
			? array_map( 'sanitize_text_field', $options['disabled_thumbnail_sizes'] )
			: array();

		return $sanitized;
	}

	/**
	 * AJAX: Save settings
	 */
	public function ajax_save_settings() {
		check_ajax_referer( 'swift_offload_image_optimizer_nonce_action', 'security' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied', 'swift-offload' ) ) );
		}

		$settings = array(
			'enabled'                      => isset( $_POST['enabled'] ) ? sanitize_text_field( wp_unslash( $_POST['enabled'] ) ) : 'no',
			'convert_webp'                 => isset( $_POST['convert_webp'] ) ? sanitize_text_field( wp_unslash( $_POST['convert_webp'] ) ) : 'no',
			'webp_quality'                 => isset( $_POST['webp_quality'] ) ? absint( $_POST['webp_quality'] ) : 80,
			'max_image_dimension'          => isset( $_POST['max_image_dimension'] ) ? absint( $_POST['max_image_dimension'] ) : 2560,
			'resize_enabled'               => isset( $_POST['resize_enabled'] ) ? sanitize_text_field( wp_unslash( $_POST['resize_enabled'] ) ) : 'no',
			'disable_wp_scaling'           => isset( $_POST['disable_wp_scaling'] ) ? sanitize_text_field( wp_unslash( $_POST['disable_wp_scaling'] ) ) : 'no',
			'disable_thumbnail_generation' => isset( $_POST['disable_thumbnail_generation'] ) ? sanitize_text_field( wp_unslash( $_POST['disable_thumbnail_generation'] ) ) : 'no',
			'disabled_thumbnail_sizes'     => isset( $_POST['disabled_thumbnail_sizes'] ) ? array_map( 'sanitize_text_field', wp_unslash( (array) $_POST['disabled_thumbnail_sizes'] ) ) : array(),
		);

		$settings = $this->sanitize_settings( $settings );
		update_option( $this->option_key, $settings );

		wp_send_json_success(
			array(
				'message'  => __( 'Settings saved successfully', 'swift-offload' ),
				'settings' => $settings,
			)
		);
	}

	/**
	 * AJAX: Update stats
	 */
	public function ajax_update_stats() {
		check_ajax_referer( 'swift_offload_image_optimizer', 'nonce' );

		if ( ! current_user_can( 'upload_files' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied', 'swift-offload' ) ) );
		}

		$original_size  = isset( $_POST['original_size'] ) ? absint( $_POST['original_size'] ) : 0;
		$optimized_size = isset( $_POST['optimized_size'] ) ? absint( $_POST['optimized_size'] ) : 0;

		if ( $original_size > 0 && $optimized_size > 0 && $optimized_size < $original_size ) {
			$stats = $this->get_stats();
			++$stats['total_files'];
			$stats['original_size']  += $original_size;
			$stats['optimized_size'] += $optimized_size;
			$stats['total_saved']     = $stats['original_size'] - $stats['optimized_size'];

			update_option( $this->stats_key, $stats );
			wp_send_json_success( array( 'stats' => $stats ) );
		}

		wp_send_json_error( array( 'message' => __( 'Invalid data', 'swift-offload' ) ) );
	}

	/**
	 * AJAX: Reset stats
	 */
	public function ajax_reset_stats() {
		check_ajax_referer( 'swift_offload_image_optimizer', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied', 'swift-offload' ) ) );
		}

		$stats = array(
			'total_files'    => 0,
			'total_saved'    => 0,
			'original_size'  => 0,
			'optimized_size' => 0,
		);

		update_option( $this->stats_key, $stats );

		wp_send_json_success(
			array(
				'message' => __( 'Stats reset successfully', 'swift-offload' ),
				'stats'   => $stats,
			)
		);
	}

	/**
	 * AJAX: Get unoptimized attachments for bulk optimize
	 */
	public function ajax_get_unoptimized_attachments() {
		check_ajax_referer( 'swift_offload_image_optimizer', 'nonce' );

		if ( ! current_user_can( 'upload_files' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied', 'swift-offload' ) ) );
		}

		// Get all image attachments that don't have optimization data
		$args = array(
			'post_type'      => 'attachment',
			'post_mime_type' => array( 'image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp' ),
			'post_status'    => 'inherit',
			'posts_per_page' => 100, // Limit to 100 for performance
			'meta_query'     => array(
				'relation' => 'OR',
				array(
					'key'     => '_swift_offload_optimization_data',
					'compare' => 'NOT EXISTS',
				),
				array(
					'key'     => '_swift_offload_optimization_data',
					'value'   => '',
					'compare' => '=',
				),
			),
		);

		$query       = new \WP_Query( $args );
		$attachments = array();

		if ( $query->have_posts() ) {
			foreach ( $query->posts as $post ) {
				$file          = get_attached_file( $post->ID );
				$attachments[] = array(
					'id'       => $post->ID,
					'filename' => basename( $file ),
					'size'     => file_exists( $file ) ? filesize( $file ) : 0,
				);
			}
		}

		// Get settings for display
		$settings = $this->get_settings();
		$driver   = 'GD';
		if ( class_exists( 'Imagick' ) && extension_loaded( 'imagick' ) ) {
			$version_string = \Imagick::getVersion()['versionString'];
			// Extract just the version number (e.g., "7.1.1-15" from full string)
			if ( preg_match( '/(\d+\.\d+\.\d+-\d+)/', $version_string, $matches ) ) {
				$driver = 'Imagick ' . $matches[1];
			} else {
				$driver = 'Imagick';
			}
		}

		wp_send_json_success(
			array(
				'attachments' => $attachments,
				'total'       => count( $attachments ),
				'settings'    => array(
					'source_format' => 'JPEG/PNG',
					'output_format' => ( $settings['convert_webp'] ?? 'yes' ) === 'yes' ? 'WEBP' : 'Original',
					'driver'        => $driver,
					'quality'       => $settings['webp_quality'] ?? 80,
				),
			)
		);
	}

	/**
	 * AJAX: Save attachment optimization data
	 */
	public function ajax_save_attachment_optimization() {
		check_ajax_referer( 'swift_offload_image_optimizer', 'nonce' );

		$attachment_id  = isset( $_POST['attachment_id'] ) ? absint( $_POST['attachment_id'] ) : 0;

		if ( ! current_user_can( 'edit_post', $attachment_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied', 'swift-offload' ) ) );
		}
		$original_size  = isset( $_POST['original_size'] ) ? absint( $_POST['original_size'] ) : 0;
		$optimized_size = isset( $_POST['optimized_size'] ) ? absint( $_POST['optimized_size'] ) : 0;

		$processing_time = isset( $_POST['processing_time'] ) ? absint( $_POST['processing_time'] ) : 0;
		$original_format = isset( $_POST['original_format'] ) ? sanitize_text_field( wp_unslash( $_POST['original_format'] ) ) : '';
		$output_format   = isset( $_POST['output_format'] ) ? sanitize_text_field( wp_unslash( $_POST['output_format'] ) ) : 'webp';
		$lqip_data       = isset( $_POST['lqip_data'] ) ? sanitize_text_field( wp_unslash( $_POST['lqip_data'] ) ) : '';
		$driver          = isset( $_POST['driver'] ) ? sanitize_text_field( wp_unslash( $_POST['driver'] ) ) : 'Browser';
		$quality         = isset( $_POST['quality'] ) ? absint( $_POST['quality'] ) : 80;

		if ( $attachment_id > 0 && $original_size > 0 && $optimized_size > 0 ) {
			$optimization_data = array(
				'original_size'   => $original_size,
				'optimized_size'  => $optimized_size,
				'saved'           => $original_size - $optimized_size,
				'percentage'      => round( ( 1 - $optimized_size / $original_size ) * 100, 2 ),
				'processing_time' => $processing_time,
				'original_format' => $original_format,
				'output_format'   => $output_format,
				'driver'          => $driver,
				'quality'         => $quality,
				'optimized_at'    => current_time( 'mysql' ),
			);

			// Save LQIP if generated
			if ( ! empty( $lqip_data ) ) {
				$optimization_data['lqip'] = $lqip_data;
			}

			update_post_meta( $attachment_id, '_swift_offload_optimization_data', $optimization_data );

			wp_send_json_success( array( 'message' => __( 'Optimization data saved', 'swift-offload' ) ) );
		}

		wp_send_json_error( array( 'message' => __( 'Invalid data', 'swift-offload' ) ) );
	}

	/**
	 * AJAX: Bulk optimize single attachment (server-side)
	 */
	public function ajax_bulk_optimize_attachment() {
		check_ajax_referer( 'swift_offload_image_optimizer', 'security' );

		if ( ! current_user_can( 'upload_files' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied', 'swift-offload' ) ) );
		}

		$attachment_id = isset( $_POST['attachment_id'] ) ? absint( $_POST['attachment_id'] ) : 0;

		if ( ! $attachment_id || ! wp_attachment_is_image( $attachment_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid attachment', 'swift-offload' ) ) );
		}

		// Check if already optimized
		$existing_data = get_post_meta( $attachment_id, '_swift_offload_optimization_data', true );
		if ( ! empty( $existing_data ) && isset( $existing_data['percentage'] ) ) {
			wp_send_json_success(
				array(
					'already_optimized' => true,
					'saved'             => $existing_data['saved'],
					'original_size'     => $existing_data['original_size'],
					'optimized_size'    => $existing_data['optimized_size'],
				)
			);
		}

		// Get the file path
		$file_path = get_attached_file( $attachment_id );
		if ( ! $file_path || ! file_exists( $file_path ) ) {
			wp_send_json_error( array( 'message' => __( 'File not found', 'swift-offload' ) ) );
		}

		$original_size = filesize( $file_path );
		$settings      = $this->get_settings();

		// Determine target format based on conversion settings
		$extension     = strtolower( pathinfo( $file_path, PATHINFO_EXTENSION ) );
		$target_format = 'webp'; // default

		if ( $settings['conversion_mode'] === 'general' ) {
			$target_format = $settings['general_target_format'] ?? 'webp';
		} elseif ( ! empty( $settings['conversion_rules'] ) ) {
			// Find matching rule for this format
			foreach ( $settings['conversion_rules'] as $rule ) {
				$source = strtolower( $rule['source'] );
				if ( $source === $extension || ( $source === 'jpeg' && $extension === 'jpg' ) || ( $source === 'jpg' && $extension === 'jpeg' ) ) {
					$target_format = $rule['target'];
					break;
				}
			}
		}

		// Get quality setting
		$quality = intval( $settings['webp_quality'] ?? 80 );

		// Perform server-side optimization using WP Image Editor
		$editor = wp_get_image_editor( $file_path );
		if ( is_wp_error( $editor ) ) {
			wp_send_json_error( array( 'message' => $editor->get_error_message() ) );
		}

		// Resize if needed
		if ( $settings['resize_enabled'] === 'yes' ) {
			$max_dim = intval( $settings['max_image_dimension'] ?? 2560 );
			$size    = $editor->get_size();
			if ( $size['width'] > $max_dim || $size['height'] > $max_dim ) {
				$editor->resize( $max_dim, $max_dim, false );
			}
		}

		// Set quality
		$editor->set_quality( $quality );

		// Determine output mime type
		$mime_types  = array(
			'webp' => 'image/webp',
			'jpeg' => 'image/jpeg',
			'jpg'  => 'image/jpeg',
			'png'  => 'image/png',
		);
		$output_mime = isset( $mime_types[ $target_format ] ) ? $mime_types[ $target_format ] : 'image/webp';

		// Save to temp file
		$temp_file = wp_tempnam( $file_path );
		$saved     = $editor->save( $temp_file, $output_mime );

		if ( is_wp_error( $saved ) ) {
			wp_delete_file( $temp_file );
			wp_send_json_error( array( 'message' => $saved->get_error_message() ) );
		}

		$optimized_size = filesize( $saved['path'] );

		// Get original format before any changes
		$original_format = strtolower( pathinfo( $file_path, PATHINFO_EXTENSION ) );
		if ( $original_format === 'jpg' ) {
			$original_format = 'jpeg';
		}

		// Detect image editor driver
		$driver = 'GD';
		if ( class_exists( 'Imagick' ) && extension_loaded( 'imagick' ) ) {
			$driver = 'IMAGICK';
		}

		// Measure processing time
		$start_time = microtime( true );

		// Only replace if smaller
		if ( $optimized_size < $original_size ) {
			// Create backup directory for originals
			$upload_dir = wp_upload_dir();
			$backup_dir = $upload_dir['basedir'] . '/swift-offload-originals';
			if ( ! file_exists( $backup_dir ) ) {
				wp_mkdir_p( $backup_dir );
			}

			// Create new filename with new extension if needed
			$new_extension = $target_format === 'jpeg' ? 'jpg' : $target_format;
			$new_path      = preg_replace( '/\.[^.]+$/', '.' . $new_extension, $file_path );

			// Backup original file
			$backup_path = $backup_dir . '/' . $attachment_id . '-' . basename( $file_path );
			copy( $file_path, $backup_path );

			// Move optimized file to replace original
			if ( $file_path !== $new_path ) {
				// phpcs:ignore WordPress.WP.AlternativeFunctions.rename_rename -- Atomic file operation required
				rename( $saved['path'], $new_path );
				wp_delete_file( $file_path ); // Remove old file

				// Update attachment metadata
				$filetype = wp_check_filetype( $new_path );
				wp_update_post(
					array(
						'ID'             => $attachment_id,
						'post_mime_type' => $filetype['type'],
					)
				);
				update_attached_file( $attachment_id, $new_path );
			} else {
				// phpcs:ignore WordPress.WP.AlternativeFunctions.rename_rename -- Atomic file operation required
				rename( $saved['path'], $file_path );
			}

			// Regenerate thumbnails
			$metadata = wp_generate_attachment_metadata( $attachment_id, isset( $new_path ) ? $new_path : $file_path );
			wp_update_attachment_metadata( $attachment_id, $metadata );

			// Calculate processing time
			$processing_time = round( ( microtime( true ) - $start_time ) * 1000 ); // in ms

			// Save optimization data
			$saved_amount = $original_size - $optimized_size;
			$percentage   = round( ( 1 - $optimized_size / $original_size ) * 100, 2 );

			$optimization_data = array(
				'original_size'   => $original_size,
				'optimized_size'  => $optimized_size,
				'saved'           => $saved_amount,
				'percentage'      => $percentage,
				'original_format' => $original_format,
				'output_format'   => $target_format,
				'original_file'   => $backup_path,
				'driver'          => $driver,
				'quality'         => $quality,
				'processing_time' => $processing_time,
				'optimized_at'    => current_time( 'mysql' ),
				'bulk_optimized'  => true,
			);

			update_post_meta( $attachment_id, '_swift_offload_optimization_data', $optimization_data );

			// Update global stats
			$stats = $this->get_stats();
			++$stats['total_files'];
			$stats['total_saved']    += $saved_amount;
			$stats['original_size']  += $original_size;
			$stats['optimized_size'] += $optimized_size;
			update_option( $this->stats_key, $stats );

			wp_send_json_success(
				array(
					'saved'          => $saved_amount,
					'original_size'  => $original_size,
					'optimized_size' => $optimized_size,
					'percentage'     => $percentage,
				)
			);
		} else {
			// Clean up temp file - don't save metadata, just inform user
			wp_delete_file( $saved['path'] );

			wp_send_json_success(
				array(
					'saved'          => 0,
					'original_size'  => $original_size,
					'optimized_size' => $original_size,
					'no_reduction'   => true,
				)
			);
		}
	}

	/**
	 * Get path to bundled binary for image conversion
	 *
	 * @param string $binary_name Name of the binary (avifenc)
	 * @return string|false Path to binary or false if not found
	 */
	private function get_bundled_binary( $binary_name ) {
		$bin_dir = __DIR__ . '/bin';

		// Determine platform-specific binary name
		$os   = PHP_OS_FAMILY;
		$arch = php_uname( 'm' );

		// Map architecture names
		if ( in_array( $arch, array( 'x86_64', 'amd64' ) ) ) {
			$arch = 'x64';
		} elseif ( in_array( $arch, array( 'aarch64', 'arm64' ) ) ) {
			$arch = 'arm64';
		}

		// Platform-specific paths to try
		$paths_to_try = array();

		if ( $os === 'Darwin' ) {
			// macOS
			$paths_to_try[] = $bin_dir . '/macos-' . $arch . '/' . $binary_name;
			$paths_to_try[] = $bin_dir . '/macos/' . $binary_name;
		} elseif ( $os === 'Linux' ) {
			// Linux
			$paths_to_try[] = $bin_dir . '/linux-' . $arch . '/' . $binary_name;
			$paths_to_try[] = $bin_dir . '/linux/' . $binary_name;
		} elseif ( $os === 'Windows' ) {
			// Windows
			$paths_to_try[] = $bin_dir . '/windows-' . $arch . '/' . $binary_name . '.exe';
			$paths_to_try[] = $bin_dir . '/windows/' . $binary_name . '.exe';
		}

		// Generic fallback
		$paths_to_try[] = $bin_dir . '/' . $binary_name;
		$paths_to_try[] = $bin_dir . '/' . $binary_name . '.exe';

		foreach ( $paths_to_try as $path ) {
			if ( file_exists( $path ) && is_executable( $path ) ) {
				return $path;
			}
		}

		// Also check if binary is available in system PATH.
		$which_cmd = $os === 'Windows' ? 'where' : 'which';
		// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.system_calls_shell_exec -- Required to locate system image conversion binaries (avifenc). Input is escaped with escapeshellarg().
		$result = @shell_exec( $which_cmd . ' ' . escapeshellarg( $binary_name ) . ' 2>/dev/null' );
		if ( $result ) {
			$system_path = trim( $result );
			if ( file_exists( $system_path ) && is_executable( $system_path ) ) {
				return $system_path;
			}
		}

		return false;
	}

	/**
	 * Convert image to AVIF using avifenc CLI tool
	 *
	 * @param string $input_path Path to input image
	 * @param string $output_path Path for output AVIF file
	 * @param int    $quality Quality (0-100)
	 * @return bool Success
	 */
	private function convert_to_avif_cli( $input_path, $output_path, $quality = 80 ) {
		$avifenc_path = $this->get_bundled_binary( 'avifenc' );
		if ( ! $avifenc_path ) {
			return false;
		}

		// avifenc uses --min and --max for quality range (0-63, lower is better)
		// Convert 0-100 quality to 63-0 range
		$avif_quality = max( 0, min( 63, (int) ( ( 100 - $quality ) * 63 / 100 ) ) );

		$cmd = sprintf(
			'%s --min %d --max %d -s 6 %s %s 2>&1',
			escapeshellarg( $avifenc_path ),
			$avif_quality,
			$avif_quality,
			escapeshellarg( $input_path ),
			escapeshellarg( $output_path )
		);

		$output      = array();
		$return_code = 0;
		// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.system_calls_exec -- Required for AVIF image conversion via avifenc CLI. All arguments are escaped with escapeshellarg().
		exec( $cmd, $output, $return_code );

		return $return_code === 0 && file_exists( $output_path );
	}

	/**
	 * AJAX: Convert single attachment to specific format
	 */
	public function ajax_convert_attachment() {
		check_ajax_referer( 'swift_offload_image_optimizer', 'security' );

		$attachment_id = isset( $_POST['attachment_id'] ) ? absint( $_POST['attachment_id'] ) : 0;

		if ( ! current_user_can( 'edit_post', $attachment_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied', 'swift-offload' ) ) );
		}

		$target_format = isset( $_POST['target_format'] ) ? sanitize_text_field( wp_unslash( $_POST['target_format'] ) ) : '';

		// Validate attachment
		if ( ! $attachment_id || ! wp_attachment_is_image( $attachment_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid attachment', 'swift-offload' ) ) );
		}

		// Validate target format
		$allowed_formats = array( 'webp', 'avif', 'jpeg', 'png' );
		if ( ! in_array( $target_format, $allowed_formats ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid target format', 'swift-offload' ) ) );
		}

		// Get the file path
		$file_path = get_attached_file( $attachment_id );
		if ( ! $file_path || ! file_exists( $file_path ) ) {
			wp_send_json_error( array( 'message' => __( 'File not found', 'swift-offload' ) ) );
		}

		$original_size = filesize( $file_path );
		$settings      = $this->get_settings();

		// Get original format
		$original_format = strtolower( pathinfo( $file_path, PATHINFO_EXTENSION ) );
		if ( $original_format === 'jpg' ) {
			$original_format = 'jpeg';
		}

		// Get quality setting
		$quality = intval( $settings['webp_quality'] ?? 80 );

		// Start timing
		$start_time = microtime( true );

		// Create new filename with target extension
		$new_extension = $target_format === 'jpeg' ? 'jpg' : $target_format;
		$new_path      = preg_replace( '/\.[^.]+$/', '.' . $new_extension, $file_path );

		// Use Imagick directly for better format support
		$driver             = 'GD';
		$conversion_success = false;

		// Check if Imagick supports the target format
		$imagick_formats = array();
		if ( class_exists( 'Imagick' ) && extension_loaded( 'imagick' ) ) {
			$imagick_formats = array_map( 'strtolower', \Imagick::queryFormats() );
		}

		// Format mapping for Imagick format names
		$format_map = array(
			'webp' => 'webp',
			'avif' => 'avif',
			'jpeg' => 'jpeg',
			'png'  => 'png',
		);

		$imagick_format_name     = $format_map[ $target_format ] ?? $target_format;
		$imagick_supports_format = in_array( $imagick_format_name, $imagick_formats );

		// Check for CLI tools availability
		$cli_avif_available = $this->get_bundled_binary( 'avifenc' ) !== false;

		// For AVIF - try CLI tool if Imagick doesn't support it
		if ( $target_format === 'avif' && ! $imagick_supports_format && ! $conversion_success ) {
			if ( $cli_avif_available ) {
				$driver = 'CLI-AVIFENC';
				// avifenc works best with PNG input
				$temp_input         = $file_path;
				$needs_temp_cleanup = false;

				// If source is not PNG, convert to PNG first using GD
				if ( ! in_array( strtolower( pathinfo( $file_path, PATHINFO_EXTENSION ) ), array( 'png' ) ) ) {
					$editor = wp_get_image_editor( $file_path );
					if ( ! is_wp_error( $editor ) ) {
						// Resize if needed
						if ( $settings['resize_enabled'] === 'yes' ) {
							$max_dim = intval( $settings['max_image_dimension'] ?? 2560 );
							$size    = $editor->get_size();
							if ( $size['width'] > $max_dim || $size['height'] > $max_dim ) {
								$editor->resize( $max_dim, $max_dim, false );
							}
						}
						$temp_png = sys_get_temp_dir() . '/' . uniqid( 'swift_offload_' ) . '.png';
						$saved    = $editor->save( $temp_png, 'image/png' );
						if ( ! is_wp_error( $saved ) ) {
							$temp_input         = $saved['path'];
							$needs_temp_cleanup = true;
						}
					}
				}

				// Convert using avifenc CLI tool
				if ( $this->convert_to_avif_cli( $temp_input, $new_path, $quality ) ) {
					$conversion_success = true;
				}

				// Clean up temp file
				if ( $needs_temp_cleanup && file_exists( $temp_input ) ) {
					wp_delete_file( $temp_input );
				}
			}

			if ( ! $conversion_success ) {
				wp_send_json_error(
					array(
						'message' => sprintf(
							__( 'AVIF conversion requires either ImageMagick with AVIF support or the avifenc CLI tool. Please install avifenc in the plugin bin directory or try WEBP instead.', 'swift-offload' )
						),
					)
				);
			}
		}

		if ( ! $conversion_success && class_exists( 'Imagick' ) && extension_loaded( 'imagick' ) && $imagick_supports_format ) {
			$driver = 'IMAGICK';
			try {
				$imagick = new \Imagick( $file_path );

				// Resize if needed
				if ( $settings['resize_enabled'] === 'yes' ) {
					$max_dim = intval( $settings['max_image_dimension'] ?? 2560 );
					$width   = $imagick->getImageWidth();
					$height  = $imagick->getImageHeight();
					if ( $width > $max_dim || $height > $max_dim ) {
						$imagick->resizeImage( $max_dim, $max_dim, \Imagick::FILTER_LANCZOS, 1, true );
					}
				}

				// Strip metadata for smaller file size
				$imagick->stripImage();

				// Set format and quality based on target
				switch ( $target_format ) {
					case 'webp':
						$imagick->setImageFormat( 'webp' );
						$imagick->setOption( 'webp:method', '6' );
						$imagick->setImageCompressionQuality( $quality );
						break;
					case 'avif':
						$imagick->setImageFormat( 'avif' );
						$imagick->setImageCompressionQuality( $quality );
						break;
					case 'jpeg':
						$imagick->setImageFormat( 'jpeg' );
						$imagick->setImageCompressionQuality( $quality );
						$imagick->setInterlaceScheme( \Imagick::INTERLACE_PLANE );
						break;
					case 'png':
						$imagick->setImageFormat( 'png' );
						$imagick->setOption( 'png:compression-level', '9' );
						break;
				}

				// Write to new path
				$imagick->writeImage( $new_path );
				$imagick->destroy();

				// Verify file was created with correct format
				if ( file_exists( $new_path ) ) {
					// Verify actual mime type of created file
					$finfo       = new \finfo( FILEINFO_MIME_TYPE );
					$actual_mime = $finfo->file( $new_path );

					$expected_mimes = array(
						'webp' => 'image/webp',
						'avif' => 'image/avif',
						'jpeg' => 'image/jpeg',
						'png'  => 'image/png',
					);
					$expected_mime  = $expected_mimes[ $target_format ] ?? '';

					// Check if output format matches expected (allow JPEG variants)
					$mime_ok = ( $actual_mime === $expected_mime ) ||
								( $target_format === 'jpeg' && in_array( $actual_mime, array( 'image/jpeg', 'image/jpg' ) ) );

					if ( $mime_ok ) {
						$conversion_success = true;
					} else {
						// Format mismatch - delete the file and try GD fallback
						wp_delete_file( $new_path );
						$driver = 'GD';
					}
				}
			} catch ( \Exception $e ) {
				// Imagick failed, will try GD fallback
				$driver = 'GD';
			}
		}

		// Fallback to WordPress Image Editor (GD) for basic formats
		if ( ! $conversion_success ) {
			// GD only supports webp, jpeg, png
			if ( ! in_array( $target_format, array( 'webp', 'jpeg', 'png' ) ) ) {
				wp_send_json_error(
					array(
						/* translators: %s: image format name (e.g., AVIF, WEBP) */
						'message' => sprintf(
							__( 'Your server does not support %1$s format. ImageMagick with %2$s support is required. Try WEBP, JPEG or PNG instead.', 'swift-offload' ),
							strtoupper( $target_format ),
							strtoupper( $target_format )
						),
					)
				);
			}

			$editor = wp_get_image_editor( $file_path );
			if ( is_wp_error( $editor ) ) {
				wp_send_json_error( array( 'message' => $editor->get_error_message() ) );
			}

			// Resize if needed
			if ( $settings['resize_enabled'] === 'yes' ) {
				$max_dim = intval( $settings['max_image_dimension'] ?? 2560 );
				$size    = $editor->get_size();
				if ( $size['width'] > $max_dim || $size['height'] > $max_dim ) {
					$editor->resize( $max_dim, $max_dim, false );
				}
			}

			$editor->set_quality( $quality );

			$mime_types  = array(
				'webp' => 'image/webp',
				'jpeg' => 'image/jpeg',
				'png'  => 'image/png',
			);
			$output_mime = $mime_types[ $target_format ] ?? 'image/webp';

			$saved = $editor->save( $new_path, $output_mime );

			if ( is_wp_error( $saved ) ) {
				wp_send_json_error( array( 'message' => $saved->get_error_message() ) );
			}

			// Verify the file was saved correctly
			$actual_extension   = strtolower( pathinfo( $saved['path'], PATHINFO_EXTENSION ) );
			$expected_extension = $target_format === 'jpeg' ? 'jpg' : $target_format;

			if ( $actual_extension !== $expected_extension ) {
				wp_delete_file( $saved['path'] );
				wp_send_json_error(
					array(
						/* translators: %s: image format name (e.g., WEBP, JPEG) */
						'message' => sprintf(
							__( 'Failed to convert to %s format.', 'swift-offload' ),
							strtoupper( $target_format )
						),
					)
				);
			}

			$new_path           = $saved['path'];
			$conversion_success = true;
		}

		if ( ! $conversion_success || ! file_exists( $new_path ) ) {
			wp_send_json_error( array( 'message' => __( 'Conversion failed', 'swift-offload' ) ) );
		}

		$converted_size = filesize( $new_path );

		// Calculate processing time
		$processing_time = round( ( microtime( true ) - $start_time ) * 1000 ); // in ms

		// Normalize formats for comparison (jpg/jpeg are the same)
		$normalized_original = ( $original_format === 'jpg' ) ? 'jpeg' : $original_format;
		$normalized_target   = ( $target_format === 'jpg' ) ? 'jpeg' : $target_format;
		$is_format_change    = ( $normalized_original !== $normalized_target );

		// If converted size is larger or equal AND it's the same format, skip
		// But allow format conversions even if file is larger (user explicitly requested format change)
		if ( $converted_size >= $original_size && ! $is_format_change ) {
			wp_delete_file( $new_path );
			wp_send_json_success(
				array(
					'saved'          => 0,
					'original_size'  => $original_size,
					'converted_size' => $original_size,
					'percentage'     => 0,
					'no_reduction'   => true,
					'message'        => __( 'No size reduction possible with this format', 'swift-offload' ),
				)
			);
		}

		// Create backup directory for originals
		$upload_dir = wp_upload_dir();
		$backup_dir = $upload_dir['basedir'] . '/swift-offload-originals';
		if ( ! file_exists( $backup_dir ) ) {
			wp_mkdir_p( $backup_dir );
		}

		// Backup original file before removing
		$backup_path = $backup_dir . '/' . $attachment_id . '-' . basename( $file_path );
		copy( $file_path, $backup_path );

		// Remove old file if extension changed
		if ( $file_path !== $new_path ) {
			wp_delete_file( $file_path );
		}

		// Update attachment metadata with new file
		$filetype = wp_check_filetype( $new_path );
		wp_update_post(
			array(
				'ID'             => $attachment_id,
				'post_mime_type' => $filetype['type'],
			)
		);
		update_attached_file( $attachment_id, $new_path );

		// Regenerate thumbnails
		$metadata = wp_generate_attachment_metadata( $attachment_id, $new_path );
		wp_update_attachment_metadata( $attachment_id, $metadata );

		// Save conversion data
		$saved_amount = $original_size - $converted_size;
		$percentage   = $original_size > 0 ? round( ( 1 - $converted_size / $original_size ) * 100, 2 ) : 0;

		$optimization_data = array(
			'original_size'   => $original_size,
			'optimized_size'  => $converted_size,
			'saved'           => $saved_amount,
			'percentage'      => $percentage,
			'original_format' => $original_format,
			'output_format'   => $target_format,
			'original_file'   => $backup_path,
			'driver'          => $driver,
			'quality'         => $quality,
			'processing_time' => $processing_time,
			'converted_at'    => current_time( 'mysql' ),
			'converted'       => true,
		);

		update_post_meta( $attachment_id, '_swift_offload_optimization_data', $optimization_data );

		// Update global stats
		$stats = $this->get_stats();
		++$stats['total_files'];
		$stats['total_saved']    += $saved_amount;
		$stats['original_size']  += $original_size;
		$stats['optimized_size'] += $converted_size;
		update_option( $this->stats_key, $stats );

		wp_send_json_success(
			array(
				'saved'          => $saved_amount,
				'original_size'  => $original_size,
				'converted_size' => $converted_size,
				'percentage'     => $percentage,
				'new_format'     => strtoupper( $target_format ),
				'size_increased' => ( $converted_size > $original_size ),
			)
		);
	}

	/**
	 * AJAX: Optimize single attachment (same as bulk but for single button)
	 */
	public function ajax_optimize_single_attachment() {
		// This uses the same logic as bulk optimize
		$this->ajax_bulk_optimize_attachment();
	}

	/**
	 * Add optimization fields to attachment edit screen
	 */
	public function add_optimization_fields( $form_fields, $post ) {
		if ( ! wp_attachment_is_image( $post->ID ) ) {
			return $form_fields;
		}

		$optimization_data = get_post_meta( $post->ID, '_swift_offload_optimization_data', true );

		// Only show optimization card if image was optimized
		if ( ! empty( $optimization_data ) && isset( $optimization_data['percentage'] ) ) {
			$percentage      = $optimization_data['percentage'];
			$processing_time = isset( $optimization_data['processing_time'] ) ? $optimization_data['processing_time'] : 0;
			$output_format   = isset( $optimization_data['output_format'] ) ? strtoupper( $optimization_data['output_format'] ) : 'WEBP';
			$saved_bytes     = isset( $optimization_data['saved'] ) ? $optimization_data['saved'] : 0;

			// Format processing time
			$time_display = '';
			if ( $processing_time > 0 ) {
				if ( $processing_time >= 1000 ) {
					$time_display = round( $processing_time / 1000, 1 ) . ' sec';
				} else {
					$time_display = $processing_time . ' ms';
				}
			}

			// Card style matching user's preferred format
			$html = '<div class="swift-offload-optimization-card swift-offload-optimized-card" style="display:inline-block; margin: 20px 0 16px 0; padding: 20px; background: #f0fdf4; border: 1px solid #bbf7d0; border-radius: 12px; font-family: -apple-system, BlinkMacSystemFont, sans-serif;">';

			// Header row with checkmark and title - single line
			$html .= '<div style="display: flex; align-items: center; gap: 8px; margin-bottom: 20px; color: #166534; font-weight: 600; font-size: 13px; white-space: nowrap;">';
			$html .= '<svg width="18" height="18" viewBox="0 0 24 24" fill="none" style="flex-shrink: 0;"><circle cx="12" cy="12" r="11" fill="#22c55e"/><path d="M8 12l3 3 5-6" stroke="#fff" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"/></svg>';
			$html .= '<span>Optimized by Swift Offload</span>';
			$html .= '</div>';

			// Large percentage
			$html .= '<div style="font-size: 26px; font-weight: 700; color: #16a34a; margin-bottom: 20px;">Saved ' . number_format( $percentage, 2 ) . '%</div>';

			// Original, Optimized, and Saved on separate lines
			$html .= '<div style="font-size: 14px; color: #374151; margin-bottom: 20px; line-height: 1.8;">';
			$html .= '<div>Original: <strong>' . size_format( $optimization_data['original_size'] ) . '</strong></div>';
			$html .= '<div>Optimized: <strong style="color: #16a34a;">' . size_format( $optimization_data['optimized_size'] ) . '</strong></div>';
			$html .= '<div>Saved: <strong style="color: #16a34a;">-' . size_format( $saved_bytes ) . '</strong></div>';
			$html .= '</div>';

			// Format and time on separate lines
			$html .= '<div style="font-size: 13px; color: #6b7280; line-height: 1.8;">';
			$html .= '<div>Converted to <strong style="color: #111;">' . $output_format . '</strong></div>';
			if ( $time_display ) {
				$html .= '<div>Done in <strong style="color: #111;">' . $time_display . '</strong></div>';
			}
			$html .= '</div>';

			$html .= '</div>';

			$form_fields['swift_offload_optimization'] = array(
				'label' => __( 'Optimization', 'swift-offload' ),
				'input' => 'html',
				'html'  => $html,
			);
		}
		// Don't show anything for non-optimized images - cleaner UI

		return $form_fields;
	}

	/**
	 * Add optimization data to attachment JS object (for media modal)
	 */
	public function add_optimization_to_attachment_js( $response, $attachment, $meta ) {
		if ( ! wp_attachment_is_image( $attachment->ID ) ) {
			return $response;
		}

		$optimization_data = get_post_meta( $attachment->ID, '_swift_offload_optimization_data', true );

		// Only add optimization data if image was actually optimized
		if ( ! empty( $optimization_data ) && isset( $optimization_data['percentage'] ) ) {
			$response['swiftOffloadOptimization'] = array(
				'optimized'          => true,
				'originalSize'       => $optimization_data['original_size'],
				'optimizedSize'      => $optimization_data['optimized_size'],
				'saved'              => $optimization_data['saved'],
				'percentage'         => number_format( $optimization_data['percentage'], 2 ),
				'originalFormatted'  => size_format( $optimization_data['original_size'] ),
				'optimizedFormatted' => size_format( $optimization_data['optimized_size'] ),
				'savedFormatted'     => size_format( $optimization_data['saved'] ),
				'processingTime'     => isset( $optimization_data['processing_time'] ) ? $optimization_data['processing_time'] : 0,
				'outputFormat'       => isset( $optimization_data['output_format'] ) ? $optimization_data['output_format'] : 'webp',
			);
		}
		// Don't add anything for non-optimized images - cleaner UI

		return $response;
	}

	/**
	 * Render statistics bar on Media Library page
	 */
	public function render_media_library_stats_bar() {
		$screen = get_current_screen();
		if ( ! $screen || $screen->id !== 'upload' ) {
			return;
		}

		// Check if in grid view and notice not dismissed - use WordPress pointer
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reading view mode parameter, no security impact.
		$mode             = isset( $_GET['mode'] ) ? sanitize_text_field( wp_unslash( $_GET['mode'] ) ) : get_user_option( 'media_library_mode', get_current_user_id() );
		$mode             = $mode ? $mode : 'grid'; // Default is grid
		$notice_dismissed = get_user_meta( get_current_user_id(), '_swift_offload_list_view_notice_dismissed', true );

		if ( $mode === 'grid' && ! $notice_dismissed ) {
			// Pointer assets are enqueued via enqueue_media_library_pointer() on admin_enqueue_scripts.
			// CSS is in image-optimizer-admin.css, enqueued via inject_early_plupload_hook().
		}

		// Get image counts from database
		global $wpdb;

		// Count total images
		$total_count = (int) $wpdb->get_var(
			"SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = 'attachment' AND post_mime_type LIKE 'image/%' AND post_status = 'inherit'"
		);

		// Count optimized images (those with optimization data in meta)
		$optimized_count = (int) $wpdb->get_var(
			"SELECT COUNT(DISTINCT p.ID) FROM {$wpdb->posts} p
			INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
			WHERE p.post_type = 'attachment'
			AND p.post_mime_type LIKE 'image/%'
			AND p.post_status = 'inherit'
			AND pm.meta_key = '_swift_offload_optimization_data'"
		);

		// Get size stats
		$stats          = $this->get_stats();
		$original_size  = isset( $stats['original_size'] ) ? (int) $stats['original_size'] : 0;
		$optimized_size = isset( $stats['optimized_size'] ) ? (int) $stats['optimized_size'] : 0;
		$total_saved    = isset( $stats['total_saved'] ) ? (int) $stats['total_saved'] : 0;

		// Calculate percentages (cap at 100%)
		$space_saved_pct      = $original_size > 0 ? min( 100, round( ( $total_saved / $original_size ) * 100, 1 ) ) : 0;
		$images_optimized_pct = $total_count > 0 ? min( 100, round( ( $optimized_count / $total_count ) * 100, 1 ) ) : 0;
		$unoptimized_count    = max( 0, $total_count - $optimized_count );

		// SVG ring progress helper
		$ring_size     = 70;
		$stroke_width  = 6;
		$radius        = ( $ring_size - $stroke_width ) / 2;
		$circumference = 2 * 3.14159 * $radius;

		// phpcs:disable WordPress.Security.EscapeOutput.OutputNotEscaped -- Computed numeric values are safe.
		?>
		<div class="swift-offload-media-stats-bar" style="background: #fff; border: 1px solid #e5e7eb; border-radius: 8px; padding: 24px 30px; margin: 20px 20px 20px 0; display: grid; grid-template-columns: 1fr 1fr; gap: 40px; font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; box-shadow: 0 1px 3px rgba(0,0,0,0.05);">

			<!-- Space Saved -->
			<div style="display: flex; align-items: center; gap: 20px;">
				<div style="position: relative; width: <?php echo (int) $ring_size; ?>px; height: <?php echo (int) $ring_size; ?>px;">
					<svg width="<?php echo (int) $ring_size; ?>" height="<?php echo (int) $ring_size; ?>" viewBox="0 0 <?php echo (int) $ring_size; ?> <?php echo (int) $ring_size; ?>">
						<circle cx="<?php echo (int) ( $ring_size / 2 ); ?>" cy="<?php echo (int) ( $ring_size / 2 ); ?>" r="<?php echo (int) $radius; ?>" fill="none" stroke="#e5e7eb" stroke-width="<?php echo (int) $stroke_width; ?>"/>
						<circle cx="<?php echo (int) ( $ring_size / 2 ); ?>" cy="<?php echo (int) ( $ring_size / 2 ); ?>" r="<?php echo (int) $radius; ?>" fill="none" stroke="#9333ea" stroke-width="<?php echo (int) $stroke_width; ?>" stroke-linecap="round" stroke-dasharray="<?php echo esc_attr( $circumference ); ?>" stroke-dashoffset="<?php echo esc_attr( $circumference * ( 1 - $space_saved_pct / 100 ) ); ?>" transform="rotate(-90 <?php echo (int) ( $ring_size / 2 ); ?> <?php echo (int) ( $ring_size / 2 ); ?>)"/>
					</svg>
					<div style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); font-size: 14px; font-weight: 600; color: #9333ea;"><?php echo esc_html( $space_saved_pct ); ?>%</div>
				</div>
				<div>
					<div style="font-size: 13px; color: #6b7280; margin-bottom: 6px;">Initial size: <strong style="color: #111;"><?php echo esc_html( size_format( $original_size ) ); ?></strong></div>
					<div style="font-size: 13px; color: #6b7280;">Current size: <strong style="color: #111;"><?php echo esc_html( size_format( $optimized_size ) ); ?></strong></div>
					<div style="font-size: 12px; color: #9ca3af; margin-top: 10px; display: flex; align-items: center; gap: 4px;">
						Space saved: <strong style="color: #16a34a;"><?php echo esc_html( size_format( $total_saved ) ); ?></strong>
						<span title="Total space saved by optimizing images" style="cursor: help;">
							<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="10"/><line x1="12" y1="16" x2="12" y2="12"/><line x1="12" y1="8" x2="12.01" y2="8"/></svg>
						</span>
					</div>
				</div>
			</div>

			<!-- Images Optimized -->
			<div style="display: flex; align-items: center; gap: 20px;">
				<div style="position: relative; width: <?php echo (int) $ring_size; ?>px; height: <?php echo (int) $ring_size; ?>px;">
					<svg width="<?php echo (int) $ring_size; ?>" height="<?php echo (int) $ring_size; ?>" viewBox="0 0 <?php echo (int) $ring_size; ?> <?php echo (int) $ring_size; ?>">
						<circle cx="<?php echo (int) ( $ring_size / 2 ); ?>" cy="<?php echo (int) ( $ring_size / 2 ); ?>" r="<?php echo (int) $radius; ?>" fill="none" stroke="#e5e7eb" stroke-width="<?php echo (int) $stroke_width; ?>"/>
						<circle cx="<?php echo (int) ( $ring_size / 2 ); ?>" cy="<?php echo (int) ( $ring_size / 2 ); ?>" r="<?php echo (int) $radius; ?>" fill="none" stroke="#9333ea" stroke-width="<?php echo (int) $stroke_width; ?>" stroke-linecap="round" stroke-dasharray="<?php echo esc_attr( $circumference ); ?>" stroke-dashoffset="<?php echo esc_attr( $circumference * ( 1 - $images_optimized_pct / 100 ) ); ?>" transform="rotate(-90 <?php echo (int) ( $ring_size / 2 ); ?> <?php echo (int) ( $ring_size / 2 ); ?>)"/>
					</svg>
					<div style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); font-size: 14px; font-weight: 600; color: #9333ea;"><?php echo esc_html( $images_optimized_pct ); ?>%</div>
				</div>
				<div>
					<div style="font-size: 13px; color: #6b7280; margin-bottom: 6px;">Total: <strong style="color: #111;"><?php echo number_format( $total_count ); ?></strong></div>
					<div style="font-size: 13px; color: #6b7280;">Unoptimized: <strong style="color: #111;"><?php echo number_format( $unoptimized_count ); ?></strong></div>
					<div style="font-size: 12px; color: #9ca3af; margin-top: 10px; display: flex; align-items: center; gap: 4px;">
						Images optimized: <strong style="color: #16a34a;"><?php echo number_format( $optimized_count ); ?></strong>
						<span title="Number of images that have been optimized" style="cursor: help;">
							<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="10"/><line x1="12" y1="16" x2="12" y2="12"/><line x1="12" y1="8" x2="12.01" y2="8"/></svg>
						</span>
					</div>
				</div>
			</div>

		</div>
		<?php
		// phpcs:enable WordPress.Security.EscapeOutput.OutputNotEscaped
	}

	/**
	 * Add Optimize Info column to Media Library list view
	 */
	public function add_media_column( $columns ) {
		$columns['swift_offload_optimize'] = __( 'Optimize Info', 'swift-offload' );
		return $columns;
	}

	/**
	 * Render Optimize Info column content
	 */
	public function render_media_column( $column_name, $post_id ) {
		// Handle Optimize column only
		if ( $column_name !== 'swift_offload_optimize' ) {
			return;
		}

		$post      = get_post( $post_id );
		$mime_type = $post->post_mime_type;

		// Only show for images
		if ( strpos( $mime_type, 'image/' ) !== 0 ) {
			echo '<span style="color: #9ca3af;">-</span>';
			return;
		}

		// Get optimization data
		$optimization_data = get_post_meta( $post_id, '_swift_offload_optimization_data', true );

		// Get file extension/format
		$file      = get_attached_file( $post_id );
		$extension = strtoupper( pathinfo( $file, PATHINFO_EXTENSION ) );
		$ext_lower = strtolower( $extension );

		// Format badge color map
		$format_colors = array(
			'WEBP' => array(
				'bg'   => '#dbeafe',
				'text' => '#1d4ed8',
			),
			'AVIF' => array(
				'bg'   => '#fce7f3',
				'text' => '#be185d',
			),
			'JXL'  => array(
				'bg'   => '#f3e8ff',
				'text' => '#7c3aed',
			),
			'PNG'  => array(
				'bg'   => '#fee2e2',
				'text' => '#dc2626',
			),
			'JPEG' => array(
				'bg'   => '#fef3c7',
				'text' => '#d97706',
			),
			'JPG'  => array(
				'bg'   => '#fef3c7',
				'text' => '#d97706',
			),
			'GIF'  => array(
				'bg'   => '#d1fae5',
				'text' => '#059669',
			),
			'HEIC' => array(
				'bg'   => '#e0e7ff',
				'text' => '#4f46e5',
			),
		);

		$badge_style = isset( $format_colors[ $extension ] )
			? $format_colors[ $extension ]
			: array(
				'bg'   => '#f3f4f6',
				'text' => '#374151',
			);

		// Container
		echo '<div class="swift-offload-media-column" style="display: flex; flex-direction: column; gap: 4px; align-items: flex-start;">';

		// If optimized, show detailed info
		if ( ! empty( $optimization_data ) && isset( $optimization_data['percentage'] ) ) {
			$original_format = isset( $optimization_data['original_format'] ) ? strtoupper( $optimization_data['original_format'] ) : $extension;
			$output_format   = isset( $optimization_data['output_format'] ) ? strtoupper( $optimization_data['output_format'] ) : $extension;
			$processing_time = isset( $optimization_data['processing_time'] ) ? $optimization_data['processing_time'] : 0;
			$original_size   = isset( $optimization_data['original_size'] ) ? $optimization_data['original_size'] : 0;
			$optimized_size  = isset( $optimization_data['optimized_size'] ) ? $optimization_data['optimized_size'] : 0;
			$saved           = isset( $optimization_data['saved'] ) ? $optimization_data['saved'] : 0;
			$percentage      = isset( $optimization_data['percentage'] ) ? $optimization_data['percentage'] : 0;
			$skipped_larger  = isset( $optimization_data['skipped_larger'] ) && $optimization_data['skipped_larger'];
			$driver          = isset( $optimization_data['driver'] ) ? $optimization_data['driver'] : 'GD';
			$quality         = isset( $optimization_data['quality'] ) ? $optimization_data['quality'] : 80;

			// Format time
			$time_display = '';
			if ( $processing_time > 0 ) {
				if ( $processing_time >= 1000 ) {
					$time_display = round( $processing_time / 1000, 2 ) . 's';
				} else {
					$time_display = $processing_time . 'ms';
				}
			}

			// Legacy: Skipped Larger (for old data) - show as not optimized with option to retry
			if ( $skipped_larger ) {
				// Show format badge
				echo '<span class="swift-offload-format-badge" style="display: inline-block; padding: 2px 8px; font-size: 10px; font-weight: 600; border-radius: 4px; background: ' . esc_attr( $badge_style['bg'] ) . '; color: ' . esc_attr( $badge_style['text'] ) . ';">' . esc_html( $extension ) . '</span>';
				echo '<div style="color: #9ca3af; font-size: 10px; margin-top: 2px;">No reduction possible</div>';
			} elseif ( $percentage > 0 || $original_format !== $output_format ) {
				// Format conversion badge (show even if file size increased for format changes)
				$orig_color = isset( $format_colors[ $original_format ] ) ? $format_colors[ $original_format ] : array(
					'bg'   => '#f3f4f6',
					'text' => '#374151',
				);
				$out_color  = isset( $format_colors[ $output_format ] ) ? $format_colors[ $output_format ] : array(
					'bg'   => '#f3f4f6',
					'text' => '#374151',
				);

				echo '<div style="display: flex; align-items: center; gap: 4px; font-size: 10px;">';
				echo '<span style="padding: 1px 5px; border-radius: 3px; font-weight: 600; background: ' . esc_attr( $orig_color['bg'] ) . '; color: ' . esc_attr( $orig_color['text'] ) . ';">' . esc_html( $original_format ) . '</span>';
				echo '<span style="color: #9ca3af;">-&gt;</span>';
				echo '<span style="padding: 1px 5px; border-radius: 3px; font-weight: 600; background: ' . esc_attr( $out_color['bg'] ) . '; color: ' . esc_attr( $out_color['text'] ) . ';">' . esc_html( $output_format ) . '</span>';
				if ( $time_display ) {
					echo '<span style="color: #9ca3af; font-size: 9px;">' . esc_html( $time_display ) . '</span>';
				}
				echo '</div>';

				// Stats line - show size change (green for reduction, orange for increase)
				if ( $percentage > 0 ) {
					echo '<div style="color: #16a34a; font-size: 11px; font-weight: 500;">';
					echo '<strong>' . number_format( $percentage, 0 ) . '%</strong> ';
				} else {
					echo '<div style="color: #d97706; font-size: 11px; font-weight: 500;">';
					echo '<strong>+' . number_format( abs( $percentage ), 0 ) . '%</strong> ';
				}
				echo '<span style="color: #6b7280; font-weight: 400;">(' . esc_html( size_format( $original_size ) ) . ' -&gt; ' . esc_html( size_format( $optimized_size ) ) . ')</span>';
				echo '</div>';

				// Driver/quality
				echo '<div style="color: #9ca3af; font-size: 9px;">';
				echo esc_html( strtoupper( $driver ) ) . '@' . esc_html( $quality );
				echo '</div>';
			} else {
				// Already optimized or no change
				echo '<span class="swift-offload-format-badge" style="display: inline-block; padding: 2px 8px; font-size: 10px; font-weight: 600; border-radius: 4px; background: ' . esc_attr( $badge_style['bg'] ) . '; color: ' . esc_attr( $badge_style['text'] ) . ';">' . esc_html( $extension ) . '</span>';
				echo '<div style="color: #6b7280; font-size: 10px;">Already optimized</div>';
			}
		} else {
			// Not optimized - show format badge and action buttons
			echo '<span class="swift-offload-format-badge" style="display: inline-block; padding: 2px 8px; font-size: 10px; font-weight: 600; border-radius: 4px; background: ' . esc_attr( $badge_style['bg'] ) . '; color: ' . esc_attr( $badge_style['text'] ) . ';">' . esc_html( $extension ) . '</span>';

			// Available target formats (exclude current format)
			$all_formats    = array( 'webp', 'avif', 'jpeg', 'png' );
			$normalized_ext = ( $ext_lower === 'jpg' ) ? 'jpeg' : $ext_lower;
			$target_formats = array_filter(
				$all_formats,
				function ( $f ) use ( $normalized_ext ) {
					return $f !== $normalized_ext;
				}
			);

			echo '<div style="display: flex; flex-direction: column; gap: 4px;">';

			// Convert button with popover
			echo '<div class="swift-offload-convert-dropdown" style="position: relative; display: inline-block;">';
			echo '<button type="button" class="button button-small swift-offload-convert-btn" data-attachment-id="' . esc_attr( $post_id ) . '" data-current-format="' . esc_attr( $ext_lower ) . '" style="font-size: 11px; padding: 2px 8px; height: auto; line-height: 1.6;">';
			echo 'Convert &#9662;';
			echo '</button>';

			// Popover menu
			echo '<div class="swift-offload-convert-popover" style="display: none;">';
			echo '<div class="swift-offload-popover-header">' . esc_html( $extension ) . ' -&gt; ?</div>';
			foreach ( $target_formats as $format ) {
				$format_upper = strtoupper( $format );
				$format_color = isset( $format_colors[ $format_upper ] ) ? $format_colors[ $format_upper ] : array(
					'bg'   => '#f3f4f6',
					'text' => '#374151',
				);
				echo '<a href="#" class="swift-offload-convert-option" data-attachment-id="' . esc_attr( $post_id ) . '" data-target-format="' . esc_attr( $format ) . '">';
				echo '<span class="swift-offload-format-tag" style="background: ' . esc_attr( $format_color['bg'] ) . '; color: ' . esc_attr( $format_color['text'] ) . ';">' . esc_html( $format_upper ) . '</span>';
				echo '</a>';
			}
			echo '</div>';
			echo '</div>';

			// Optimize button
			echo '<button type="button" class="button button-small button-primary swift-offload-optimize-single-btn" data-attachment-id="' . esc_attr( $post_id ) . '" style="font-size: 11px; padding: 2px 8px; height: auto; line-height: 1.6;">';
			echo 'Optimize';
			echo '</button>';

			echo '</div>';
		}

		echo '</div>';
	}

	/**
	 * Render optimization card on attachment edit page (post.php?action=edit)
	 */
	public function render_attachment_edit_optimization_card( $post ) {
		if ( ! wp_attachment_is_image( $post->ID ) ) {
			return;
		}

		$optimization_data = get_post_meta( $post->ID, '_swift_offload_optimization_data', true );

		// Only show if image was optimized
		if ( empty( $optimization_data ) || ! isset( $optimization_data['percentage'] ) ) {
			return;
		}

		$percentage      = $optimization_data['percentage'];
		$processing_time = isset( $optimization_data['processing_time'] ) ? $optimization_data['processing_time'] : 0;
		$output_format   = isset( $optimization_data['output_format'] ) ? strtoupper( $optimization_data['output_format'] ) : 'WEBP';
		$saved_bytes     = isset( $optimization_data['saved'] ) ? $optimization_data['saved'] : 0;

		// Format processing time
		$time_display = '';
		if ( $processing_time > 0 ) {
			if ( $processing_time >= 1000 ) {
				$time_display = round( $processing_time / 1000, 1 ) . ' sec';
			} else {
				$time_display = $processing_time . ' ms';
			}
		}

		echo '<div class="misc-pub-section swift-offload-optimization-card-section">';
		echo '<div class="swift-offload-optimization-card swift-offload-optimized-card" style="margin: 12px 0 0 0; padding: 16px; background: #f0fdf4; border: 1px solid #bbf7d0; border-radius: 8px; font-family: -apple-system, BlinkMacSystemFont, sans-serif;">';

		// Header row with checkmark and title
		echo '<div style="display: flex; align-items: center; gap: 6px; margin-bottom: 12px; color: #166534; font-weight: 600; font-size: 12px;">';
		echo '<svg width="16" height="16" viewBox="0 0 24 24" fill="none"><circle cx="12" cy="12" r="11" fill="#22c55e"/><path d="M8 12l3 3 5-6" stroke="#fff" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"/></svg>';
		echo '<span>Optimized by Swift Offload</span>';
		echo '</div>';

		// Large percentage
		echo '<div style="font-size: 20px; font-weight: 700; color: #16a34a; margin-bottom: 12px;">Saved ' . number_format( $percentage, 2 ) . '%</div>';

		// Stats
		echo '<div style="font-size: 12px; color: #374151; margin-bottom: 12px; line-height: 1.6;">';
		echo '<div>Original: <strong>' . esc_html( size_format( $optimization_data['original_size'] ) ) . '</strong></div>';
		echo '<div>Optimized: <strong style="color: #16a34a;">' . esc_html( size_format( $optimization_data['optimized_size'] ) ) . '</strong></div>';
		echo '<div>Saved: <strong style="color: #16a34a;">-' . esc_html( size_format( $saved_bytes ) ) . '</strong></div>';
		echo '</div>';

		// Format and time
		echo '<div style="font-size: 11px; color: #6b7280; line-height: 1.6;">';
		echo '<div>Converted to <strong style="color: #111;">' . esc_html( $output_format ) . '</strong></div>';
		if ( $time_display ) {
			echo '<div>Done in <strong style="color: #111;">' . esc_html( $time_display ) . '</strong></div>';
		}
		echo '</div>';

		echo '</div>';
		echo '</div>';
	}

	/**
	 * Add row actions for Restore/Delete Original
	 */
	public function add_media_row_actions( $actions, $post ) {
		if ( strpos( $post->post_mime_type, 'image/' ) !== 0 ) {
			return $actions;
		}

		$optimization_data = get_post_meta( $post->ID, '_swift_offload_optimization_data', true );

		// Only add if image has been optimized and has original backup
		if ( ! empty( $optimization_data ) && ! empty( $optimization_data['original_file'] ) ) {
			$original_exists = file_exists( $optimization_data['original_file'] );

			if ( $original_exists ) {
				// Restore Original link
				$restore_url                      = wp_nonce_url(
					admin_url( 'admin-ajax.php?action=swift_offload_restore_original&attachment_id=' . $post->ID ),
					'swift_offload_restore_' . $post->ID
				);
				$actions['swift_offload_restore'] = sprintf(
					'<a href="%s" class="swift-offload-restore-original" data-id="%d">%s</a>',
					esc_url( $restore_url ),
					$post->ID,
					__( 'Restore Original', 'swift-offload' )
				);

				// Delete Original link
				$delete_url                               = wp_nonce_url(
					admin_url( 'admin-ajax.php?action=swift_offload_delete_original&attachment_id=' . $post->ID ),
					'swift_offload_delete_' . $post->ID
				);
				$actions['swift_offload_delete_original'] = sprintf(
					'<a href="%s" class="swift-offload-delete-original submitdelete" data-id="%d" style="color: #b32d2e;">%s</a>',
					esc_url( $delete_url ),
					$post->ID,
					__( 'Delete Original', 'swift-offload' )
				);
			}
		}

		return $actions;
	}

	/**
	 * AJAX: Restore original file
	 */
	public function ajax_restore_original() {
		$attachment_id = isset( $_GET['attachment_id'] ) ? absint( $_GET['attachment_id'] ) : 0;

		if ( ! $attachment_id ) {
			wp_die( esc_html__( 'Invalid attachment', 'swift-offload' ) );
		}

		check_admin_referer( 'swift_offload_restore_' . $attachment_id );

		if ( ! current_user_can( 'edit_post', $attachment_id ) ) {
			wp_die( esc_html__( 'Permission denied', 'swift-offload' ) );
		}

		$optimization_data = get_post_meta( $attachment_id, '_swift_offload_optimization_data', true );

		if ( empty( $optimization_data['original_file'] ) || ! file_exists( $optimization_data['original_file'] ) ) {
			wp_die( esc_html__( 'Original file not found', 'swift-offload' ) );
		}

		$original_file = $optimization_data['original_file'];
		$current_file  = get_attached_file( $attachment_id );

		// Delete current optimized file
		if ( file_exists( $current_file ) ) {
			wp_delete_file( $current_file );
		}

		// Restore original
		// phpcs:ignore WordPress.WP.AlternativeFunctions.rename_rename -- Atomic file operation required for restore
		rename( $original_file, $current_file );

		// Update attachment metadata
		$filetype = wp_check_filetype( $current_file );
		wp_update_post(
			array(
				'ID'             => $attachment_id,
				'post_mime_type' => $filetype['type'],
			)
		);
		update_attached_file( $attachment_id, $current_file );

		// Regenerate thumbnails
		$metadata = wp_generate_attachment_metadata( $attachment_id, $current_file );
		wp_update_attachment_metadata( $attachment_id, $metadata );

		// Remove optimization data
		delete_post_meta( $attachment_id, '_swift_offload_optimization_data' );

		// Redirect back - preserve the referer URL
		$redirect_url = wp_get_referer();
		if ( ! $redirect_url ) {
			$redirect_url = admin_url( 'upload.php?mode=list' );
		}
		wp_safe_redirect( $redirect_url );
		exit;
	}

	/**
	 * AJAX: Delete original backup file
	 */
	public function ajax_delete_original() {
		$attachment_id = isset( $_GET['attachment_id'] ) ? absint( $_GET['attachment_id'] ) : 0;

		if ( ! $attachment_id ) {
			wp_die( esc_html__( 'Invalid attachment', 'swift-offload' ) );
		}

		check_admin_referer( 'swift_offload_delete_' . $attachment_id );

		if ( ! current_user_can( 'edit_post', $attachment_id ) ) {
			wp_die( esc_html__( 'Permission denied', 'swift-offload' ) );
		}

		$optimization_data = get_post_meta( $attachment_id, '_swift_offload_optimization_data', true );

		if ( empty( $optimization_data['original_file'] ) ) {
			wp_die( esc_html__( 'No original file to delete', 'swift-offload' ) );
		}

		// Delete original backup
		if ( file_exists( $optimization_data['original_file'] ) ) {
			wp_delete_file( $optimization_data['original_file'] );
		}

		// Update optimization data to remove original_file reference
		unset( $optimization_data['original_file'] );
		update_post_meta( $attachment_id, '_swift_offload_optimization_data', $optimization_data );

		// Redirect back - preserve the referer URL
		$redirect_url = wp_get_referer();
		if ( ! $redirect_url ) {
			$redirect_url = admin_url( 'upload.php?mode=list' );
		}
		wp_safe_redirect( $redirect_url );
		exit;
	}

	/**
	 * AJAX: Dismiss list view notice
	 */
	public function ajax_dismiss_list_view_notice() {
		check_ajax_referer( 'swift_offload_dismiss_notice', 'nonce' );

		update_user_meta( get_current_user_id(), '_swift_offload_list_view_notice_dismissed', true );

		wp_send_json_success();
	}

	/**
	 * Get LQIP (Low Quality Image Placeholder) for an attachment
	 *
	 * @param int $attachment_id The attachment ID
	 * @return string Base64-encoded LQIP data or empty string
	 */
	public function get_lqip( $attachment_id ) {
		$optimization_data = get_post_meta( $attachment_id, '_swift_offload_optimization_data', true );
		return isset( $optimization_data['lqip'] ) ? $optimization_data['lqip'] : '';
	}

	/**
	 * Add LQIP data attribute to image tags on frontend
	 *
	 * @param array   $attr       Existing attributes
	 * @param WP_Post $attachment The attachment post object
	 * @param string  $size       Image size
	 * @return array Modified attributes
	 */
	public function add_lqip_to_image( $attr, $attachment, $size ) {
		$settings = $this->get_settings();

		// Only add if LQIP is enabled
		if ( ( $settings['generate_lqip'] ?? 'yes' ) !== 'yes' ) {
			return $attr;
		}

		$lqip = $this->get_lqip( $attachment->ID );
		if ( ! empty( $lqip ) ) {
			$attr['data-lqip'] = $lqip;
		}

		return $attr;
	}

	/**
	 * Get instance
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}
}

Image_Optimizer::get_instance();
