/**
 * Clipboard utilities for copying/pasting responsive styles
 */

const CLIPBOARD_KEY = 'rbc_responsive_clipboard';

/**
 * Copy responsive styles to clipboard
 * 
 * @param {Object} attributes Block attributes containing responsive settings
 * @return {boolean} Success status
 */
export function copyResponsiveStyles( attributes ) {
	try {
		const responsiveData = {
			responsivePadding: attributes.responsivePadding || {},
			responsiveMargin: attributes.responsiveMargin || {},
			responsiveFontSize: attributes.responsiveFontSize || {},
			responsiveVisibility: attributes.responsiveVisibility || {},
			timestamp: Date.now(),
		};

		// Store in localStorage (survives page refresh)
		localStorage.setItem( CLIPBOARD_KEY, JSON.stringify( responsiveData ) );
		
		return true;
	} catch ( error ) {
		console.error( 'Failed to copy responsive styles:', error );
		return false;
	}
}

/**
 * Paste responsive styles from clipboard
 * 
 * @return {Object|null} Responsive styles or null if none available
 */
export function pasteResponsiveStyles() {
	try {
		const stored = localStorage.getItem( CLIPBOARD_KEY );
		
		if ( ! stored ) {
			return null;
		}

		const data = JSON.parse( stored );
		
		// Check if clipboard data is less than 1 hour old
		const oneHour = 60 * 60 * 1000;
		if ( Date.now() - data.timestamp > oneHour ) {
			clearClipboard();
			return null;
		}

		// Return only the responsive attributes (exclude timestamp)
		const { timestamp, ...responsiveStyles } = data;
		return responsiveStyles;
	} catch ( error ) {
		console.error( 'Failed to paste responsive styles:', error );
		return null;
	}
}

/**
 * Check if clipboard has responsive styles
 * 
 * @return {boolean} True if clipboard has data
 */
export function hasClipboardData() {
	try {
		const stored = localStorage.getItem( CLIPBOARD_KEY );
		if ( ! stored ) {
			return false;
		}

		const data = JSON.parse( stored );
		const oneHour = 60 * 60 * 1000;
		
		return ( Date.now() - data.timestamp ) < oneHour;
	} catch ( error ) {
		return false;
	}
}

/**
 * Clear clipboard
 */
export function clearClipboard() {
	try {
		localStorage.removeItem( CLIPBOARD_KEY );
	} catch ( error ) {
		console.error( 'Failed to clear clipboard:', error );
	}
}
