/**
 * Quick Action Presets
 * 
 * Pre-configured responsive settings for common use cases
 */

export const quickActionPresets = {
	hideOnMobile: {
		label: '📱 Hide on Mobile',
		description: 'Hide this block on mobile devices',
		apply: ( attributes ) => ( {
			...attributes,
			responsiveVisibility: {
				...attributes.responsiveVisibility,
				hideMobile: true,
			},
		} ),
	},
	
	responsiveHeading: {
		label: '📝 Responsive Heading',
		description: 'Optimal heading sizes for all devices',
		validFor: ( blockName ) => {
			// Only apply to heading blocks
			return blockName === 'core/heading';
		},
		apply: ( attributes ) => {
			// Get current desktop font size
			const currentDesktop = attributes.responsiveFontSize?.desktop;
			
			// Smart responsive scaling based on current size
			let desktopSize, tabletSize, mobileSize;
			
			if ( currentDesktop ) {
				// Parse the current size
				const desktopNum = parseFloat( currentDesktop );
				const unit = currentDesktop.replace( desktopNum.toString(), '' ) || 'px';
				
				// Apply smart scaling ratios
				// Tablet: ~75-80% of desktop
				// Mobile: ~60-65% of desktop
				const tabletNum = Math.round( desktopNum * 0.78 );
				const mobileNum = Math.round( desktopNum * 0.62 );
				
				desktopSize = currentDesktop;
				tabletSize = `${ tabletNum }${ unit }`;
				mobileSize = `${ mobileNum }${ unit }`;
			} else {
				// No current size - use sensible heading defaults
				// These are based on typical H2-H3 sizes
				desktopSize = '2.5rem';
				tabletSize = '2rem';
				mobileSize = '1.5rem';
			}
			
			return {
				...attributes,
				responsiveFontSize: {
					desktop: desktopSize,
					tablet: tabletSize,
					mobile: mobileSize,
				},
			};
		},
	},
	
	responsiveText: {
		label: '📄 Responsive Text',
		description: 'Optimal text sizes for body content',
		validFor: ( blockName ) => {
			// Apply to paragraphs, buttons, and other text blocks (but NOT headings)
			return blockName === 'core/paragraph' || 
			       blockName === 'core/button' || 
			       blockName === 'core/list' ||
			       blockName === 'core/quote' ||
			       blockName === 'core/pullquote';
		},
		apply: ( attributes ) => {
			// Get current desktop font size
			const currentDesktop = attributes.responsiveFontSize?.desktop;
			
			// Smart responsive scaling
			let desktopSize, tabletSize, mobileSize;
			
			if ( currentDesktop ) {
				// Parse the current size
				const desktopNum = parseFloat( currentDesktop );
				const unit = currentDesktop.replace( desktopNum.toString(), '' ) || 'px';
				
				// Apply smart scaling ratios (same as headings for consistency)
				const tabletNum = Math.round( desktopNum * 0.78 * 100 ) / 100; // Round to 2 decimals
				const mobileNum = Math.round( desktopNum * 0.62 * 100 ) / 100;
				
				desktopSize = currentDesktop;
				tabletSize = `${ tabletNum }${ unit }`;
				mobileSize = `${ mobileNum }${ unit }`;
			} else {
				// No current size - use sensible body text defaults
				desktopSize = '1.125rem'; // 18px
				tabletSize = '1rem';      // 16px
				mobileSize = '0.875rem';  // 14px
			}
			
			return {
				...attributes,
				responsiveFontSize: {
					desktop: desktopSize,
					tablet: tabletSize,
					mobile: mobileSize,
				},
			};
		},
	},
	
	mobileStack: {
		label: '📲 Mobile Stack',
		description: 'Full width on mobile, auto on desktop',
		apply: ( attributes ) => ( {
			...attributes,
			responsivePadding: {
				mobile: {
					top: '16px',
					right: '16px',
					bottom: '16px',
					left: '16px',
				},
			},
		} ),
	},
	
	compactMobile: {
		label: '🔄 Compact Mobile',
		description: 'Reduce all spacing by 50% on mobile',
		apply: ( attributes ) => {
			const desktopPadding = attributes.responsivePadding?.desktop || {};
			const desktopMargin = attributes.responsiveMargin?.desktop || {};
			
			return {
				...attributes,
				responsivePadding: {
					...attributes.responsivePadding,
					mobile: {
						top: halveValue( desktopPadding.top ),
						right: halveValue( desktopPadding.right ),
						bottom: halveValue( desktopPadding.bottom ),
						left: halveValue( desktopPadding.left ),
					},
				},
				responsiveMargin: {
					...attributes.responsiveMargin,
					mobile: {
						top: halveValue( desktopMargin.top ),
						right: halveValue( desktopMargin.right ),
						bottom: halveValue( desktopMargin.bottom ),
						left: halveValue( desktopMargin.left ),
					},
				},
			};
		},
	},
	
	fullWidthMobile: {
		label: '📐 Full Width Mobile',
		description: 'Remove side padding/margins on mobile',
		apply: ( attributes ) => ( {
			...attributes,
			responsivePadding: {
				...attributes.responsivePadding,
				mobile: {
					...attributes.responsivePadding?.mobile,
					left: '0px',
					right: '0px',
				},
			},
			responsiveMargin: {
				...attributes.responsiveMargin,
				mobile: {
					...attributes.responsiveMargin?.mobile,
					left: '0px',
					right: '0px',
				},
			},
		} ),
	},
	
	centerOnMobile: {
		label: '⬆️ Center on Mobile',
		description: 'Center content with equal margins',
		apply: ( attributes ) => ( {
			...attributes,
			responsivePadding: {
				...attributes.responsivePadding,
				mobile: {
					top: '20px',
					right: '16px',
					bottom: '20px',
					left: '16px',
				},
			},
		} ),
	},
	
	removeSpacingMobile: {
		label: '🗜️ Remove Spacing Mobile',
		description: 'Zero padding and margins on mobile',
		apply: ( attributes ) => ( {
			...attributes,
			responsivePadding: {
				...attributes.responsivePadding,
				mobile: {
					top: '0px',
					right: '0px',
					bottom: '0px',
					left: '0px',
				},
			},
			responsiveMargin: {
				...attributes.responsiveMargin,
				mobile: {
					top: '0px',
					right: '0px',
					bottom: '0px',
					left: '0px',
				},
			},
		} ),
	},
};

/**
 * Helper: Halve a CSS value
 */
function halveValue( value ) {
	if ( ! value ) return '0px';
	
	const num = parseFloat( value );
	if ( isNaN( num ) ) return value;
	
	const unit = value.replace( num.toString(), '' );
	return `${ num / 2 }${ unit || 'px' }`;
}
