/**
 * Editor Styles Utility
 *
 * Injects CSS into editor iframe for LIVE preview.
 */

/**
 * Ensure value has a unit (add 'px' if missing).
 * Handles WordPress spacing preset tokens.
 */
function ensureUnit( value ) {
	if ( ! value ) {
		return value;
	}
	// Handle WordPress spacing preset tokens (from SpacingSizesControl slider)
	// Format: "var:preset|spacing|30" → "var(--wp--preset--spacing--30)"
	if ( typeof value === 'string' && value.startsWith( 'var:preset|spacing|' ) ) {
		const presetId = value.replace( 'var:preset|spacing|', '' );
		return `var(--wp--preset--spacing--${ presetId })`;
	}
	// If value already ends with a unit, return as-is
	if ( /^[\d.]+(?:px|em|rem|%|vh|vw)$/.test( value ) ) {
		return value;
	}
	// If it's just a number, add 'px'
	if ( /^[\d.]+$/.test( value ) ) {
		return `${ value }px`;
	}
	// Otherwise return as-is (might already have unit)
	return value;
}

/**
 * Generate CSS for a specific device (for editor preview).
 * NO media queries - styles apply directly.
 */
function generateDeviceCSS( attributes, device ) {
	const {
		responsiveId,
		responsivePadding = {},
		responsiveMargin = {},
		responsiveFontSize = {},
		responsiveVisibility = {},
	} = attributes;

	if ( ! responsiveId ) {
		return '';
	}

	const selector = `.rbc-block-${ responsiveId }`;
	const rules = [];

	// Check visibility first - if hidden on this device, hide the whole block
	const hideKey = `hide${ device.charAt( 0 ).toUpperCase() + device.slice( 1 ) }`;
	if ( responsiveVisibility[ hideKey ] ) {
		return `${ selector } { display: none !important; }\n`;
	}

	// Get styles for the selected device with inheritance
	// If a value is not set for this device, fall back to larger devices
	const deviceOrder = [ 'desktop', 'tablet', 'mobile' ];
	
	// Helper to get value with inheritance
	const getInheritedValue = ( values, device, side ) => {
		const deviceIndex = deviceOrder.indexOf( device );
		// Check current device first, then fall back to larger devices
		for ( let i = deviceIndex; i >= 0; i-- ) {
			const checkDevice = deviceOrder[ i ];
			if ( values[ checkDevice ]?.[ side ] ) {
				return values[ checkDevice ][ side ];
			}
		}
		return null;
	};

	// Padding - get inherited values
	[ 'top', 'right', 'bottom', 'left' ].forEach( ( side ) => {
		const value = getInheritedValue( responsivePadding, device, side );
		if ( value ) {
			// Editor preview: use !important to override theme styles in editor
			rules.push( `padding-${ side }: ${ ensureUnit( value ) } !important;` );
		}
	} );

	// Margin - get inherited values
	[ 'top', 'right', 'bottom', 'left' ].forEach( ( side ) => {
		const value = getInheritedValue( responsiveMargin, device, side );
		if ( value ) {
			// Editor preview: use !important to override theme styles in editor
			rules.push( `margin-${ side }: ${ ensureUnit( value ) } !important;` );
		}
	} );

	// Font size - get inherited value
	const fontSizeIndex = deviceOrder.indexOf( device );
	let fontSize = null;
	for ( let i = fontSizeIndex; i >= 0; i-- ) {
		const checkDevice = deviceOrder[ i ];
		if ( responsiveFontSize[ checkDevice ] ) {
			fontSize = responsiveFontSize[ checkDevice ];
			break;
		}
	}
	if ( fontSize ) {
		// Editor preview: use !important to override theme styles in editor
		rules.push( `font-size: ${ ensureUnit( fontSize ) } !important;` );
	}

	if ( rules.length === 0 ) {
		return '';
	}

	// Build CSS with VERY high specificity to override WordPress core
	// Use body prefix + triple selector + !important (similar to frontend approach)
	// Note: .editor-styles-wrapper is a class ON the body element, not a child element
	const highSpecSelector = `body.editor-styles-wrapper ${ selector }${ selector }${ selector }`;
	let css = `${ highSpecSelector } { ${ rules.join( ' ' ) } }\n`;
	
	return css;
}

/**
 * Get editor document (iframe or main document).
 */
function getEditorDocument() {
	const iframe = document.querySelector( 'iframe[name="editor-canvas"]' );
	return iframe ? iframe.contentDocument : document;
}

/**
 * Inject styles into editor for a specific device.
 */
export function injectEditorStyles( attributes, blockId, selectedDevice = 'desktop' ) {
	if ( ! attributes.responsiveId ) {
		return;
	}

	const doc = getEditorDocument();
	if ( ! doc ) {
		return;
	}

	const styleId = `rbc-styles-${ blockId }`;
	let styleEl = doc.getElementById( styleId );

	if ( ! styleEl ) {
		styleEl = doc.createElement( 'style' );
		styleEl.id = styleId;
		doc.head.appendChild( styleEl );
	}

	const css = generateDeviceCSS( attributes, selectedDevice );
	styleEl.textContent = css;

	// Debug
	console.log( `[SRB] Injected ${ selectedDevice } styles for block:`, blockId );
	console.log( css );
}

/**
 * Remove editor styles.
 */
export function removeEditorStyles( blockId ) {
	const doc = getEditorDocument();
	if ( ! doc ) {
		return;
	}

	const styleId = `rbc-styles-${ blockId }`;
	const styleEl = doc.getElementById( styleId );

	if ( styleEl ) {
		styleEl.remove();
	}
}
