/**
 * WordPress Dependencies
 */
import { createHigherOrderComponent } from '@wordpress/compose';
import { Fragment, useEffect } from '@wordpress/element';
import { InspectorControls } from '@wordpress/block-editor';

/**
 * Internal Dependencies
 */
import { isBlockSupported, generateUniqueId } from '../utils/config';
import { removeEditorStyles } from '../utils/editor-styles';
import ResponsivePanel from '../components/ResponsivePanel';

/**
 * Add responsive controls to supported blocks.
 */
export const withResponsiveControls = createHigherOrderComponent(
	( BlockEdit ) => {
		return ( props ) => {
			const { name, attributes, setAttributes, clientId } = props;

			// Only add controls to supported blocks.
			if ( ! isBlockSupported( name ) ) {
				return <BlockEdit { ...props } />;
			}

			// Ensure responsive ID matches clientId (fixes duplication issues)
			// clientId is guaranteed unique per block, even for duplicates
			useEffect( () => {
				if ( ! attributes.responsiveId || attributes.responsiveId !== clientId ) {
					setAttributes( { responsiveId: clientId } );
				}
			}, [ clientId, attributes.responsiveId, setAttributes ] );

			// Clean up editor styles when block is unmounted
			useEffect( () => {
				return () => {
					removeEditorStyles( clientId );
				};
			}, [ clientId ] );

			return (
				<Fragment>
					<BlockEdit { ...props } />
					<InspectorControls>
						<ResponsivePanel { ...props } />
					</InspectorControls>
				</Fragment>
			);
		};
	},
	'withResponsiveControls'
);
