/**
 * WordPress Dependencies
 */
import { PanelBody, CheckboxControl, Notice } from '@wordpress/components';
import { __ } from '@wordpress/i18n';

/**
 * Visibility Control Component
 *
 * Uses PanelBody like other WordPress settings panels.
 */
const VisibilityControl = ( { attributes, setAttributes } ) => {
	const visibility = attributes.responsiveVisibility || {};

	const handleToggle = ( device, checked ) => {
		const updatedVisibility = {
			...visibility,
			[ `hide${ device.charAt( 0 ).toUpperCase() + device.slice( 1 ) }` ]: checked,
		};

		setAttributes( { responsiveVisibility: updatedVisibility } );
	};

	const allHidden =
		visibility.hideDesktop &&
		visibility.hideTablet &&
		visibility.hideMobile;

	const hasVisibilitySettings = 
		visibility.hideDesktop ||
		visibility.hideTablet ||
		visibility.hideMobile;

	return (
		<PanelBody
			title={ __( 'Visibility', 'swell-smart-responsive-blocks' ) }
			initialOpen={ hasVisibilitySettings }
		>
			<CheckboxControl
				label={ __( 'Hide on desktop', 'swell-smart-responsive-blocks' ) }
				checked={ !! visibility.hideDesktop }
				onChange={ ( checked ) => handleToggle( 'desktop', checked ) }
			/>

			<CheckboxControl
				label={ __( 'Hide on tablet', 'swell-smart-responsive-blocks' ) }
				checked={ !! visibility.hideTablet }
				onChange={ ( checked ) => handleToggle( 'tablet', checked ) }
			/>

			<CheckboxControl
				label={ __( 'Hide on mobile', 'swell-smart-responsive-blocks' ) }
				checked={ !! visibility.hideMobile }
				onChange={ ( checked ) => handleToggle( 'mobile', checked ) }
			/>

			{ allHidden && (
				<Notice status="warning" isDismissible={ false }>
					{ __( 'This block is hidden on all devices.', 'swell-smart-responsive-blocks' ) }
				</Notice>
			) }
		</PanelBody>
	);
};

export default VisibilityControl;
