/**
 * WordPress Dependencies
 */
import {
	__experimentalBoxControl as BoxControl,
	__experimentalUseCustomUnits as useCustomUnits,
	BaseControl,
	Button,
	Flex,
	FlexItem,
} from '@wordpress/components';
import { __ } from '@wordpress/i18n';
import { settings } from '@wordpress/icons';

/**
 * Spacing Control Component
 * 
 * Simple BoxControl matching WordPress core.
 */
const SpacingControl = ( {
	label,
	attributeName,
	selectedDevice,
	attributes,
	setAttributes,
} ) => {
	const spacingValues = attributes[ attributeName ] || {};
	const deviceValues = spacingValues[ selectedDevice ] || {};

	// Use WordPress custom units
	const units = useCustomUnits( {
		availableUnits: [ 'px', '%', 'em', 'rem' ],
	} );

	const handleChange = ( newValues ) => {
		// Ensure all values have units (add 'px' if missing)
		const valuesWithUnits = {};
		if ( newValues ) {
			Object.keys( newValues ).forEach( ( side ) => {
				if ( newValues[ side ] !== undefined && newValues[ side ] !== null && newValues[ side ] !== '' ) {
					const value = String( newValues[ side ] );
					// If it's just a number, add 'px'
					valuesWithUnits[ side ] = /^\d+\.?\d*$/.test( value ) ? value + 'px' : value;
				}
			} );
		}

		const updatedSpacing = {
			...spacingValues,
			[ selectedDevice ]: valuesWithUnits,
		};
		setAttributes( { [ attributeName ]: updatedSpacing } );
	};

	const handleReset = () => {
		const updatedSpacing = { ...spacingValues };
		delete updatedSpacing[ selectedDevice ];
		setAttributes( { [ attributeName ]: updatedSpacing } );
	};

	const hasValue = deviceValues && Object.keys( deviceValues ).some( side => deviceValues[ side ] );

	return (
		<BaseControl
			id={ `rbc-spacing-${attributeName}-${selectedDevice}` }
			className="rbc-spacing-control"
			help={ ! hasValue ? __( 'Inherits from larger breakpoint', 'swell-smart-responsive-blocks' ) : undefined }
		>
			<Flex justify="space-between" style={{ marginBottom: '8px' }}>
				<FlexItem>
					<BaseControl.VisualLabel>
						{ label }
					</BaseControl.VisualLabel>
				</FlexItem>
				{ hasValue && (
					<FlexItem>
						<Button
							size="small"
							variant="tertiary"
							onClick={ handleReset }
						>
							{ __( 'Reset', 'swell-smart-responsive-blocks' ) }
						</Button>
					</FlexItem>
				) }
			</Flex>
		<BoxControl
			values={ deviceValues }
			onChange={ handleChange }
			units={ units }
			sides={ [ 'top', 'bottom', 'left', 'right' ] }
			splitOnAxis={ false }
			label=""
		/>
		</BaseControl>
	);
};

export default SpacingControl;
