/**
 * WordPress Dependencies
 */
import { useState, useEffect } from '@wordpress/element';
import {
	__experimentalToolsPanel as ToolsPanel,
	__experimentalToolsPanelItem as ToolsPanelItem,
	FontSizePicker,
} from '@wordpress/components';
import {
	__experimentalSpacingSizesControl as SpacingSizesControl,
} from '@wordpress/block-editor';
import { __ } from '@wordpress/i18n';
import { useSelect } from '@wordpress/data';
import { store as editPostStore } from '@wordpress/edit-post';

/**
 * Internal Dependencies
 */
import DeviceToggle from './DeviceToggle';
import VisibilityControl from './VisibilityControl';
import CopyPasteButtons from './CopyPasteButtons';
import QuickActions from './QuickActions';
import InheritanceIndicator from './InheritanceIndicator';
import { supportsTypography } from '../utils/config';
import { injectEditorStyles } from '../utils/editor-styles';

/**
 * Responsive Panel Component
 * 
 * Matches WordPress core Dimensions panel exactly using ToolsPanel.
 */
const ResponsivePanel = ( props ) => {
	const { name, attributes, clientId, setAttributes } = props;
	const [ selectedDevice, setSelectedDevice ] = useState( 'desktop' );

	// Listen to WordPress's viewport changes (two-way sync)
	const editorDeviceType = useSelect( ( select ) => {
		// Get the current preview device type from WordPress
		const deviceType = select( editPostStore ).__experimentalGetPreviewDeviceType?.();
		return deviceType || 'Desktop'; // Default to Desktop if not available
	}, [] );

	// Sync our device selection when WordPress viewport changes
	useEffect( () => {
		if ( editorDeviceType ) {
			// Convert WordPress device type (capitalized) to our format (lowercase)
			const deviceMap = {
				'Desktop': 'desktop',
				'Tablet': 'tablet',
				'Mobile': 'mobile',
			};
			const newDevice = deviceMap[ editorDeviceType ] || 'desktop';
			if ( newDevice !== selectedDevice ) {
				setSelectedDevice( newDevice );
			}
		}
	}, [ editorDeviceType ] );

	// Inject styles whenever attributes OR device changes
	useEffect( () => {
		injectEditorStyles( attributes, clientId, selectedDevice );
	}, [ attributes, clientId, selectedDevice ] );

	// Font size presets
	const fontSizes = [
		{ name: __( 'Small', 'swell-smart-responsive-blocks' ), slug: 'small', size: '13px' },
		{ name: __( 'Medium', 'swell-smart-responsive-blocks' ), slug: 'medium', size: '20px' },
		{ name: __( 'Large', 'swell-smart-responsive-blocks' ), slug: 'large', size: '36px' },
		{ name: __( 'Extra Large', 'swell-smart-responsive-blocks' ), slug: 'x-large', size: '42px' },
	];

	// Get current device values
	const paddingValue = ( attributes.responsivePadding || {} )[ selectedDevice ];
	const marginValue = ( attributes.responsiveMargin || {} )[ selectedDevice ];
	const fontSizeValue = ( attributes.responsiveFontSize || {} )[ selectedDevice ];

	// Check if items have values
	const hasPadding = !! paddingValue && Object.keys( paddingValue ).length > 0;
	const hasMargin = !! marginValue && Object.keys( marginValue ).length > 0;
	const hasFontSize = !! fontSizeValue;

	// Helper functions to get inherited values
	const getInheritedPadding = () => {
		if ( selectedDevice === 'mobile' ) {
			return ( attributes.responsivePadding || {} ).tablet || ( attributes.responsivePadding || {} ).desktop;
		}
		if ( selectedDevice === 'tablet' ) {
			return ( attributes.responsivePadding || {} ).desktop;
		}
		return null;
	};

	const getInheritedMargin = () => {
		if ( selectedDevice === 'mobile' ) {
			return ( attributes.responsiveMargin || {} ).tablet || ( attributes.responsiveMargin || {} ).desktop;
		}
		if ( selectedDevice === 'tablet' ) {
			return ( attributes.responsiveMargin || {} ).desktop;
		}
		return null;
	};

	const getInheritedFontSize = () => {
		if ( selectedDevice === 'mobile' ) {
			return ( attributes.responsiveFontSize || {} ).tablet || ( attributes.responsiveFontSize || {} ).desktop;
		}
		if ( selectedDevice === 'tablet' ) {
			return ( attributes.responsiveFontSize || {} ).desktop;
		}
		return null;
	};

	// Handlers
	const handlePaddingChange = ( newValue ) => {
		const updated = { ...( attributes.responsivePadding || {} ), [ selectedDevice ]: newValue };
		if ( ! newValue || Object.keys( newValue ).length === 0 ) {
			delete updated[ selectedDevice ];
		}
		setAttributes( { responsivePadding: updated } );
	};

	const handleMarginChange = ( newValue ) => {
		const updated = { ...( attributes.responsiveMargin || {} ), [ selectedDevice ]: newValue };
		if ( ! newValue || Object.keys( newValue ).length === 0 ) {
			delete updated[ selectedDevice ];
		}
		setAttributes( { responsiveMargin: updated } );
	};

	const handleFontSizeChange = ( newValue ) => {
		const updated = { ...( attributes.responsiveFontSize || {} ), [ selectedDevice ]: newValue };
		if ( ! newValue ) {
			delete updated[ selectedDevice ];
		}
		setAttributes( { responsiveFontSize: updated } );
	};

	// Reset handlers
	const resetPadding = () => {
		const updated = { ...( attributes.responsivePadding || {} ) };
		delete updated[ selectedDevice ];
		setAttributes( { responsivePadding: updated } );
	};

	const resetMargin = () => {
		const updated = { ...( attributes.responsiveMargin || {} ) };
		delete updated[ selectedDevice ];
		setAttributes( { responsiveMargin: updated } );
	};

	const resetFontSize = () => {
		const updated = { ...( attributes.responsiveFontSize || {} ) };
		delete updated[ selectedDevice ];
		setAttributes( { responsiveFontSize: updated } );
	};

	const resetAll = () => {
		resetPadding();
		resetMargin();
		if ( supportsTypography( name ) || hasFontSize ) {
			resetFontSize();
		}
	};

	return (
		<>
			{/* Device Toggle */}
			<div className="rbc-device-toggle-wrapper">
				<DeviceToggle
					value={ selectedDevice }
					onChange={ setSelectedDevice }
				/>
			</div>

			{/* NEW: Workflow Features */}
			<div className="rbc-workflow-controls">
				<CopyPasteButtons 
					attributes={ attributes }
					setAttributes={ setAttributes }
				/>
				<QuickActions 
					attributes={ attributes }
					setAttributes={ setAttributes }
					blockName={ name }
				/>
			</div>

		<ToolsPanel
			label={ __( 'Responsive', 'swell-smart-responsive-blocks' ) }
			resetAll={ resetAll }
			panelId={ clientId }
			className="rbc-tools-panel"
			dropdownMenuProps={ {
				popoverProps: {
					placement: 'left-start'
				}
			} }
		>
			{/* Padding */}
			<ToolsPanelItem
				hasValue={ () => hasPadding }
				label={ __( 'Padding', 'swell-smart-responsive-blocks' ) }
				onDeselect={ resetPadding }
				isShownByDefault={ true }
				panelId={ clientId }
			>
				<InheritanceIndicator
					currentDevice={ selectedDevice }
					currentValue={ paddingValue }
					inheritedValue={ getInheritedPadding() }
					label="Padding"
					onOverride={ () => {
						// Set a default value to override
						handlePaddingChange( {
							top: '16px',
							right: '16px',
							bottom: '16px',
							left: '16px',
						} );
					} }
					onReset={ resetPadding }
				/>
				<SpacingSizesControl
					label={ __( 'Padding', 'swell-smart-responsive-blocks' ) }
					values={ paddingValue }
					onChange={ handlePaddingChange }
					sides={ [ 'top', 'right', 'bottom', 'left' ] }
				/>
			</ToolsPanelItem>

			{/* Margin */}
			<ToolsPanelItem
				hasValue={ () => hasMargin }
				label={ __( 'Margin', 'swell-smart-responsive-blocks' ) }
				onDeselect={ resetMargin }
				isShownByDefault={ true }
				panelId={ clientId }
			>
				<InheritanceIndicator
					currentDevice={ selectedDevice }
					currentValue={ marginValue }
					inheritedValue={ getInheritedMargin() }
					label="Margin"
					onOverride={ () => {
						handleMarginChange( {
							top: '0px',
							right: '0px',
							bottom: '0px',
							left: '0px',
						} );
					} }
					onReset={ resetMargin }
				/>
				<SpacingSizesControl
					label={ __( 'Margin', 'swell-smart-responsive-blocks' ) }
					values={ marginValue }
					onChange={ handleMarginChange }
					sides={ [ 'top', 'right', 'bottom', 'left' ] }
				/>
			</ToolsPanelItem>

				{/* Font Size - only for typography blocks or blocks with existing data */}
				{ ( supportsTypography( name ) || hasFontSize ) && (
					<ToolsPanelItem
						hasValue={ () => hasFontSize }
						label={ __( 'Font size', 'swell-smart-responsive-blocks' ) }
						onDeselect={ resetFontSize }
						isShownByDefault={ supportsTypography( name ) }
						panelId={ clientId }
					>
						<FontSizePicker
							label={ __( 'Font size', 'swell-smart-responsive-blocks' ) }
							value={ fontSizeValue }
							onChange={ handleFontSizeChange }
							fontSizes={ fontSizes }
							disableCustomFontSizes={ false }
							withSlider
						/>
					</ToolsPanelItem>
				) }
			</ToolsPanel>

			<VisibilityControl 
				attributes={ attributes }
				setAttributes={ setAttributes }
			/>
		</>
	);
};

export default ResponsivePanel;
