/**
 * Quick Actions Component
 */
import { Button, Dropdown, MenuGroup, MenuItem } from '@wordpress/components';
import { __ } from '@wordpress/i18n';
import { moreVertical } from '@wordpress/icons';
import { quickActionPresets } from '../utils/quick-actions';

/**
 * Quick Actions Dropdown
 */
const QuickActions = ( { attributes, setAttributes, blockName } ) => {
	/**
	 * Apply a preset
	 */
	const applyPreset = ( presetKey ) => {
		const preset = quickActionPresets[ presetKey ];
		if ( preset ) {
			// Check if preset is applicable to this block type
			if ( preset.validFor && ! preset.validFor( blockName ) ) {
				// Skip if not valid for this block
				return;
			}
			
			const newAttributes = preset.apply( attributes );
			setAttributes( newAttributes );
		}
	};

	return (
		<div className="rbc-quick-actions">
			<Dropdown
				className="rbc-quick-actions-dropdown"
				contentClassName="rbc-quick-actions-popover"
				popoverProps={ {
					placement: 'left-start'
				} }
				renderToggle={ ( { isOpen, onToggle } ) => (
					<Button
						variant="secondary"
						onClick={ onToggle }
						aria-expanded={ isOpen }
						icon={ moreVertical }
					>
						{ __( 'Quick Actions', 'swell-smart-responsive-blocks' ) }
					</Button>
				) }
				renderContent={ ( { onClose } ) => (
					<MenuGroup>
						{ Object.keys( quickActionPresets ).map( ( key ) => {
							const preset = quickActionPresets[ key ];
							
							// Check if this preset is valid for the current block type
							const isDisabled = preset.validFor && ! preset.validFor( blockName );
							
							const handleClick = () => {
								if ( ! isDisabled ) {
									applyPreset( key );
									// Small delay to ensure state updates complete
									setTimeout( () => onClose(), 10 );
								}
							};
							
							return (
								<MenuItem
									key={ key }
									onClick={ handleClick }
									disabled={ isDisabled }
								>
									<div className="rbc-menu-item-content">
										<strong>{ preset.label }</strong>
										<span className="rbc-menu-item-description">
											{ preset.description }
										</span>
									</div>
								</MenuItem>
							);
						} ) }
					</MenuGroup>
				) }
			/>
		</div>
	);
};

export default QuickActions;
