/**
 * Inheritance Indicator Component
 * 
 * Shows when a value is inherited from a larger breakpoint
 */
import { Button } from '@wordpress/components';
import { __ } from '@wordpress/i18n';

/**
 * Inheritance Indicator
 * 
 * @param {string} currentDevice - Current device (mobile/tablet/desktop)
 * @param {*} currentValue - Current value for this device
 * @param {*} inheritedValue - Value inherited from larger breakpoint
 * @param {string} label - Label for the control
 * @param {Function} onOverride - Callback when user clicks override
 * @param {Function} onReset - Callback when user clicks reset
 */
const InheritanceIndicator = ( {
	currentDevice,
	currentValue,
	inheritedValue,
	label,
	onOverride,
	onReset,
} ) => {
	// Desktop never inherits
	if ( currentDevice === 'desktop' ) {
		return null;
	}

	// If there's a current value, show reset option
	if ( currentValue !== undefined && currentValue !== null && currentValue !== '' ) {
		return (
			<div className="rbc-inheritance-indicator rbc-has-override">
				<span className="rbc-indicator-icon">🔗</span>
				<span className="rbc-indicator-text">
					{ __( 'Custom value set', 'swell-smart-responsive-blocks' ) }
				</span>
				<Button
					variant="link"
					onClick={ onReset }
					className="rbc-reset-button"
					isSmall
				>
					{ __( 'Reset to inherit', 'swell-smart-responsive-blocks' ) }
				</Button>
			</div>
		);
	}

	// If no current value but there's an inherited value, show inheritance
	if ( inheritedValue !== undefined && inheritedValue !== null && inheritedValue !== '' ) {
		const fromDevice = currentDevice === 'mobile' ? 'tablet/desktop' : 'desktop';
		
		return (
			<div className="rbc-inheritance-indicator rbc-is-inheriting">
				<span className="rbc-indicator-icon">⬆️</span>
				<span className="rbc-indicator-text">
					{ __( 'Inherited from ', 'swell-smart-responsive-blocks' ) + fromDevice }
				</span>
				<Button
					variant="link"
					onClick={ onOverride }
					className="rbc-override-button"
					isSmall
				>
					{ __( 'Override', 'swell-smart-responsive-blocks' ) }
				</Button>
			</div>
		);
	}

	return null;
};

export default InheritanceIndicator;
