/**
 * WordPress Dependencies
 */
import { __experimentalToggleGroupControl as ToggleGroupControl, __experimentalToggleGroupControlOption as ToggleGroupControlOption } from '@wordpress/components';
import { desktop, tablet, mobile } from '@wordpress/icons';
import { __ } from '@wordpress/i18n';
import { useDispatch } from '@wordpress/data';
import { useEffect } from '@wordpress/element';

/**
 * Internal Dependencies
 */
import { getBreakpoints } from '../utils/config';

/**
 * Device Toggle Component
 *
 * Allows switching between desktop, tablet, and mobile views.
 * Attempts to sync with WordPress editor preview device.
 */
const DeviceToggle = ( { value, onChange } ) => {
	const breakpoints = getBreakpoints();

	const icons = {
		desktop,
		tablet,
		smartphone: mobile,
	};

	// Try to sync with WordPress editor preview
	const { __experimentalSetPreviewDeviceType } = useDispatch( 'core/edit-post' ) || {};

	const handleChange = ( newValue ) => {
		onChange( newValue );

		// Try to change WordPress editor preview device
		if ( __experimentalSetPreviewDeviceType ) {
			const deviceMap = {
				desktop: 'Desktop',
				tablet: 'Tablet',
				mobile: 'Mobile',
			};
			__experimentalSetPreviewDeviceType( deviceMap[ newValue ] || 'Desktop' );
		}
	};

	return (
		<ToggleGroupControl
			label={ __( 'Device', 'swell-smart-responsive-blocks' ) }
			value={ value }
			onChange={ handleChange }
			isBlock
		>
			{ Object.entries( breakpoints ).map( ( [ key, config ] ) => (
				<ToggleGroupControlOption
					key={ key }
					value={ key }
					label={ config.label }
					icon={ icons[ config.icon ] }
					showTooltip
				/>
			) ) }
		</ToggleGroupControl>
	);
};

export default DeviceToggle;
