<?php
/**
 * Handle frontend CSS output.
 *
 * @package SwellResponsiveBlockControls
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class SWELLSRB_Frontend
 *
 * Outputs responsive CSS on the frontend.
 */
class SWELLSRB_Frontend {

	/**
	 * Instance of this class.
	 *
	 * @var SWELLSRB_Frontend
	 */
	private static $instance = null;

	/**
	 * Get the singleton instance.
	 *
	 * @return SWELLSRB_Frontend
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 */
	private function __construct() {
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_frontend_styles' ) );
	}

	/**
	 * Enqueue frontend styles and add inline CSS.
	 */
	public function enqueue_frontend_styles() {
		// Register a minimal handle for our inline styles with version
		wp_register_style( 'swell-smart-responsive-blocks-frontend', false, array(), SWELLSRB_VERSION );
		wp_enqueue_style( 'swell-smart-responsive-blocks-frontend' );
		
		// Generate and add inline CSS
		$css = $this->get_page_responsive_css();
		
		if ( ! empty( $css ) ) {
			wp_add_inline_style( 'swell-smart-responsive-blocks-frontend', $css );
		}
	}

	/**
	 * Get responsive CSS for the current page.
	 * 
	 * @return string Generated CSS.
	 */
	private function get_page_responsive_css() {
		global $post;

		if ( ! $post ) {
			return '';
		}

		$content = $post->post_content;
		$css     = $this->generate_css_from_content( $content );

		if ( empty( $css ) ) {
			return '';
		}

		// Sanitize CSS before output
		return $this->sanitize_css( $css );
	}

	/**
	 * Sanitize CSS output for safe display.
	 * 
	 * CSS is generated from sanitized block attributes, but this adds
	 * an extra layer of security by removing any potentially harmful content.
	 *
	 * @param string $css CSS to sanitize.
	 * @return string Sanitized CSS.
	 */
	private function sanitize_css( $css ) {
		// Remove any script tags (defense in depth - shouldn't exist)
		$css = preg_replace( '/<script\b[^>]*>.*?<\/script>/is', '', $css );
		
		// Remove any HTML tags except style-related
		$css = wp_strip_all_tags( $css );
		
		// Remove any javascript: protocol (extra safety)
		$css = preg_replace( '/javascript:/i', '', $css );
		
		return $css;
	}

	/**
	 * Generate CSS from post content.
	 *
	 * @param string $content Post content with blocks.
	 * @return string Generated CSS.
	 */
	private function generate_css_from_content( $content ) {
		$blocks = parse_blocks( $content );
		$css    = '';

		// Add visibility utility classes once.
		$css .= SWELLSRB_CSS_Generator::generate_visibility_utilities();

		// Generate CSS for each block.
		foreach ( $this->flatten_blocks( $blocks ) as $block ) {
			$css .= SWELLSRB_CSS_Generator::generate_block_css( $block );
		}

		return $css;
	}

	/**
	 * Flatten nested blocks into a single array.
	 *
	 * @param array $blocks Array of blocks.
	 * @return array Flattened array of blocks.
	 */
	private function flatten_blocks( $blocks ) {
		$flat = array();

		foreach ( $blocks as $block ) {
			$flat[] = $block;

			if ( ! empty( $block['innerBlocks'] ) ) {
				$flat = array_merge( $flat, $this->flatten_blocks( $block['innerBlocks'] ) );
			}
		}

		return $flat;
	}
}
