<?php
/**
 * Generate CSS from responsive block attributes.
 *
 * @package SwellResponsiveBlockControls
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class SWELLSRB_CSS_Generator
 *
 * Generates responsive CSS from block attributes.
 */
class SWELLSRB_CSS_Generator {

	/**
	 * Breakpoint configuration.
	 *
	 * @var array
	 */
	private static $breakpoints = array(
		'desktop' => array( 'max_width' => null ),
		'tablet'  => array( 'max_width' => 1024 ),
		'mobile'  => array( 'max_width' => 767 ),
	);

	/**
	 * Generate CSS for a single block.
	 *
	 * @param array $block Block data with attributes.
	 * @return string Generated CSS.
	 */
	public static function generate_block_css( $block ) {
		$attrs = isset( $block['attrs'] ) ? $block['attrs'] : array();
		$css   = '';

		// Skip if no responsive ID.
		if ( empty( $attrs['responsiveId'] ) ) {
			return $css;
		}

		$block_id    = sanitize_html_class( $attrs['responsiveId'] );
		$block_class = '.rbc-block-' . $block_id;

		// Generate spacing CSS.
		$css .= self::generate_spacing_css( $block_class, $attrs );

		// Generate typography CSS.
		$css .= self::generate_typography_css( $block_class, $attrs );

		// Generate visibility CSS.
		$css .= self::generate_visibility_css( $attrs );

		return $css;
	}

	/**
	 * Generate spacing (padding/margin) CSS.
	 *
	 * @param string $selector CSS selector.
	 * @param array  $attrs Block attributes.
	 * @return string Generated CSS.
	 */
	private static function generate_spacing_css( $selector, $attrs ) {
		$css = '';

		foreach ( self::$breakpoints as $breakpoint => $config ) {
			$rules = array();

			// Padding.
			if ( ! empty( $attrs['responsivePadding'][ $breakpoint ] ) ) {
				$padding = $attrs['responsivePadding'][ $breakpoint ];
				$rules[] = self::format_spacing_rule( 'padding', $padding );
			}

			// Margin.
			if ( ! empty( $attrs['responsiveMargin'][ $breakpoint ] ) ) {
				$margin = $attrs['responsiveMargin'][ $breakpoint ];
				$rules[] = self::format_spacing_rule( 'margin', $margin );
			}

		if ( ! empty( $rules ) ) {
			// Use higher specificity + !important to override WordPress core inline styles
			// Core's Dimensions panel applies padding/margin as inline styles (element.style)
			// which have higher specificity than any selector. !important is the only way to override.
			$css .= self::wrap_in_media_query(
				'body .wp-site-blocks ' . $selector . ' { ' . implode( ' ', $rules ) . ' }',
				$breakpoint
			);
		}
		}

		return $css;
	}

	/**
	 * Generate typography CSS.
	 *
	 * @param string $selector CSS selector.
	 * @param array  $attrs Block attributes.
	 * @return string Generated CSS.
	 */
	private static function generate_typography_css( $selector, $attrs ) {
		$css = '';

		if ( empty( $attrs['responsiveFontSize'] ) ) {
			return $css;
		}

		foreach ( self::$breakpoints as $breakpoint => $config ) {
			if ( ! empty( $attrs['responsiveFontSize'][ $breakpoint ] ) ) {
				$font_size = self::ensure_unit( sanitize_text_field( $attrs['responsiveFontSize'][ $breakpoint ] ) );
				// Use higher specificity + !important to override WordPress core inline styles
				$rule      = 'body .wp-site-blocks ' . $selector . ' { font-size: ' . $font_size . ' !important; }';
				$css      .= self::wrap_in_media_query( $rule, $breakpoint );
			}
		}

		return $css;
	}

	/**
	 * Generate visibility CSS.
	 *
	 * @param array $attrs Block attributes.
	 * @return string Generated CSS.
	 */
	private static function generate_visibility_css( $attrs ) {
		// Visibility CSS is handled via utility classes.
		// This is generated in the frontend class as global styles.
		return '';
	}

	/**
	 * Ensure value has a unit (add 'px' if missing).
	 * Handles WordPress spacing preset tokens.
	 *
	 * @param string $value CSS value.
	 * @return string Value with unit or CSS variable.
	 */
	private static function ensure_unit( $value ) {
		if ( empty( $value ) ) {
			return $value;
		}

		// Handle WordPress spacing preset tokens (from SpacingSizesControl slider)
		// Format: "var:preset|spacing|30" → "var(--wp--preset--spacing--30)"
		if ( strpos( $value, 'var:preset|spacing|' ) === 0 ) {
			$preset_id = str_replace( 'var:preset|spacing|', '', $value );
			return 'var(--wp--preset--spacing--' . $preset_id . ')';
		}

		// If value already ends with a unit, return as-is.
		if ( preg_match( '/^[\d.]+(?:px|em|rem|%|vh|vw)$/', $value ) ) {
			return $value;
		}

		// If it's just a number, add 'px'.
		if ( preg_match( '/^[\d.]+$/', $value ) ) {
			return $value . 'px';
		}

		// Otherwise return as-is (might already have unit or be a CSS variable).
		return $value;
	}

	/**
	 * Format spacing rule from sides object.
	 *
	 * @param string $property CSS property (padding or margin).
	 * @param array  $sides Sides object with top/right/bottom/left.
	 * @return string Formatted CSS rule.
	 */
	private static function format_spacing_rule( $property, $sides ) {
		$parts = array();

		foreach ( array( 'top', 'right', 'bottom', 'left' ) as $side ) {
			// Check if key exists and value is not null/empty string
			if ( isset( $sides[ $side ] ) && $sides[ $side ] !== '' && $sides[ $side ] !== null ) {
				$raw_value = sanitize_text_field( $sides[ $side ] );
				
				// Skip "0" and "0px" values - let blocks use their default styles
				// If user wants 0 padding, they should clear the responsive setting
				if ( $raw_value === '0' || $raw_value === '0px' ) {
					continue;
				}
				
				$value   = self::ensure_unit( $raw_value );
				// Use !important to override WordPress core's inline spacing styles
				$parts[] = $property . '-' . $side . ': ' . $value . ' !important;';
			}
		}

		return implode( ' ', $parts );
	}

	/**
	 * Wrap CSS rule in media query if needed.
	 *
	 * @param string $rule CSS rule.
	 * @param string $breakpoint Breakpoint name.
	 * @return string CSS with or without media query.
	 */
	private static function wrap_in_media_query( $rule, $breakpoint ) {
		// Use specific ranges to prevent cascade issues
		if ( 'desktop' === $breakpoint ) {
			// Desktop: 1025px and above
			return "@media (min-width: 1025px) { {$rule} }\n";
		}

		if ( 'tablet' === $breakpoint ) {
			// Tablet: 768px to 1024px (specific range)
			return "@media (min-width: 768px) and (max-width: 1024px) { {$rule} }\n";
		}

		if ( 'mobile' === $breakpoint ) {
			// Mobile: 767px and below
			return "@media (max-width: 767px) { {$rule} }\n";
		}

		return $rule . "\n";
	}

	/**
	 * Generate visibility utility classes.
	 *
	 * @return string CSS for visibility utilities.
	 */
	public static function generate_visibility_utilities() {
		$css = '';

		// Hide on desktop ONLY (screens 1025px and above).
		$css .= "@media (min-width: 1025px) { .rbc-hide-desktop { display: none !important; } }\n";

		// Hide on tablet ONLY (screens 768px - 1024px).
		$css .= "@media (min-width: 768px) and (max-width: 1024px) { .rbc-hide-tablet { display: none !important; } }\n";

		// Hide on mobile ONLY (screens 767px and below).
		$css .= "@media (max-width: 767px) { .rbc-hide-mobile { display: none !important; } }\n";

		return $css;
	}
}
