<?php
if (!defined('ABSPATH')) {
    exit;
}
const SURF_SR_DEBUG_SR = false;
class SURF_SR_HyperDBReplace
{
    private $wpdb;
    private $batch_size = 500;
    private $reports = [];
    private $table_skipped = '';
    // New property to accumulate detailed metrics per table/column
    private $report_details = [];



    // New property to control case sensitivity (default true for case‐sensitive)
    private $case_insensitive = false;



    private $modified_table = [];





    public function __construct()
    {

        global $wpdb;
        $this->wpdb = $wpdb;
        add_action('wp_ajax_surf_sr_process_replace', [$this, 'ajax_process_replace']);
    }






    private static function log($msg)
    {
        if (SURF_SR_DEBUG_SR)
            error_log('[SURFL SR] ' . $msg);
    }



    private function replace_value($value, $search, $replace)
    {

        // Ensure inputs are strings
        $value_str = (string) $value;

        // Use mb_stripos for UTF-8 aware position check
        if ($this->case_insensitive) {
            $found = mb_stripos($value_str, $search);
        } else {
            $found = mb_strpos($value_str, $search);
        }

        if ($found === false) {
            return $value;
        }

        if ($this->is_serialized_data($value)) {
            try {
                $unserialized = @unserialize($value, ['allowed_classes' => false]);
                $modified = $this->deep_replace($search, $replace, $unserialized, $this->case_insensitive);
                $reserialized = serialize($modified);

                // Validate the reserialized data
                if ($this->is_serialized_data($reserialized)) {
                    return $reserialized;
                }
                return $value; // Fallback if reserialization fails
            } catch (Exception $e) {
                self::log('Serialization error in replace_value: ' . $e->getMessage());
                return $value;
            }
        }

        // MULTILINGUAL & UTF-8 COMPATIBILITY
        if ($this->case_insensitive) {
            // Quote the search string so regex characters (like ., *, +) are treated as text
            $quoted_search = preg_quote($search, '/');
            // 'u' modifier = UTF-8, 'i' modifier = Case Insensitive
            $data = preg_replace('/' . $quoted_search . '/iu', $replace, $value_str);
        } else {
            // Standard str_replace is usually fine for Case-Sensitive UTF-8
            $data = str_replace($search, $replace, $value_str);
        }



        return $data;
    }




    // Modified to accept $dry_run. In a dry run, we do not update the DB.
    private function process_options_table($table, $search, $replace, $dry_run, $batch_size = 500)
    {
        $offset = 0;
        $siteurlRow = null;

        if (!$dry_run) {
            $this->wpdb->query('START TRANSACTION');
        }

        try {
            while (true) {
                $rows = $this->wpdb->get_results(
                    $this->wpdb->prepare(
                        "SELECT option_name, option_value 
                     FROM $table 
                     WHERE option_name != %s
                     LIMIT %d OFFSET %d",
                        'siteurl',
                        $this->batch_size,
                        $offset
                    ),
                    ARRAY_A
                );

                if (empty($rows)) {
                    break; // no more rows
                }

                foreach ($rows as $row) {
                    $original_name = $row['option_name'];
                    $original_value = $row['option_value'];

                    // Search and replace on both option_name and option_value
                    $modified_name = $this->replace_value($original_name, $search, $replace);
                    $modified_value = $this->replace_value($original_value, $search, $replace);

                    // If either option_name or option_value was modified
                    if ($modified_name !== $original_name || $modified_value !== $original_value) {
                        // Count occurrences in both option_name and option_value
                        $occ_name = $this->count_occurrences($search, $original_name);
                        $occ_value = $this->count_occurrences($search, $original_value);
                        $total_occurrences = $occ_name + $occ_value;

                        // Update report details for both columns
                        if (!isset($this->report_details[$table]['columns']['option_name'])) {
                            $this->report_details[$table]['columns']['option_name'] = ['occurrences' => 0];
                        }
                        if (!isset($this->report_details[$table]['columns']['option_value'])) {
                            $this->report_details[$table]['columns']['option_value'] = ['occurrences' => 0];
                        }
                        $this->report_details[$table]['columns']['option_name']['occurrences'] += $occ_name;
                        $this->report_details[$table]['columns']['option_value']['occurrences'] += $occ_value;

                        // Update the row if it's not a dry run
                        if (!$dry_run) {
                            $result = $this->wpdb->update(
                                $table,
                                ['option_name' => $modified_name, 'option_value' => $modified_value],
                                ['option_name' => $original_name]
                            );
                            if ($result === false) {
                                throw new Exception("Failed to update option '" . $original_name . "' in table $table: " . $this->wpdb->last_error);
                            }
                        }
                        $this->reports[$table] = ($this->reports[$table] ?? 0) + 1;
                    }
                }

                $offset += $batch_size;
            }

            // Handle siteurl last
            $siteurlRow = $this->wpdb->get_row(
                "SELECT option_name, option_value FROM $table WHERE option_name = 'siteurl'",
                ARRAY_A
            );

            if ($siteurlRow) {

                $original_value = $siteurlRow['option_value'];

                $modified_value = $this->replace_value($original_value, $search, $replace);

                if ($modified_value !== $original_value) {

                    $occ_value = $this->count_occurrences($search, $original_value);
                    $total_occurrences = $occ_value;

                    if (!isset($this->report_details[$table]['columns']['option_value'])) {
                        $this->report_details[$table]['columns']['option_value'] = ['occurrences' => 0];
                    }
                    // Update report details for siteurl
                    $this->report_details[$table]['columns']['option_value']['occurrences'] += $occ_value;

                    if (!$dry_run) {
                        $result = $this->wpdb->update(
                            $table,
                            ['option_value' => $modified_value],
                            ['option_name' => 'siteurl']
                        );
                        if ($result === false) {
                            throw new Exception("Failed to update siteurl option in table $table: " . $this->wpdb->last_error);
                        }
                    }
                    $this->reports[$table] = ($this->reports[$table] ?? 0) + 1;
                }
            }

            if (!$dry_run) {
                $this->wpdb->query('COMMIT');
            }
        } catch (Exception $e) {
            if (!$dry_run) {
                $this->wpdb->query('ROLLBACK');
            }
            throw $e;
        }
    }


    // Modified to accept $dry_run. In dry run mode, the update queries are not executed.
    private function process_batch($table, $primary_key, $columns, $rows, $search, $replace, $dry_run, $is_numeric)
    {
        $case_statements = [];
        $ids = [];

        foreach ($rows as $row) {
            if (!isset($row[$primary_key])) {
                throw new Exception("Primary key $primary_key not found in table $table");
            }

            foreach ($columns as $col) {
                $original = $row[$col];
                $modified = $this->replace_value($original, $search, $replace);

                // Only prepare a CASE update if the value has been modified.
                if ($original !== $modified) {
                    // Count occurrences in the original cell.
                    $occ = $this->count_occurrences($search, $original);
                    if (!isset($this->report_details[$table]['columns'][$col])) {
                        $this->report_details[$table]['columns'][$col] = ['occurrences' => 0];
                    }
                    $this->report_details[$table]['columns'][$col]['occurrences'] += $occ;


                    $placeholder = $is_numeric === false ? '%s' : '%d';

                    if (!$dry_run) {
                        $case_statements[$col][] = $this->wpdb->prepare(
                            "WHEN {$placeholder} THEN %s",
                            $row[$primary_key],
                            $modified
                        );
                    }

                    $ids[] = $row[$primary_key];
                }
            }
        }

        if (!empty($case_statements)) {
            if (!$dry_run) {
                $this->wpdb->query('START TRANSACTION');

                foreach ($case_statements as $col => $cases) {
                    $_table = $this->sanitize_identifier($table);
                    $_primary_key = $this->sanitize_identifier($primary_key);
                    $_col = $this->sanitize_identifier($col);

                    $unique_ids = array_unique($ids);
                    $id_placeholder = $is_numeric ? '%d' : '%s';
                    $id_placeholders = implode(',', array_fill(0, count($unique_ids), $id_placeholder));
                    $case_clause = implode(' ', $cases);

                    $sql = $this->wpdb->prepare(
                        "UPDATE $_table SET $_col = CASE $_primary_key $case_clause END WHERE $_primary_key IN ($id_placeholders)",
                        ...$unique_ids
                    );

                    $result = $this->wpdb->query($sql);
                    if ($result === false) {
                        $this->wpdb->query('ROLLBACK');
                        throw new Exception("Update failed for column $col in table $table: " . $this->wpdb->last_error);
                    }
                }

                $this->wpdb->query('COMMIT');
            }
            $this->reports[$table] = ($this->reports[$table] ?? 0) + count($ids);
        }
    }

    // Helper method to validate and sanitize identifiers (table names, column names, primary keys)
    private function sanitize_identifier($identifier)
    {
        // Allow only alphanumeric characters and underscores
        if (!preg_match('/^[A-Za-z0-9_]+$/', $identifier)) {
            throw new Exception("Invalid identifier: " . $identifier);
        }
        return "`" . $identifier . "`";
    }

    private function get_primary_key($table)
    {
        $columns = $this->wpdb->get_results("DESCRIBE $table", ARRAY_A);
        if (!$columns) {
            $this->table_skipped .= "Failed to DESCRIBE $table.\n";
            return false;
        }

        $primary_keys = [];
        $auto_increment = null;
        $unique_keys = [];
        $column_types = [];

        foreach ($columns as $col) {
            $column_types[$col['Field']] = strtolower($col['Type']);
            if ($col['Key'] === 'PRI') {
                $primary_keys[] = $col['Field'];
            }
            if (stripos($col['Extra'], 'auto_increment') !== false) {
                $auto_increment = $col['Field'];
            }
        }

        // ✅ Unique indexes via SHOW INDEXES (DESCRIBE won't show them)
        $indexes = $this->wpdb->get_results("SHOW INDEXES FROM $table WHERE Non_unique = 0", ARRAY_A);
        foreach ($indexes as $idx) {
            if ($idx['Key_name'] !== 'PRIMARY') {
                $unique_keys[] = $idx['Column_name'];
            }
        }

        $pick = null;

        if (count($primary_keys) === 1) {
            $pick = $primary_keys[0];
        } elseif (count($primary_keys) > 1 && $auto_increment) {
            $pick = $auto_increment;
        } elseif ($auto_increment) {
            $pick = $auto_increment;
        } elseif (count($unique_keys) === 1) {
            $pick = $unique_keys[0];
        } elseif (!empty($columns[0]['Field']) && preg_match('/(^id$|_id$|^i_id$)/i', $columns[0]['Field'])) {
            $pick = $columns[0]['Field'];
        }

        if (!$pick) {
            if (count($primary_keys) > 1) {
                $this->table_skipped .= "Table $table skipped: composite PK without auto_increment.\n";
            } else {
                $this->table_skipped .= "Table $table skipped: no safe PK, AI, UNI, or ID-like first column.\n";
            }
            return false;
        }

        // Detect numeric vs string
        $type = $column_types[$pick] ?? '';
        $is_numeric = preg_match('/int|decimal|float|double|bit|bool/i', $type);

        return [
            'name' => $pick,
            'type' => $is_numeric ? 'numeric' : 'string'
        ];
    }


    private function get_text_columns($table)
    {
        $columns = $this->wpdb->get_results(
            $this->wpdb->prepare(
                "SELECT COLUMN_NAME
            FROM INFORMATION_SCHEMA.COLUMNS
            WHERE TABLE_SCHEMA = DATABASE()
            AND TABLE_NAME = %s
            AND DATA_TYPE IN ('varchar', 'text', 'tinytext', 'mediumtext', 'longtext', 'char', 'enum')",
                $table
            ),
            ARRAY_A
        );

        return array_column($columns, 'COLUMN_NAME');
    }



    private function is_serialized_data($data)
    {
        // First check if the data is a string
        if (!is_string($data)) {
            return false;
        }

        $data = trim($data);

        // Check for serialized data pattern
        if (preg_match('/^([adObis]:|N;)/', $data)) {
            // Verify serialization integrity with strict checking
            try {
                $unserialized = @unserialize($data, ['allowed_classes' => false]);
                // Additional check to prevent false positives
                if ($unserialized === false && $data !== 'b:0;') {
                    return false;
                }
                return true;
            } catch (Exception $e) {
                self::log('Serialization error: ' . $e->getMessage());
                return false;
            }
        }

        return false;
    }

    private function deep_replace($search, $replace, $data, $insensitive = true)
    {



        self::log('unserialized data: ' . print_r($data, true));

        if (is_array($data)) {
            foreach ($data as $key => $value) {
                $data[$key] = $this->deep_replace($search, $replace, $value, $insensitive);
            }
        } elseif (is_object($data)) {
            if ($data instanceof \__PHP_Incomplete_Class) {
                // Skip incomplete objects
                return $data;
            }
            foreach ($data as $key => $value) {
                $data->$key = $this->deep_replace($search, $replace, $value, $insensitive);
            }
        } elseif (is_string($data) && $this->is_serialized_data($data)) {
            $unserialized = @unserialize($data, ['allowed_classes' => false]);

            $modified = $this->deep_replace($search, $replace, $unserialized, $insensitive);
            $reserialized = serialize($modified);

            // Validate the reserialized data
            if ($this->is_serialized_data($reserialized)) {
                return $reserialized;
            }

            return $data;
        } elseif (is_string($data)) {
            $original = $data;

            if ($insensitive) {
                // UTF-8 Safe Case Insensitive Replace
                $quoted_search = preg_quote($search, '/');
                $data = preg_replace('/' . $quoted_search . '/iu', $replace, $data);
            } else {
                $data = str_replace($search, $replace, $data);
            }
        }

        return $data;
    }





    // Helper function to count occurrences in strings, arrays or objects.
    private function count_occurrences($search, $data)
    {
        if (is_array($data)) {
            $count = 0;
            foreach ($data as $item) {
                $count += $this->count_occurrences($search, $item);
            }
            return $count;
        } elseif (is_object($data)) {
            $count = 0;
            foreach (get_object_vars($data) as $item) {
                $count += $this->count_occurrences($search, $item);
            }
            return $count;
        }
        $string_data = (string) $data;
        if ($this->case_insensitive) {
            // Use mb_substr_count with mb_strtolower for UTF-8 safety
            // Requires 'mbstring' extension (Standard in WP environments)
            if (function_exists('mb_strtolower') && function_exists('mb_substr_count')) {
                return mb_substr_count(
                    mb_strtolower($string_data, 'UTF-8'),
                    mb_strtolower($search, 'UTF-8'),
                    'UTF-8'
                );
            }
            // Fallback if mbstring is missing (rare)
            return mb_substr_count($string_data, $search, 'UTF-8');
        }

        return substr_count($string_data, $search);
    }
    private function table_exists($table)
    {
        return $this->wpdb->get_var(
            $this->wpdb->prepare(
                "SHOW TABLES LIKE %s",
                $table
            )
        ) === $table;
    }



    private function get_valid_tables($tables)
    {
        $valid_tables = [];
        $options_tables = [];

        foreach ($tables as $table) {
            // Validate table name to allow only letters, numbers, and underscores
            if (!preg_match('/^[A-Za-z0-9_]+$/', $table)) {
                continue;
            }

            if ($this->table_exists($table)) {
                // Push options table separately
                if (stripos($table, 'options') !== false) {
                    $options_tables[] = $table;
                } else {
                    $valid_tables[] = $table;
                }
            }
        }

        // Append options tables at the end
        return array_merge($valid_tables, $options_tables);
    }




    public function render_ui()
    {
        if (!current_user_can('manage_options')) {
            wp_die('Access denied');
        }
        require_once SURF_SR_PATH . "templates/surf-sr-html.php";
    }



    public function ajax_process_replace()
    {
        check_ajax_referer('surf_sr', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => esc_html__('Access denied', 'surflab-search-replace')]);
        }

        $search = trim(wp_unslash($_POST['search']));
        $replace = trim(wp_unslash($_POST['replace']));
        $tables = isset($_POST['tables']) ? $_POST['tables'] : [];
        self::log('search: ' . $search);
        self::log('replace: ' . $replace);
        self::log('tables ' . print_r($tables, true));

        if (empty($search) || empty($replace) || empty($tables)) {
            wp_send_json_error(['message' => 'Missing Input']);
        }

        if ($search === $replace) {
            wp_send_json_error(['message' => 'Search and Replace input are same']);
        }
        $valid_tables = $this->get_valid_tables($tables);
        $dry_run = isset($_POST['dry_run']) && $_POST['dry_run'] == '1';
        $replace_guid = isset($_POST['replace_guid']) && $_POST['replace_guid'] == '1';
        $case_insensitive = isset($_POST['case_insensitive']) && $_POST['case_insensitive'] == '1';

        $offset = isset($_POST['offset']) ? intval($_POST['offset']) : 0;
        $current_table_index = isset($_POST['current_table_index']) ? intval($_POST['current_table_index']) : 0;

        $transient_key = 'surf_sr_batch_state';

        self::log("Transient Key for this request: " . $transient_key);

        $state = get_transient($transient_key);

        if ($offset === 0 && $current_table_index === 0 || $state === false) {
            // Initial call or state expired/not found
            $state = [
                'total_rows' => 0,
                'processed_rows' => 0,
                'current_table_index' => 0,
                'current_table_offset' => 0,
                'tables_to_process' => $valid_tables,
                'reports' => [],
                'report_details' => [], // Initialize report_details in state
                'modified_table' => [], // Initialize modified_table in state
                'start_time' => microtime(true),


            ];

            // Calculate total rows for all tables (excluding options table for now)
            foreach ($valid_tables as $table) {
                if ($table !== $this->wpdb->options) {
                    $state['total_rows'] += (int) $this->wpdb->get_var("SELECT COUNT(*) FROM $table");
                }
            }
            // For options table, count all rows.
            if (in_array($this->wpdb->options, $valid_tables)) {
                $state['total_rows'] += (int) $this->wpdb->get_var("SELECT COUNT(*) FROM {$this->wpdb->options}");
            }
        } else {
            // Load existing report details and modified tables from state
            $state['current_table_index'] = $current_table_index;
            $this->report_details = $state['report_details'];
            $this->modified_table = $state['modified_table'];
        }

        $is_complete = false;
        $message = '';
        $reports = $state['reports'];
        $location = '';

        // Process tables in batches
        if ($state['current_table_index'] < count($state['tables_to_process'])) {
            $current_table = $state['tables_to_process'][$state['current_table_index']];
            $location = $current_table;

            // Ensure report_details for current_table is fully initialized
            if (!isset($this->report_details[$current_table])) {
                $this->report_details[$current_table] = ['columns' => [], 'time' => 0];
            }

            if ($current_table === $this->wpdb->options) {
                // Handle options table separately
                $this->case_insensitive = $case_insensitive;


                $table_start = microtime(true);
                $this->process_options_table($current_table, $search, $replace, $dry_run);

                // Accumulate report details and modified tables
                // Redundant isset checks removed as per plan
                $this->report_details[$current_table]['time'] += microtime(true) - $table_start;


                $option_value_report = $this->report_details[$current_table]['columns']['option_value'] ?? null;

                $occurrences = $option_value_report['occurrences'] ?? 0;
                $rows_modified = $option_value_report['rows_modified'] ?? 0;

                if (!in_array($current_table, $this->modified_table) && ($occurrences > 0 || $rows_modified > 0)) {
                    $this->modified_table[] = $current_table;
                }

                $reports[$current_table] = ($reports[$current_table] ?? 0) + ($this->report_details[$current_table]['columns']['option_value']['rows_modified'] ?? 0);
                $state['processed_rows'] += ($this->report_details[$current_table]['columns']['option_value']['rows_modified'] ?? 0);

                $message = sprintf(esc_html__('Processed options table %s.', 'surflab-search-replace'), $current_table);


                $state['current_table_index']++;
                $state['current_table_offset'] = 0; // Reset offset for next table
            } else {
                // Handle other tables
                $primary_key = '';

                $primary_key_info = $this->get_primary_key($current_table);
                $rows = [];

                if ($primary_key_info === false) {
                    // 4️⃣ No safe column found → skip table
                    $reports['skipped'][] = $this->table_skipped;
                } else {

                    $primary_key = $primary_key_info['name'];
                    $is_numeric = $primary_key_info['type'] === 'numeric';



                    $columns = $this->get_text_columns($current_table);

                    if (!$replace_guid) {
                        $columns = array_filter($columns, function ($col) {
                            return strtolower($col) !== 'guid';
                        });
                    }

                    // Wrap primary key and column names in backticks
                    $_primary_key = $this->sanitize_identifier($primary_key);
                    $_columns = array_map([$this, 'sanitize_identifier'], $columns);
                    $columns_sql = !empty($_columns) ? ', ' . implode(', ', $_columns) : '';

                    $rows = $this->wpdb->get_results(
                        "SELECT $_primary_key$columns_sql FROM {$current_table} LIMIT {$state['current_table_offset']}, $this->batch_size",
                        ARRAY_A
                    );
                }

                if (!empty($rows)) {
                    $this->case_insensitive = $case_insensitive;


                    $table_start = microtime(true);
                    $this->process_batch($current_table, $primary_key, $columns, $rows, $search, $replace, $dry_run, $is_numeric);

                    // Accumulate report details and modified tables

                    $this->report_details[$current_table]['time'] += microtime(true) - $table_start;



                    $hasOccurrences = false;
                    foreach ($this->report_details[$current_table]['columns'] as $col_name => $col_data) {
                        if ($col_data['occurrences'] > 0) {
                            $hasOccurrences = true;
                            break;
                        }
                    }
                    if ($hasOccurrences && !in_array($current_table, $this->modified_table)) {
                        $this->modified_table[] = $current_table;
                    }

                    $reports[$current_table] = ($reports[$current_table] ?? 0) + (count($rows)); // Count rows processed in this batch

                    self::log("BEFORE increment: current_table_offset={$state['current_table_offset']}, processed_rows={$state['processed_rows']}");

                    $state['processed_rows'] += count($rows);

                    $state['current_table_offset'] += $this->batch_size;


                    $message = sprintf(esc_html__('Processing table %s.', 'surflab-search-replace'), $current_table); // Simplified message


                } else {
                    // Current table finished, move to next
                    $message = sprintf(esc_html__('Finished table %s.', 'surflab-search-replace'), $current_table);

                    $state['current_table_index']++;
                    if ($state['current_table_index'] >= count($state['tables_to_process'])) {
                        $is_complete = true;
                    }

                    $state['current_table_offset'] = 0; // Reset offset for next table
                    self::log("ajax_process_replace - Finished table: {$current_table}.");
                }
            }
        } else {
            $is_complete = true;
            self::log("ajax_process_replace - All tables processed. Operation complete.");
        }

        $state['reports'] = $reports;
        $state['report_details'] = $this->report_details; // Save accumulated report details
        $state['modified_table'] = $this->modified_table; // Save accumulated modified tables
        // AFTER all state modifications

        $result = set_transient($transient_key, $state, HOUR_IN_SECONDS);
        if ($result === false) {
            self::log("ERROR: Failed to save transient for key: " . $transient_key);
        } else {
            self::log("SUCCESS: Transient saved for key: " . $transient_key);
        }





        if ($is_complete) {



            $elapsed_time = round(microtime(true) - $state['start_time'], 4);
            if (empty($reports['errors'])) {
                $msg = $dry_run
                    ? "Dry run completed in {$elapsed_time} seconds. No changes were made."
                    : "Operation completed in {$elapsed_time} seconds.";
                $reports['success'][] = $msg;
            }
            $reports['dry_run'] = $dry_run;
            $this->reports = $reports;
            $this->reports['details'] = $this->report_details; // Ensure final reports include accumulated details

            ob_start();
            require SURF_SR_PATH . "templates/surf-sr-report.php";
            $report_html = ob_get_clean();


            delete_transient($transient_key); // Clean up transient


            wp_send_json_success([
                'is_complete' => true,
                'report_html' => $report_html,

                'reports_data' => $this->reports,
                'message' => $msg,
                'current_table_index' => $state['current_table_index']
            ]);
        } else {

            self::log("offset before  wp send josn : " . $state['current_table_offset']);
            wp_send_json_success([
                'is_complete' => false,
                'offset' => $state['current_table_offset'],
                'total_rows' => $state['total_rows'],
                'processed_rows' => $state['processed_rows'],
                'message' => $message,
                'current_table' => $current_table, // Pass current_table
                'current_table_index' => $state['current_table_index'], // Pass current_table
            ]);
        }
    }
}
