<?php
/**
 * SummPress Settings Page
 *
 * This file contains the class responsible for creating and managing
 * the admin settings page for SummPress plugin.
 *
 * @package SummPress
 * @since 1.1.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * SummPressSettings Class
 *
 * Handles the admin settings page with tabbed interface for:
 * - General Settings
 * - Google Gemini API
 * - OpenAI API
 *
 * @since 1.1.0
 */
class SummPressSettings {

	/**
	 * The single instance of the class.
	 *
	 * @since 1.1.0
	 * @var SummPressSettings
	 */
	protected static $instance = null;

	/**
	 * Settings array.
	 *
	 * @since 1.1.0
	 * @var array
	 */
	private $settings = array();

	/**
	 * Main Settings Instance.
	 *
	 * Ensures only one instance of Settings is loaded or can be loaded.
	 *
	 * @since 1.1.0
	 * @static
	 * @return SummPress_Settings - Main instance.
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * @since 1.1.0
	 */
	public function __construct() {
		add_action( 'admin_menu', array( $this, 'add_settings_menu' ) );
		add_action( 'admin_init', array( $this, 'register_settings' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );

		// Add plugin action links.
		add_filter( 'plugin_action_links_' . plugin_basename(SUMMPRESS_PLUGIN_FILE), array( $this, 'add_plugin_action_links' ) );

		// Get settings from database.
		$this->settings = get_option( 'summpress_settings', array() );

	}

	/**
	 * Add settings menu in WordPress admin.
	 *
	 * @since 1.1.0
	 * @return void
	 */
	public function add_settings_menu() {
		$icon = SUMMPRESS_PLUGIN_URL. 'assets/icons/summpress.png';
		add_menu_page(__( 'SummPressAI Settings', 'summpressai' ), __( 'SummPressAI', 'summpressai' ), 'manage_options', 'summpressai', array( $this, 'render_settings_page' ), $icon);
	}

	/**
	 * Add plugin action links.
	 *
	 * @since 1.0.0
	 * @param array $links Plugin action links.
	 * @return array Modified plugin action links.
	 */
	public function add_plugin_action_links( $links ) {
		$plugin_links = array(
			'<a href="' . admin_url( 'admin.php?page=summpressai' ) . '">' . __( 'Settings', 'summpressai' ) . '</a>',
		);
		return array_merge( $plugin_links, $links );
	}

	/**
	 * Enqueue admin scripts and styles.
	 *
	 * @since 1.1.0
	 * @param string $hook The current admin page.
	 * @return void
	 */
	public function enqueue_admin_scripts() {


		// Add custom admin CSS
		wp_enqueue_style(
			'summpress-admin',
			SUMMPRESS_PLUGIN_URL . 'assets/css/admin.css',
			array(),
			SUMMPRESS_VERSION
		);

		// Add custom admin JS
		wp_enqueue_script(
			'summpress-admin',
			SUMMPRESS_PLUGIN_URL . 'assets/js/admin.js',
			array( 'jquery' ),
			SUMMPRESS_VERSION,
			true
		);
	}

	/**
	 * Register plugin settings.
	 *
	 * @since 1.1.0
	 * @return void
	 */
	public function register_settings() {
		// Register the setting.
		register_setting(
			'summpress_settings',
			'summpress_settings',
			array( $this, 'sanitize_settings' )
		);

		// General Settings Section.
		add_settings_section(
			'summpress_general_section',
			__( 'General Settings', 'summpressai' ),
			array( $this, 'render_general_section' ),
			'summpress_general'
		);

		add_settings_field(
			'extension_google_chrome_url',
			__( 'Google Chrome Extension URL', 'summpressai' ),
			array( $this, 'render_extension_google_chrome_url_field' ),
			'summpress_general',
			'summpress_general_section'
		);

		add_settings_field(
			'extension_endpoint',
			__( 'Browser Extension API Endpoint', 'summpressai' ),
			array( $this, 'render_extension_endpoint_field' ),
			'summpress_general',
			'summpress_general_section'
		);

		add_settings_field(
			'extension_token',
			__( 'Browser Extension API Token', 'summpressai' ),
			array( $this, 'render_extension_token_field' ),
			'summpress_general',
			'summpress_general_section'
		);

		add_settings_field(
			'summary_language',
			__( 'Summary Language', 'summpressai' ),
			array( $this, 'render_summary_language_field' ),
			'summpress_general',
			'summpress_general_section'
		);

		add_settings_field(
			'post_type',
			__( 'Post Type', 'summpressai' ),
			array( $this, 'render_post_types_field' ),
			'summpress_general',
			'summpress_general_section'
		);

		add_settings_field(
			'post_status',
			__( 'Default Post Status', 'summpressai' ),
			array( $this, 'render_post_status_field' ),
			'summpress_general',
			'summpress_general_section'
		);

		add_settings_field(
			'default_ai_service',
			__( 'Default AI Service', 'summpressai' ),
			array( $this, 'render_default_ai_service_field' ),
			'summpress_general',
			'summpress_general_section'
		);

		// Google Gemini API Section.
		add_settings_section(
			'summpress_gemini_section',
			__( 'Google Gemini API Settings', 'summpressai' ),
			array( $this, 'render_gemini_section' ),
			'summpress_gemini'
		);

		add_settings_field(
			'gemini_api_key',
			__( 'API Key', 'summpressai' ),
			array( $this, 'render_gemini_api_key_field' ),
			'summpress_gemini',
			'summpress_gemini_section'
		);

		add_settings_field(
			'gemini_model',
			__( 'Model Version', 'summpressai' ),
			array( $this, 'render_gemini_model_field' ),
			'summpress_gemini',
			'summpress_gemini_section'
		);

		// OpenAI Section.
		add_settings_section(
			'summpress_openai_section',
			__( 'OpenAI API Settings', 'summpressai' ),
			array( $this, 'render_openai_section' ),
			'summpress_openai'
		);

		add_settings_field(
			'openai_api_key',
			__( 'API Key', 'summpressai' ),
			array( $this, 'render_openai_api_key_field' ),
			'summpress_openai',
			'summpress_openai_section'
		);

		add_settings_field(
			'openai_model',
			__( 'Model Version', 'summpressai' ),
			array( $this, 'render_openai_model_field' ),
			'summpress_openai',
			'summpress_openai_section'
		);
	}

	/**
	 * Sanitize settings before saving.
	 *
	 * @since 1.1.0
	 * @param array $input The settings array to sanitize.
	 * @return array Sanitized settings.
	 */
	public function sanitize_settings( $input ) {
		if (
			! isset( $_POST['summpress_settings_nonce'] ) ||
			! wp_verify_nonce(
				sanitize_text_field( wp_unslash( $_POST['summpress_settings_nonce'] ) ),
				'summpress_settings_verify'
			)
		) {
			wp_die( esc_html__( 'Security check failed. Nonce verification failed.', 'summpressai' ) );
		}

		$existing = get_option( 'summpress_settings', array() );
		$sanitized = $existing;

		// General settings.
		if ( isset( $input['extension_token'] ) ) {
			$sanitized['extension_token'] = sanitize_text_field( $input['extension_token'] );
		}

		if ( isset( $input['summary_language'] ) ) {
			$sanitized['summary_language'] = sanitize_text_field( $input['summary_language'] );
		}

		if ( isset( $input['post_type'] ) ) {
			$sanitized['post_type'] = sanitize_text_field( $input['post_type'] );
		}

		if ( isset( $input['post_status'] ) ) {
			$sanitized['post_status'] = sanitize_text_field( $input['post_status'] );
		}

		if ( isset( $input['default_ai_service'] ) ) {
			$sanitized['default_ai_service'] = sanitize_text_field( $input['default_ai_service'] );
		}

		// Gemini API settings.
		if ( isset( $input['gemini_api_key'] ) ) {
			$sanitized['gemini_api_key'] = sanitize_text_field( $input['gemini_api_key'] );
		}

		if ( isset( $input['gemini_model'] ) ) {
			$sanitized['gemini_model'] = sanitize_text_field( $input['gemini_model'] );
		}

		// OpenAI API settings.
		if ( isset( $input['openai_api_key'] ) ) {
			$sanitized['openai_api_key'] = sanitize_text_field( $input['openai_api_key'] );
		}

		if ( isset( $input['openai_model'] ) ) {
			$sanitized['openai_model'] = sanitize_text_field( $input['openai_model'] );
		}

		return $sanitized;
	}


	/**
	 * Render the settings page.
	 *
	 * @since 1.1.0
	 * @return void
	 */
	public function render_settings_page() {

		$nonce = wp_create_nonce('summpress_tab_nonce');

		$active_tab = 'general';

		if (isset($_GET['_wpnonce']) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'summpress_tab_nonce' )) {
			$active_tab = isset($_GET['tab']) ? sanitize_text_field( wp_unslash( $_GET['tab'] ) ) : 'general';
		}

		$banner_url = plugins_url( 'assets/icons/summpress.png', plugin_dir_path( __FILE__ ) );
		?>
		<div class="wrap summpress">
			<h1><?php echo esc_html__( 'SummPressAI Settings', 'summpressai' ); ?></h1>

            <div class="notice-browser-extension-requires">
                ⚠️ <strong><?php esc_html_e('Attention:', 'summpressai'); ?></strong>
                <?php esc_html_e('SummPressAI plugin requires the', 'summpressai'); ?>
                <strong><a href="https://chromewebstore.google.com/detail/summpressai/dlamobieollhbdnkmdciooagnjagjngn" target="_blank"> <?php esc_html_e('Chrome Extension', 'summpressai'); ?></a></strong>.
                <?php esc_html_e('Install the extension to unlock full functionality.', 'summpressai'); ?>
            </div>

            <div class="summpress-banner">
                <div class="summpress-banner__logo">
                    <img src="<?php echo esc_url( $banner_url ); ?>" alt="<?php echo esc_html__( 'SummPressAI Pro', 'summpressai' ); ?>">
                </div>
                <div class="summpress-banner__text">
                    <div class="summpress-banner__title">SummPressAI Pro</div>
                    <div class="summpress-banner__subtitle">Coming soon with more features</div>
                </div>
            </div>


            <?php settings_errors(); ?>
            <h2 class="nav-tab-wrapper">
                <a href="?page=summpressai&tab=general&_wpnonce=<?php echo esc_attr($nonce); ?>" class="nav-tab <?php echo 'general' === $active_tab ? 'nav-tab-active' : ''; ?>">
					<?php esc_html_e( 'General', 'summpressai' ); ?>
                </a>
                <a href="?page=summpressai&tab=gemini&_wpnonce=<?php echo esc_attr($nonce); ?>" class="nav-tab <?php echo 'gemini' === $active_tab ? 'nav-tab-active' : ''; ?>">
					<?php esc_html_e( 'Google Gemini', 'summpressai' ); ?>
                </a>
                <a href="?page=summpressai&tab=openai&_wpnonce=<?php echo esc_attr($nonce); ?>" class="nav-tab <?php echo 'openai' === $active_tab ? 'nav-tab-active' : ''; ?>">
					<?php esc_html_e( 'OpenAI', 'summpressai' ); ?>
                </a>
            </h2>

			<form method="post" action="options.php">
				<?php
                wp_nonce_field( 'summpress_settings_verify', 'summpress_settings_nonce' );

				settings_fields( 'summpress_settings' );

				switch ( $active_tab ) {
					case 'gemini':
						do_settings_sections( 'summpress_gemini' );
						break;

					case 'openai':
						do_settings_sections( 'summpress_openai' );
						break;

					default: // 'general'
						do_settings_sections( 'summpress_general' );
						break;
				}

				submit_button();
				?>
			</form>
		</div>
		<?php
	}

	/**
	 * Render General section description.
	 *
	 * @since 1.1.0
	 * @return void
	 */
	public function render_general_section() {
		echo '<p>' . esc_html__( 'Configure general settings for SummPress.', 'summpressai' ) . '</p>';
	}


	/**
	 * Render extension google chrome url field.
	 *
	 * @since 1.1.0
	 * @return void
	 */
	public function render_extension_google_chrome_url_field() {
    ?>
        <span class="extension-google-chrome-url">
            <a target="_blank" href="https://chromewebstore.google.com/detail/summpressai/dlamobieollhbdnkmdciooagnjagjngn"><?php esc_html_e( 'Click to install the SummPressAI for Google Chrome', 'summpressai' ); ?></a>
        </span>

        <p class="description">
			<?php esc_html_e( 'Get the SummPressAI extension on the Chrome Web Store.', 'summpressai' ); ?>
        </p>

		<?php
	}


	/**
	 * Render extension endpoint field.
	 *
	 * @since 1.1.0
	 * @return void
	 */
	public function render_extension_endpoint_field() {
	?>
        <span class="extension-endpoint-url"><?php echo esc_url( home_url( '/wp-json/summpress-ai/v1/summarize' ) ); ?></span>

        <button type="button" class="button button-secondary summpress-copy-extension-endpoint-url">
			<?php esc_html_e( 'Copy', 'summpressai' ); ?>
        </button>

        <p class="description">
			<?php esc_html_e( 'Copy API Endpoint for browser extension.', 'summpressai' ); ?>
        </p>
        <p class="description">
			<?php esc_html_e( 'API Endpoint needs to be inserted in the settings of the browser extension SummPressAI.', 'summpressai' ); ?>
        </p>
		<?php
	}


	/**
	 * Render extension token field.
	 *
	 * @since 1.1.0
	 * @return void
	 */
	public function render_extension_token_field() {
		$value = isset( $this->settings['extension_token'] ) ? $this->settings['extension_token'] : '';
		?>
        <input type="password"
               id="summpress_extension_token"
               name="summpress_settings[extension_token]"
               value="<?php echo esc_attr( $value ); ?>"
               class="regular-text"
        />
        <button type="button" class="button button-secondary summpress-toggle-extension-token" data-target="summpress_extension_token">
			<?php esc_html_e( 'Show', 'summpressai' ); ?>
        </button>
        <button type="button" class="button button-secondary summpress-generate-extension-token" data-target="summpress_extension_token">
			<?php esc_html_e( 'Generate Token', 'summpressai' ); ?>
        </button>
        <p class="description">
			<?php esc_html_e( 'Generate API Token for browser extension.', 'summpressai' ); ?>
        </p>
        <p class="description">
			<?php esc_html_e( 'API Token needs to be inserted in the settings of the browser extension SummPressAI.', 'summpressai' ); ?>
        </p>
		<?php
	}

	/**
	 * Render summary language field.
	 *
	 * @since 1.1.0
	 * @return void
	 */
	public function render_summary_language_field() {
		$value = isset( $this->settings['summary_language'] ) ? $this->settings['summary_language'] : 'original';
		?>
		<select name="summpress_settings[summary_language]" id="summpress_summary_language">
			<option value="original" <?php selected( $value, 'original' ); ?>><?php esc_html_e( 'Original Language', 'summpressai' ); ?></option>
			<option value="english" <?php selected( $value, 'english' ); ?>><?php esc_html_e( 'English', 'summpressai' ); ?></option>
			<option value="ukrainian" <?php selected( $value, 'ukrainian' ); ?>><?php esc_html_e( 'Ukrainian', 'summpressai' ); ?></option>
			<option value="russian" <?php selected( $value, 'russian' ); ?>><?php esc_html_e( 'Russian', 'summpressai' ); ?></option>
			<option value="german" <?php selected( $value, 'german' ); ?>><?php esc_html_e( 'German', 'summpressai' ); ?></option>
			<option value="french" <?php selected( $value, 'french' ); ?>><?php esc_html_e( 'French', 'summpressai' ); ?></option>
			<option value="spanish" <?php selected( $value, 'spanish' ); ?>><?php esc_html_e( 'Spanish', 'summpressai' ); ?></option>
		</select>
		<p class="description">
			<?php esc_html_e( 'Choose the language for generated summaries.', 'summpressai' ); ?>
		</p>
		<?php
	}

	/**
	 * Render custom post type.
	 *
	 * @since 1.1.0
	 * @return void
	 */
	public function render_post_types_field() {
		$value = isset( $this->settings['post_type'] ) ? $this->settings['post_type'] : '';
        $all_post_types = $this->get_all_post_types()

		?>
        <select name="summpress_settings[post_type]" id="summpress_post_type">
            <option value="post"><?php esc_html_e( 'Posts', 'summpressai' ); ?></option>
            <?php foreach ($all_post_types as $post_type) : ?>
                <option value="<?php echo esc_attr( $post_type->name ); ?>" <?php selected( $value, $post_type->name ); ?>>
		            <?php echo esc_html( $post_type->label ); ?>
                </option>
            <?php endforeach; ?>
        </select>
        <p class="description">
			<?php esc_html_e( 'Custom Post Type for generated summary.', 'summpressai' ); ?>
        </p>
		<?php
	}

	/**
	 * Render post status field.
	 *
	 * @since 1.1.0
	 * @return void
	 */
	public function render_post_status_field() {
		$value = isset( $this->settings['post_status'] ) ? $this->settings['post_status'] : 'draft';
		?>
		<select name="summpress_settings[post_status]" id="summpress_post_status">
			<option value="draft" <?php selected( $value, 'draft' ); ?>><?php esc_html_e( 'Draft', 'summpressai' ); ?></option>
			<option value="publish" <?php selected( $value, 'publish' ); ?>><?php esc_html_e( 'Published', 'summpressai' ); ?></option>
			<option value="pending" <?php selected( $value, 'pending' ); ?>><?php esc_html_e( 'Pending Review', 'summpressai' ); ?></option>
		</select>
		<p class="description">
			<?php esc_html_e( 'Default status for generated summary posts.', 'summpressai' ); ?>
		</p>
		<?php
	}

	/**
	 * Render default AI service field.
	 *
	 * @since 1.1.0
	 * @return void
	 */
	public function render_default_ai_service_field() {
		$value = isset( $this->settings['default_ai_service'] ) ? $this->settings['default_ai_service'] : 'gemini';
		?>
		<select name="summpress_settings[default_ai_service]" id="summpress_default_ai_service">
			<option value="gemini" <?php selected( $value, 'gemini' ); ?>><?php esc_html_e( 'Google Gemini', 'summpressai' ); ?></option>
			<option value="openai" <?php selected( $value, 'openai' ); ?>><?php esc_html_e( 'OpenAI', 'summpressai' ); ?></option>
		</select>
		<p class="description">
			<?php esc_html_e( 'Choose the default AI service for generating summaries.', 'summpressai' ); ?>
		</p>
		<?php
	}

	/**
	 * Render Gemini section description.
	 *
	 * @since 1.1.0
	 * @return void
	 */
	public function render_gemini_section() {
		echo '<p>' . esc_html__( 'Configure Google Gemini API settings for generating summaries.', 'summpressai' ) . '</p>';
		echo '<p><a href="https://ai.google.dev/gemini" target="_blank" rel="noopener noreferrer">' . esc_html__( 'Get your Google Gemini API key →', 'summpressai' ) . '</a></p>';
	}

	/**
	 * Render Gemini API key field.
	 *
	 * @since 1.1.0
	 * @return void
	 */
	public function render_gemini_api_key_field() {
		$value = isset( $this->settings['gemini_api_key'] ) ? $this->settings['gemini_api_key'] : '';
		?>
		<input type="password"
		       id="summpress_gemini_api_key"
		       name="summpress_settings[gemini_api_key]"
		       value="<?php echo esc_attr( $value ); ?>"
		       class="regular-text"
		/>
		<button type="button" class="button button-secondary summpress-toggle-password" data-target="summpress_gemini_api_key">
			<?php esc_html_e( 'Show', 'summpressai' ); ?>
		</button>
		<p class="description">
			<?php esc_html_e( 'Enter your Google Gemini API key.', 'summpressai' ); ?>
		</p>
		<?php
	}

	/**
	 * Render Gemini model field.
	 *
	 * @since 1.1.0
	 * @return void
	 */
	public function render_gemini_model_field() {
		$value = isset( $this->settings['gemini_model'] ) ? $this->settings['gemini_model'] : 'gemini-2.0-flash';
		?>
		<select name="summpress_settings[gemini_model]" id="summpress_gemini_model">
			<option value="gemini-2.0-flash" <?php selected( $value, 'gemini-2.0-flash' ); ?>>
                <?php esc_html_e( 'Gemini 2.0 Flash', 'summpressai' ); ?>
            </option>
			<option disabled>
                <?php esc_html_e( 'Gemini 2.5 Flash  - PRO Version', 'summpressai' ); ?>
            </option>
            <option disabled>
                <?php esc_html_e( 'Gemini 2.5 Pro  - PRO Version', 'summpressai' ); ?>
            </option>
		</select>
		<p class="description">
			<?php esc_html_e( 'Select the Gemini model to use for summarization.', 'summpressai' ); ?>
		</p>
        <p class="description">
            <?php esc_html_e('The models', 'summpressai'); ?>
            <strong><?php esc_html_e('Gemini 2.5 Flash', 'summpressai'); ?></strong>
            <?php esc_html_e('and', 'summpressai'); ?>
            <strong><?php esc_html_e('Gemini 2.5 Pro', 'summpressai'); ?></strong>
            <?php esc_html_e('are available in the', 'summpressai'); ?>
            <strong><a target="_blank" href="https://vned.pro/summpressai/"> <?php esc_html_e('SummPressAI Pro', 'summpressai'); ?> </a></strong>
            <?php esc_html_e('version of the plugin.', 'summpressai'); ?>
        </p>
		<?php
	}

	/**
	 * Render OpenAI section description.
	 *
	 * @since 1.1.0
	 * @return void
	 */
	public function render_openai_section() {
		echo '<p>' . esc_html__( 'Configure OpenAI API settings for generating summaries.', 'summpressai' ) . '</p>';
		echo '<p><a href="https://platform.openai.com/signup" target="_blank" rel="noopener noreferrer">' . esc_html__( 'Get your OpenAI API key →', 'summpressai' ) . '</a></p>';
	}

	/**
	 * Render OpenAI API key field.
	 *
	 * @since 1.1.0
	 * @return void
	 */
	public function render_openai_api_key_field() {
		$value = isset( $this->settings['openai_api_key'] ) ? $this->settings['openai_api_key'] : '';
		?>
		<input type="password"
		       id="summpress_openai_api_key"
		       name="summpress_settings[openai_api_key]"
		       value="<?php echo esc_attr( $value ); ?>"
		       class="regular-text"
		/>
		<button type="button" class="button button-secondary summpress-toggle-password" data-target="summpress_openai_api_key">
			<?php esc_html_e( 'Show', 'summpressai' ); ?>
		</button>
		<p class="description">
			<?php esc_html_e( 'Enter your OpenAI API key.', 'summpressai' ); ?>
		</p>
		<?php
	}

	/**
	 * Render OpenAI model field.
	 *
	 * @since 1.1.0
	 * @return void
	 */
	public function render_openai_model_field() {
		$value = isset( $this->settings['openai_model'] ) ? $this->settings['openai_model'] : 'gpt-3.5-turbo';
		?>
		<select name="summpress_settings[openai_model]" id="summpress_openai_model">
            <option value="gpt-4" <?php selected( $value, 'gpt-4' ); ?>>
				<?php esc_html_e( 'GPT-4 (8K tokens)', 'summpressai' ); ?>
            </option>
            <option disabled>
				<?php esc_html_e( 'GPT-4 Turbo (128K tokens) - PRO Version', 'summpressai' ); ?>
            </option>
            <option disabled>
				<?php esc_html_e( 'GPT-4o (128K tokens) - PRO Version', 'summpressai' ); ?>
            </option>
		</select>
		<p class="description">
			<?php esc_html_e( 'Select the OpenAI model to use for summarization.', 'summpressai' ); ?>
		</p>
        <p class="description">
            <?php esc_html_e('The models', 'summpressai'); ?>
            <strong><?php esc_html_e('GPT-4 Turbo (128K tokens)', 'summpressai'); ?></strong>
            <?php esc_html_e('and', 'summpressai'); ?>
            <strong><?php esc_html_e('GPT-4o (128K tokens)', 'summpressai'); ?></strong>
            <?php esc_html_e('are available in the', 'summpressai'); ?>
            <strong><a target="_blank" href="https://vned.pro/summpressai/"><?php esc_html_e('SummPressAI Pro', 'summpressai'); ?> </a></strong>
            <?php esc_html_e('version of the plugin.', 'summpressai'); ?>
        </p>
		<?php
	}


    private function  get_all_post_types() {

	    $post_types = get_post_types(
		    [
			    '_builtin' => false,
			    'public'   => true,
			    'show_ui'  => true,
			    'capability_type' => 'post',
		    ],
		    'objects' // return only names, not full objects
	    );

	   return $post_types;
    }

}

/**
 * Returns the main instance of SummPress_Settings.
 *
 * @since 1.1.0
 * @return SummPressSettings
 */
function SummPressSettings() {
	return SummPressSettings::instance();
}

// Initialize the settings.
add_action( 'plugins_loaded', function () {
	SummPressSettings();
} );
