<?php
/**
 * Front-end: enqueue assets, render markup.
 *
 * @package SummaTap
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}


class SummaTap_Public {

	public function __construct() {
        add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_assets' ) );
		add_action( 'wp_footer', array( $this, 'render_button_markup' ) );
	}

	private function should_display() {
		if ( is_front_page() || is_home() ) return false;
		if ( ! is_singular() ) return false;

		$opts = summatap_get_options();
		$post_id = get_queried_object_id();

		// Excluded IDs.
		if ( $post_id && in_array( $post_id, (array) $opts['excluded_ids'], true ) ) {
			return false;
		}

		// Allowed post types.
		$pt = get_post_type( $post_id );
		if ( ! in_array( $pt, (array) $opts['post_types'], true ) ) {
			return false;
		}
		return true;
	}

	public function enqueue_assets() {
		if ( ! $this->should_display() ) return;

		wp_enqueue_style(
			'summatap-front',
			SUMMATAP_URL . 'public/css/summatap.css',
			array(),
			summatap_asset_version( SUMMATAP_PATH . 'public/css/summatap.css', SUMMATAP_VERSION )
		);
		wp_enqueue_script(
			'summatap-front',
			SUMMATAP_URL . 'public/js/summatap.js',
			array( 'jquery' ),
			summatap_asset_version( SUMMATAP_PATH . 'public/js/summatap.js', SUMMATAP_VERSION ),
			true
		);

		$opts   = summatap_get_options();
		$locale = get_locale();
		if ( 'auto' !== $opts['language'] ) $locale = $opts['language'];

		$label = $opts['custom_text'];
		if ( ! $label ) $label = summatap_default_label_for_locale( $locale );

		$post_id    = get_queried_object_id();
		$title      = get_the_title( $post_id );
		$meta_title = summatap_get_meta_title( $post_id );
		$heads      = summatap_extract_headings( $post_id );

        wp_localize_script( 'summatap-front', 'SummatapData', array(
			'buttonText' => $label,
			'position'   => $opts['position'],
			'offsetX'    => (int) $opts['offset_x'],
			'offsetY'    => (int) $opts['offset_y'],
            'revealRatio'=> 0.05,
			'chatUrls'   => array( 'https://chatgpt.com', 'https://chat.openai.com' ),
			'page'       => array(
				'url'       => get_permalink( $post_id ),
				'title'     => is_string( $title ) ? $title : '',
				'metaTitle' => is_string( $meta_title ) ? $meta_title : '',
				'postType'  => get_post_type( $post_id ),
				'h1'        => isset( $heads['h1'] ) ? $heads['h1'] : array(),
				'h2'        => isset( $heads['h2'] ) ? $heads['h2'] : array(),
				'h3'        => isset( $heads['h3'] ) ? $heads['h3'] : array(),
				'meta'      => isset( $heads['meta'] ) ? $heads['meta'] : array(),
			),
			'icon'       => SUMMATAP_URL . 'public/img/summatap-logo.svg',
			'relAttrs'   => 'nofollow noopener noreferrer',
		) );
	}

	public function render_button_markup() {
		if ( ! $this->should_display() ) return;
        $opts = summatap_get_options();
        $extra_class = is_string( $opts['button_class'] ) ? trim( $opts['button_class'] ) : '';
        $btn_classes = 'summatap-button' . ( $extra_class ? ' ' . $extra_class : '' );
        ?>
        <div id="summatap-container" class="summatap-container" aria-live="polite" aria-hidden="true">
            <a href="#" id="summatap-button" class="<?php echo esc_attr( $btn_classes ); ?>" role="button" aria-label="<?php echo esc_attr__( 'Open ChatGPT to summarize this page', 'summatap' ); ?>">
				<img class="summatap-icon" src="<?php echo esc_url( SUMMATAP_URL . 'public/img/summatap-logo.svg' ); ?>" alt="" aria-hidden="true" />
				<span class="summatap-text"></span>
			</a>
		</div>
		<?php
	}
}
