<?php
/**
 * Helpers for SummaTap.
 *
 * @package SummaTap
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Default options.
 *
 * @return array
 */
function summatap_default_options() {
	return array(
		'language'     => 'auto',
		'custom_text'  => '',
		'position'     => 'bottom-right',
		'offset_x'     => 20,
		'offset_y'     => 20,
		'post_types'   => array( 'post', 'page' ),
		'excluded_ids' => array(),
        'button_class' => '',
	);
}

/**
 * Merged options with defaults.
 */
function summatap_get_options() {
	$defaults = summatap_default_options();
	$opts     = get_option( 'summatap_options', array() );
	$merged   = wp_parse_args( $opts, $defaults );

	// Normalize.
	$merged['offset_x'] = (int) $merged['offset_x'];
	$merged['offset_y'] = (int) $merged['offset_y'];
	if ( ! is_array( $merged['post_types'] ) ) {
		$merged['post_types'] = array( 'post', 'page' );
	}
	if ( ! is_array( $merged['excluded_ids'] ) ) {
		$merged['excluded_ids'] = array();
	}
    if ( ! is_string( $merged['button_class'] ) ) {
        $merged['button_class'] = '';
    }
	return $merged;
}

// Telemetry-related helpers removed.

/** Asset version from filemtime (cache busting). */
function summatap_asset_version( $abs_path, $fallback ) {
	if ( file_exists( $abs_path ) ) {
		$mtime = @filemtime( $abs_path );
		if ( $mtime ) {
			return (string) $mtime;
		}
	}
	return (string) $fallback;
}

/** Sanitize CSV into unique positive integers. */
function summatap_sanitize_id_csv( $csv ) {
	$ids = array();
	if ( is_string( $csv ) && strlen( trim( $csv ) ) ) {
		$parts = preg_split( '/[\s,;]+/', $csv );
		foreach ( $parts as $p ) {
			$n = (int) trim( $p );
			if ( $n > 0 ) $ids[] = $n;
		}
	} elseif ( is_array( $csv ) ) {
		foreach ( $csv as $p ) {
			$n = (int) $p;
			if ( $n > 0 ) $ids[] = $n;
		}
	}
	return array_values( array_unique( $ids ) );
}

/** Fallback labels by locale. */
function summatap_locale_label_map() {
	return array(
		'en_US' => 'Summarize with AI',
		'es_ES' => 'Resumir con IA',
		'pt_PT' => 'Resumir com IA',
		'pt_BR' => 'Resumir com IA',
		'fr_FR' => 'Résumer avec l’IA',
		'de_DE' => 'Mit KI zusammenfassen',
		'it_IT' => 'Riassumi con IA',
		'nl_NL' => 'Samenvatten met AI',
		'ru_RU' => 'Резюмировать с ИИ',
		'zh_CN' => '用AI总结',
		'zh_TW' => '用AI摘要',
		'ja_JP' => 'AIで要約',
		'ko_KR' => 'AI로 요약',
		'ar'    => 'لخّص بالذكاء الاصطناعي',
		'hi_IN' => 'एआई से सारांश',
		'bn_BD' => 'এআই দিয়ে সারসংক্ষেপ',
		'tr_TR' => 'Yapay Zekâ ile Özetle',
		'vi_VN' => 'Tóm tắt bằng AI',
		'id_ID' => 'Ringkas dengan AI',
		'th_TH' => 'สรุปด้วย AI',
		'pl_PL' => 'Streść z AI',
	);
}

/** Default label for locale (with language-only fallback). */
function summatap_default_label_for_locale( $locale ) {
	$map = summatap_locale_label_map();
	if ( isset( $map[ $locale ] ) ) return $map[ $locale ];
	$lang = substr( $locale, 0, 2 );
	foreach ( $map as $k => $v ) {
		if ( substr( $k, 0, 2 ) === $lang ) return $v;
	}
	return 'Summarize with AI';
}

/**
 * Robust heading extraction for Gutenberg + Classic.
 * Returns array('h1' => [...], 'h2' => [...], 'h3' => [...], 'meta' => array('h1_omitted'=>N, 'h2_omitted'=>M, 'h3_omitted'=>K) )
 */
function summatap_extract_headings( $post_id ) {
    $h1 = array();
    $h2 = array();
    $h3 = array();

	// 1) From Gutenberg blocks.
	if ( function_exists( 'parse_blocks' ) ) {
		$post = get_post( $post_id );
		if ( $post ) {
			$blocks = parse_blocks( $post->post_content );
            foreach ( $blocks as $b ) {
                summatap_collect_heading_from_block( $b, $h1, $h2, $h3 );
            }
		}
	}

	// 2) Fallback from rendered content (regex).
	$content = get_post_field( 'post_content', $post_id );
	$html    = apply_filters( 'the_content', $content );

	if ( preg_match_all( '/<h1[^>]*>(.*?)<\/h1>/is', $html, $m1 ) ) {
		foreach ( $m1[1] as $m ) {
			$txt = summatap_clean_heading_text( $m );
			if ( $txt !== '' ) $h1[] = $txt;
		}
	}
    if ( preg_match_all( '/<h2[^>]*>(.*?)<\/h2>/is', $html, $m2 ) ) {
		foreach ( $m2[1] as $m ) {
			$txt = summatap_clean_heading_text( $m );
			if ( $txt !== '' ) $h2[] = $txt;
		}
	}
    if ( preg_match_all( '/<h3[^>]*>(.*?)<\/h3>/is', $html, $m3 ) ) {
        foreach ( $m3[1] as $m ) {
            $txt = summatap_clean_heading_text( $m );
            if ( $txt !== '' ) $h3[] = $txt;
        }
    }

	// Deduplicate while preserving order.
    $h1 = array_values( array_unique( $h1 ) );
    $h2 = array_values( array_unique( $h2 ) );
    $h3 = array_values( array_unique( $h3 ) );

	// Trim each heading to 90 chars.
    $h1 = array_map( 'summatap_trim_heading', $h1 );
    $h2 = array_map( 'summatap_trim_heading', $h2 );
    $h3 = array_map( 'summatap_trim_heading', $h3 );

    // Do not omit headings: include all found H1/H2/H3 (still trimmed individually above).
    $h1_omitted = 0;
    $h2_omitted = 0;
    $h3_omitted = 0;

	return array(
        'h1'   => $h1,
        'h2'   => $h2,
        'h3'   => $h3,
        'meta' => array( 'h1_omitted' => $h1_omitted, 'h2_omitted' => $h2_omitted, 'h3_omitted' => $h3_omitted ),
	);
}

/** Recursively collect heading data from a Gutenberg block. */
function summatap_collect_heading_from_block( $block, &$h1, &$h2, &$h3 ) {
	if ( ! is_array( $block ) ) return;
	if ( isset( $block['blockName'] ) && $block['blockName'] === 'core/heading' ) {
		$attrs = isset( $block['attrs'] ) && is_array( $block['attrs'] ) ? $block['attrs'] : array();
		$level = isset( $attrs['level'] ) ? (int) $attrs['level'] : 2;
		$inner = '';
		if ( ! empty( $block['innerHTML'] ) ) {
			$inner = $block['innerHTML'];
		} elseif ( ! empty( $block['innerContent'] ) && is_array( $block['innerContent'] ) ) {
			$inner = implode( '', $block['innerContent'] );
		}
		$txt = summatap_clean_heading_text( $inner );
		if ( $txt !== '' ) {
			if ( 1 === $level ) $h1[] = $txt;
			if ( 2 === $level ) $h2[] = $txt;
            if ( 3 === $level ) $h3[] = $txt;
		}
	}
	// Recurse into innerBlocks.
	if ( isset( $block['innerBlocks'] ) && is_array( $block['innerBlocks'] ) ) {
		foreach ( $block['innerBlocks'] as $child ) {
            summatap_collect_heading_from_block( $child, $h1, $h2, $h3 );
		}
	}
}

/** Clean heading text (strip tags/entities/whitespace). */
function summatap_clean_heading_text( $html_snippet ) {
	$txt = wp_strip_all_tags( $html_snippet );
	$txt = html_entity_decode( $txt, ENT_QUOTES | ENT_HTML5, 'UTF-8' );
	$txt = trim( $txt );
	return $txt;
}

/** Trim helper with multibyte safety. */
function summatap_trim_heading( $s, $max = 90 ) {
	if ( function_exists( 'mb_strlen' ) && function_exists( 'mb_substr' ) ) {
		return ( mb_strlen( $s, 'UTF-8' ) > $max ) ? ( mb_substr( $s, 0, $max, 'UTF-8' ) . '…' ) : $s;
	}
	return ( strlen( $s ) > $max ) ? ( substr( $s, 0, $max ) . '…' ) : $s;
}

/** Get meta title from common SEO plugins or fallback to post title. */
function summatap_get_meta_title( $post_id ) {
	$keys = array( '_yoast_wpseo_title', 'rank_math_title', '_seopress_titles_title', '_aioseo_title' );
	foreach ( $keys as $k ) {
		$val = get_post_meta( $post_id, $k, true );
		if ( is_string( $val ) && strlen( trim( $val ) ) ) {
			return wp_strip_all_tags( $val );
		}
	}
	$t = get_the_title( $post_id );
	return is_string( $t ) ? $t : '';
}

// Endpoint helper removed with telemetry.
