<?php
/**
 * Admin UI: settings (Basic/Advanced), global POST+nonce notice, live preview.
 *
 * @package SummaTap
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class SummaTap_Admin {

	public function __construct() {
		add_action( 'admin_menu', array( $this, 'register_menu' ) );
		add_action( 'admin_init', array( $this, 'register_settings' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_assets' ) );
	}

	public function register_menu() {
		add_menu_page(
			__( 'SummaTap', 'summatap' ),
			__( 'SummaTap', 'summatap' ),
			'manage_options',
			'summatap',
			array( $this, 'render_settings_page' ),
			'dashicons-format-chat', // chat bubble
			58
		);
	}

	public function register_settings() {
		register_setting( 'summatap_options_group', 'summatap_options', array( $this, 'sanitize_options' ) );

		add_settings_section( 'summatap_basic', __( 'Basic', 'summatap' ), '__return_false', 'summatap' );
		add_settings_section( 'summatap_advanced', __( 'Advanced', 'summatap' ), '__return_false', 'summatap' );

		add_settings_field( 'language', __( 'Button language', 'summatap' ), array( $this, 'field_language' ), 'summatap', 'summatap_basic' );
		add_settings_field( 'custom_text', __( 'Custom button text', 'summatap' ), array( $this, 'field_custom_text' ), 'summatap', 'summatap_basic' );
		add_settings_field( 'position', __( 'Button position', 'summatap' ), array( $this, 'field_position' ), 'summatap', 'summatap_basic' );
		add_settings_field( 'offsets', __( 'Offsets (px)', 'summatap' ), array( $this, 'field_offsets' ), 'summatap', 'summatap_basic' );
		add_settings_field( 'post_types', __( 'Post types', 'summatap' ), array( $this, 'field_post_types' ), 'summatap', 'summatap_basic' );

		// Advanced.
		add_settings_field( 'excluded_ids', __( 'Exclude IDs (posts/pages, CSV)', 'summatap' ), array( $this, 'field_excluded_ids' ), 'summatap', 'summatap_advanced' );
		add_settings_field( 'button_class', __( 'Custom CSS class for button', 'summatap' ), array( $this, 'field_button_class' ), 'summatap', 'summatap_advanced' );
	}

	public function sanitize_options( $input ) {
		$prev = summatap_get_options();
		$opts = $prev;

		if ( ! is_array( $input ) ) {
			add_settings_error( 'summatap', 'bad_input', __( 'Invalid settings payload.', 'summatap' ) );
			return $prev;
		}

		// Language & custom text.
		$opts['language'] = isset( $input['language'] ) ? sanitize_text_field( $input['language'] ) : $prev['language'];
		$opts['custom_text'] = isset( $input['custom_text'] ) ? sanitize_text_field( $input['custom_text'] ) : $prev['custom_text'];

		// Position.
		$valid_positions = array( 'top-left', 'top-right', 'bottom-right', 'bottom-left' );
		if ( isset( $input['position'] ) ) {
			$p = sanitize_text_field( $input['position'] );
			$opts['position'] = in_array( $p, $valid_positions, true ) ? $p : $prev['position'];
		}

		// Offsets (clamp 0..120).
		foreach ( array( 'offset_x', 'offset_y' ) as $k ) {
			if ( isset( $input[ $k ] ) ) {
				$v = (int) $input[ $k ];
				if ( $v < 0 ) $v = 0;
				if ( $v > 120 ) $v = 120;
				$opts[ $k ] = $v;
			}
		}

		// Post types (only public).
		$public = get_post_types( array( 'public' => true ), 'names' );
		if ( isset( $input['post_types'] ) && is_array( $input['post_types'] ) ) {
			$chosen = array();
			foreach ( $input['post_types'] as $pt ) {
				$pt = sanitize_text_field( $pt );
				if ( in_array( $pt, $public, true ) ) $chosen[] = $pt;
			}
			$opts['post_types'] = ! empty( $chosen ) ? $chosen : $prev['post_types'];
		}

		// Excluded IDs.
		if ( isset( $input['excluded_ids'] ) ) {
			$opts['excluded_ids'] = summatap_sanitize_id_csv( $input['excluded_ids'] );
		}

		// Custom button CSS class (single class name, safe characters only).
		if ( isset( $input['button_class'] ) ) {
			$raw = sanitize_text_field( $input['button_class'] );
			// Allow only letters, numbers, hyphens, underscores, and spaces (split take first class).
			$raw = trim( preg_replace( '/[^A-Za-z0-9_\- ]+/', '', $raw ) );
			// If multiple classes separated by spaces, keep all (user might want multiple) but clamp length
			$raw = substr( $raw, 0, 100 );
			$opts['button_class'] = $raw;
		}

		return $opts;
	}

	public function enqueue_assets( $hook ) {
		if ( $hook !== 'toplevel_page_summatap' ) return;
		wp_enqueue_style(
			'summatap-admin',
			SUMMATAP_URL . 'admin/css/summatap-admin.css',
			array(),
			summatap_asset_version( SUMMATAP_PATH . 'admin/css/summatap-admin.css', SUMMATAP_VERSION )
		);
		wp_enqueue_script(
			'summatap-admin',
			SUMMATAP_URL . 'admin/js/summatap-admin.js',
			array( 'jquery' ),
			summatap_asset_version( SUMMATAP_PATH . 'admin/js/summatap-admin.js', SUMMATAP_VERSION ),
			true
		);
	}

	public function field_language() {
		$opts = summatap_get_options();
		$val  = $opts['language'];
		$locales = array(
			'auto' => __( 'Auto (based on site language)', 'summatap' ),
			'en_US' => 'English (US)',
			'es_ES' => 'Spanish (ES)',
			'pt_PT' => 'Portuguese (PT)',
			'pt_BR' => 'Portuguese (BR)',
			'fr_FR' => 'French',
			'de_DE' => 'German',
			'it_IT' => 'Italian',
			'nl_NL' => 'Dutch',
			'ru_RU' => 'Russian',
			'zh_CN' => 'Chinese (Simplified)',
			'zh_TW' => 'Chinese (Traditional)',
			'ja_JP' => 'Japanese',
			'ko_KR' => 'Korean',
			'ar'    => 'Arabic',
			'hi_IN' => 'Hindi',
			'bn_BD' => 'Bengali',
			'tr_TR' => 'Turkish',
			'vi_VN' => 'Vietnamese',
			'id_ID' => 'Indonesian',
			'th_TH' => 'Thai',
			'pl_PL' => 'Polish',
		);
		echo '<select name="summatap_options[language]">';
		foreach ( $locales as $k => $label ) {
			printf( '<option value="%s"%s>%s</option>',
				esc_attr( $k ),
				selected( $val, $k, false ),
				esc_html( $label )
			);
		}
		echo '</select>';
	}

	public function field_custom_text() {
		$opts = summatap_get_options();
		printf(
			'<input type="text" class="regular-text" name="summatap_options[custom_text]" value="%s" placeholder="%s" />',
			esc_attr( $opts['custom_text'] ),
			esc_attr__( 'Leave empty to auto-translate', 'summatap' )
		);
	}

	public function field_position() {
		$opts = summatap_get_options();
		$positions = array(
			'top-left' => __( 'Top left', 'summatap' ),
			'top-right' => __( 'Top right', 'summatap' ),
			'bottom-right' => __( 'Bottom right', 'summatap' ),
			'bottom-left' => __( 'Bottom left', 'summatap' ),
		);
		foreach ( $positions as $k => $label ) {
			printf(
				'<label><input type="radio" name="summatap_options[position]" value="%s" %s /> %s</label><br />',
				esc_attr( $k ),
				checked( $opts['position'], $k, false ),
				esc_html( $label )
			);
		}
	}

	public function field_offsets() {
		$opts = summatap_get_options();
		?>
		<div class="summatap-offsets">
			<label><?php esc_html_e( 'Horizontal offset (px)', 'summatap' ); ?></label>
			<input type="range" min="0" max="120" step="1" name="summatap_options[offset_x]" value="<?php echo esc_attr( (int) $opts['offset_x'] ); ?>" class="summatap-offset-x" />
			<span class="summatap-offset-x-val"><?php echo esc_html( (int) $opts['offset_x'] ); ?></span>
			<br/>
			<label><?php esc_html_e( 'Vertical offset (px)', 'summatap' ); ?></label>
			<input type="range" min="0" max="120" step="1" name="summatap_options[offset_y]" value="<?php echo esc_attr( (int) $opts['offset_y'] ); ?>" class="summatap-offset-y" />
			<span class="summatap-offset-y-val"><?php echo esc_html( (int) $opts['offset_y'] ); ?></span>
		</div>
		<div class="summatap-preview-wrap">
			<h3><?php esc_html_e( 'Live Preview', 'summatap' ); ?></h3>
			<div id="summatap-preview-canvas" data-position="<?php echo esc_attr( $opts['position'] ); ?>" data-x="<?php echo esc_attr( (int) $opts['offset_x'] ); ?>" data-y="<?php echo esc_attr( (int) $opts['offset_y'] ); ?>">
				<a href="#" class="summatap-preview-button" aria-label="<?php esc_attr_e( 'Preview button', 'summatap' ); ?>">
					<img class="summatap-preview-icon" src="<?php echo esc_url( SUMMATAP_URL . 'public/img/summatap-logo.svg' ); ?>" alt="" />
					<span class="summatap-preview-text"><?php echo esc_html( summatap_admin_preview_text() ); ?></span>
				</a>
			</div>
		</div>
		<?php
	}

	public function field_post_types() {
		$opts   = summatap_get_options();
		$public = get_post_types( array( 'public' => true ), 'objects' );
		foreach ( $public as $pt => $obj ) {
			printf(
				'<label><input type="checkbox" name="summatap_options[post_types][]" value="%s" %s /> %s</label><br/>',
				esc_attr( $pt ),
				checked( in_array( $pt, (array) $opts['post_types'], true ), true, false ),
				esc_html( $obj->labels->name )
			);
		}
	}



	public function field_excluded_ids() {
		$opts = summatap_get_options();
		$val  = is_array( $opts['excluded_ids'] ) ? implode( ',', $opts['excluded_ids'] ) : '';
		printf(
			'<textarea name="summatap_options[excluded_ids]" rows="3" class="large-text" placeholder="%s">%s</textarea>',
			esc_attr__( 'e.g. 12,45,99', 'summatap' ),
			esc_textarea( $val )
		);
		echo '<p class="description">' . esc_html__( 'CSV of post/page IDs where the floating button will be hidden.', 'summatap' ) . '</p>';
	}

	public function field_button_class() {
		$opts = summatap_get_options();
		printf(
			'<input type="text" class="regular-text" name="summatap_options[button_class]" value="%s" placeholder="%s" />',
			esc_attr( $opts['button_class'] ),
			esc_attr__( 'Optional: additional CSS class(es) to apply to the button', 'summatap' )
		);
		echo '<p class="description">' . esc_html__( 'Example: my-custom-btn or multiple classes separated by spaces.', 'summatap' ) . '</p>';
	}

	public function render_settings_page() {
		?>
		<div class="wrap">
			<h1><?php esc_html_e( 'SummaTap Settings', 'summatap' ); ?></h1>
			<form method="post" action="options.php" id="summatap-settings-form">
				<?php
					settings_fields( 'summatap_options_group' );
					do_settings_sections( 'summatap' );
					submit_button();
				?>
			</form>
		</div>
		<?php
	}


}

/** Admin preview text helper. */
function summatap_admin_preview_text() {
	$opts = summatap_get_options();
	$text = $opts['custom_text'];
	if ( ! $text ) {
		$locale = get_locale();
		if ( 'auto' !== $opts['language'] ) $locale = $opts['language'];
		$text = summatap_default_label_for_locale( $locale );
	}
	return $text;
}
