<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://wpgenie.org
 * @since      1.0.0
 *
 * @package    Subscription_Gifts_For_Woocommerce
 * @subpackage Subscription_Gifts_For_Woocommerce/admin
 */
/**
 * The admin-specific functionality of the plugin.
 *
 * @package    Subscription_Gifts_For_Woocommerce
 * @subpackage Subscription_Gifts_For_Woocommerce/admin
 * @author     wpgenie <info@wpgenie.org>
 */
class Subscription_Gifts_For_Woocommerce_Admin {
    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param      string $plugin_name       The name of this plugin.
     * @param      string $version    The version of this plugin.
     */
    public function __construct( $plugin_name, $version ) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
    }

    /**
     * Register the stylesheets for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_styles() {
        $screen = get_current_screen();
        if ( !$screen || 'product' !== $screen->id ) {
            return;
        }
        wp_enqueue_style(
            $this->plugin_name,
            plugin_dir_url( __FILE__ ) . 'css/subscription-gifts-for-woocommerce-admin.css',
            array(),
            $this->version,
            'all'
        );
    }

    /**
     * Register the JavaScript for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_scripts() {
        $screen = get_current_screen();
        if ( !$screen || 'product' !== $screen->id ) {
            return;
        }
        wp_register_script(
            $this->plugin_name,
            plugin_dir_url( __FILE__ ) . 'js/subscription-gifts-for-woocommerce-admin.js',
            array('jquery'),
            $this->version,
            false
        );
        $params = array(
            'ajax_url ' => admin_url( 'admin-ajax.php' ),
            'nonce'     => wp_create_nonce( 'sgfw-search' ),
        );
        wp_localize_script( 'sgfwAdmin', $this->plugin_name, $params );
        wp_enqueue_script( $this->plugin_name );
    }

    /**
     * Add meta box for subscription product
     *
     * @return void
     */
    public function add_meta_box() {
        global $post;
        if ( !class_exists( 'WC_Subscriptions_Product' ) ) {
            return;
        }
        $screen = get_current_screen();
        if ( !$screen || 'product' !== $screen->id ) {
            return;
        }
        $is_new_product = 'add' === $screen->action;
        $is_subscription = false;
        if ( $is_new_product ) {
            $is_subscription = true;
        } elseif ( isset( $post->ID ) && $post->ID > 0 ) {
            $is_subscription = WC_Subscriptions_Product::is_subscription( $post->ID );
        }
        if ( $is_subscription ) {
            add_meta_box(
                'subscription-gifts-for-woocommerce',
                __( 'Subscription Free Gifts', 'subscription-gifts-for-woocommerce' ),
                array($this, 'render_meta_box'),
                'product',
                'advanced',
                'default'
            );
        }
    }

    /**
     * Render meta box for subscription product
     *
     * @param  mixed $post post ovject.
     * @return void
     */
    public function render_meta_box( $post ) {
        wp_nonce_field( 'save_subscription_gift_products_data', 'custom_linked_products_nonce' );
        $linked_products = get_post_meta( $post->ID, '_subscription_gift_products', true );
        if ( !is_array( $linked_products ) ) {
            $linked_products = array();
        }
        ?>
	<div id="custom_linked_products_fields" class="woocommerce_options_panel">
		<p class="description">
		<?php 
        esc_html_e( 'Add products that will be associated with this subscription, defining the quantity and application time (On Start / On Renewal)...', 'subscription-gifts-for-woocommerce' );
        ?>
		</p>
		<table class="widefat wc_input_table sortable" cellspacing="0">
			<thead>
				<tr>
					<th class="sort">&nbsp;</th>
					<th><?php 
        esc_html_e( 'Products', 'subscription-gifts-for-woocommerce' );
        ?></th>
					<?php 
        ?>
					<th><?php 
        esc_html_e( 'Application', 'subscription-gifts-for-woocommerce' );
        ?></th>
					<th class="remove">&nbsp;</th>
				</tr>
			</thead>
			<tbody class="custom-linked-products-container">
			<?php 
        if ( !empty( $linked_products ) ) {
            foreach ( $linked_products as $index => $item ) {
                $this->display_subscription_gift_product_row( $index, $item );
            }
        } else {
            $this->display_subscription_gift_product_row( 0 );
        }
        ?>
			</tbody>
			<tfoot>
				<tr>
					<th colspan="5">
						<button type="button" class="button button-primary add_custom_linked_product_row"><?php 
        esc_html_e( 'Add Product', 'subscription-gifts-for-woocommerce' );
        ?></button>
					</th>
				</tr>
			</tfoot>
		</table>
	</div>
	<script type="text/template" id="tmpl-custom-linked-product-row">
		<?php 
        $this->display_subscription_gift_product_row( '__i__' );
        ?>
	</script>

		<?php 
    }

    /**
     * Display subscription gift product row
     *
     * @param  mixed $index index.
     * @param  mixed $item item.
     * @return void
     */
    public function display_subscription_gift_product_row( $index, $item = array() ) {
        $product_id = ( isset( $item['product_id'] ) ? absint( $item['product_id'] ) : '' );
        $qty = ( isset( $item['qty'] ) ? wc_stock_amount( $item['qty'] ) : 1 );
        $timing = ( isset( $item['timing'] ) ? sanitize_text_field( $item['timing'] ) : 'on_start' );
        $product_title = '';
        if ( $product_id ) {
            $product = wc_get_product( $product_id );
            $product_title = ( $product ? $product->get_formatted_name() : '' );
        }
        ?>
			<tr class="custom-linked-product-row">
				<td class="sort ui-sortable-handle"></td>
				<td>
					<select class="wc-product-search" style="width: 100%;" name="_subscription_gift_products[<?php 
        echo esc_html( $index );
        ?>][product_id]" data-placeholder="<?php 
        esc_attr_e( 'Search products&hellip;', 'subscription-gifts-for-woocommerce' );
        ?>" data-action="woocommerce_json_search_products_and_variations" data-allow_clear="true">
					<?php 
        if ( $product_id ) {
            ?>
							<option value="<?php 
            echo esc_attr( $product_id );
            ?>" selected="selected"><?php 
            echo wp_kses_post( $product_title );
            ?></option>
						<?php 
        }
        ?>
					</select>
				</td>
				<?php 
        ?>
				<td>
					<select name="_subscription_gift_products[<?php 
        echo esc_html( $index );
        ?>][timing]">
						<?php 
        ?>
						<option value="on_start" <?php 
        selected( $timing, 'on_start' );
        ?>><?php 
        esc_html_e( 'On Start', 'subscription-gifts-for-woocommerce' );
        ?></option>
						<?php 
        ?>
					</select>
				</td>
				<td class="remove">
					<a href="#" class="delete_custom_linked_product_row"><?php 
        esc_html_e( 'Remove', 'subscription-gifts-for-woocommerce' );
        ?></a>
				</td>
			</tr>
			<?php 
    }

    /**
     * Save subscription gift products data
     *
     * @param  mixed $post_id post id.
     * @return void
     */
    public function save_subscription_gift_products_data( $post_id ) {
        if ( !isset( $_POST['custom_linked_products_nonce'] ) || !wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['custom_linked_products_nonce'] ) ), 'save_subscription_gift_products_data' ) ) {
            return;
        }
        if ( !current_user_can( 'edit_product', $post_id ) ) {
            return;
        }
        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
            return;
        }
        if ( isset( $_POST['_subscription_gift_products'] ) && is_array( $_POST['_subscription_gift_products'] ) ) {
            $linked_products = array();
            $sanitized_products = map_deep( wp_unslash( $_POST['_subscription_gift_products'] ), 'sanitize_text_field' );
            foreach ( $sanitized_products as $item ) {
                $product_id = ( isset( $item['product_id'] ) ? absint( $item['product_id'] ) : 0 );
                $qty = ( isset( $item['qty'] ) ? wc_stock_amount( $item['qty'] ) : 1 );
                $timing = ( isset( $item['timing'] ) && in_array( $item['timing'], array('on_start', 'on_renewal'), true ) ? sanitize_text_field( $item['timing'] ) : 'on_start' );
                // code...
                if ( $product_id > 0 ) {
                    $linked_products[] = array(
                        'product_id' => $product_id,
                        'qty'        => $qty,
                        'timing'     => $timing,
                    );
                }
            }
            update_post_meta( $post_id, '_subscription_gift_products', $linked_products );
        } else {
            delete_post_meta( $post_id, '_subscription_gift_products' );
        }
    }

    /**
     * Add free product to renewal order
     *
     * @param  mixed $renewal_order order object.
     * @param  mixed $subscription subscription.
     * @return object
     */
    public function add_free_product_to_renewal_order( $renewal_order, $subscription ) {
        foreach ( $subscription->get_items() as $item ) {
            $product_id = $item->get_product_id();
            $linked_products_data = get_post_meta( $product_id, '_subscription_gift_products', true );
            if ( empty( $linked_products_data ) || !is_array( $linked_products_data ) ) {
                continue;
            }
            foreach ( $linked_products_data as $item_data ) {
                $linked_product_id = absint( $item_data['product_id'] ?? 0 );
                $linked_qty = absint( $item_data['qty'] ?? 1 );
                $timing_setting = sanitize_text_field( $item_data['timing'] ?? 'on_start' );
                if ( $linked_product_id > 0 && $linked_qty > 0 && ('on_renewal' === $timing_setting || 'on_both' === $timing_setting) ) {
                    $item_id = $renewal_order->add_product( wc_get_product( $linked_product_id ), $linked_qty, array(
                        'subtotal' => 0.0,
                        'total'    => 0.0,
                    ) );
                    if ( $item_id && !is_wp_error( $item_id ) ) {
                        $order_item = $renewal_order->get_item( $item_id );
                        if ( $order_item ) {
                            $order_item->add_meta_data( __( 'Note', 'subscription-gifts-for-woocommerce' ), __( 'Free from subscription', 'subscription-gifts-for-woocommerce' ), true );
                            $order_item->add_meta_data( '_sgfw', '1' );
                            $order_item->set_subtotal( 0.0 );
                            $order_item->set_total( 0.0 );
                            $order_item->save();
                        }
                    }
                }
            }
        }
        $renewal_order->calculate_totals();
        return $renewal_order;
    }

    /**
     * Add free product to order before payment
     *
     * @param  object $order order object.
     * @return void
     */
    public function add_free_product_to_order_before_payment( $order ) {
        $contains_subscription = false;
        foreach ( $order->get_items() as $item ) {
            $product = $item->get_product();
            if ( $product && $product->is_type( array('subscription', 'variable-subscription', 'subscription_variation') ) ) {
                $contains_subscription = true;
                break;
            }
        }
        if ( !$contains_subscription ) {
            return;
        }
        if ( $order->get_meta( 'subscription_gifts_for_woocommerce_proccesed' ) ) {
            return;
        }
        $order->update_meta_data( 'subscription_gifts_for_woocommerce_proccesed', time() );
        $order->save();
        $products_added = false;
        foreach ( $order->get_items() as $order_item ) {
            $product_id = $order_item->get_product_id();
            $linked_products_data = get_post_meta( $product_id, '_subscription_gift_products', true );
            if ( empty( $linked_products_data ) || !is_array( $linked_products_data ) ) {
                continue;
            }
            foreach ( $linked_products_data as $item_data ) {
                $linked_product_id = absint( $item_data['product_id'] ?? 0 );
                $linked_qty = absint( $item_data['qty'] ?? 1 );
                $timing_setting = sanitize_text_field( $item_data['timing'] ?? 'on_start' );
                if ( $linked_product_id > 0 && $linked_qty > 0 && ('on_start' === $timing_setting || 'on_both' === $timing_setting) ) {
                    $product_to_add = wc_get_product( $linked_product_id );
                    if ( !$product_to_add ) {
                        continue;
                    }
                    $is_already_added = false;
                    foreach ( $order->get_items() as $existing_item ) {
                        if ( $existing_item->get_product_id() === $linked_product_id ) {
                            $is_already_added = true;
                            break;
                        }
                    }
                    if ( $is_already_added ) {
                        continue;
                    }
                    $item_id = $order->add_product( $product_to_add, $linked_qty, array(
                        'subtotal' => 0.0,
                        'total'    => 0.0,
                    ) );
                    if ( $item_id && !is_wp_error( $item_id ) ) {
                        $order_item = $order->get_item( $item_id );
                        if ( $order_item ) {
                            $order_item->set_subtotal( 0.0 );
                            $order_item->set_total( 0.0 );
                            $order_item->add_meta_data( __( 'Note', 'subscription-gifts-for-woocommerce' ), __( 'Free from subscription', 'subscription-gifts-for-woocommerce' ), true );
                            $order_item->add_meta_data( '_sgfw', '1', true );
                            $order_item->save();
                            $products_added = true;
                        }
                    }
                }
            }
        }
        if ( $products_added ) {
            $order->calculate_totals();
        }
        $order->save();
    }

    public function hidden_order_itemmeta( $meta_keys ) {
        $meta_keys[] = '_sgfw';
        return $meta_keys;
    }

}
