<?php
/**
 * Plugin Name: StreamNexus.io Embed Videos
 * Plugin URI: https://www.streamnexus.io/
 * Description: Embed StreamNexus videos with a simple shortcode. HTML5 player with adaptive HLS streaming for smooth playback on any device.
 * Version: 2.2.2
 * Author: StreamNexus.io
 * Text Domain: streamnexus-io-embed-videos
 * License: GPLv3
 * License URI: https://www.gnu.org/licenses/gpl-3.0.html
*/

if ( ! defined( 'ABSPATH' ) ) exit;

final class StreamNexusIntegration {
    // Option keys
    private const OPT_DEBUG_MODE   = 'streamnexus_debug_mode';

    private const VERSION          = '2.2.2';

    // Defaults
    private const DEF_PLAYER_URL   = 'https://api.streamnexus.io/player-page.html';
    private const DEF_HEIGHT_RATIO = '0.5625'; // 16:9

    // Runtime
    private string $default_player_url = self::DEF_PLAYER_URL;
    private string $default_height_ratio = self::DEF_HEIGHT_RATIO;
    private bool $debug_mode = false;


    public function __construct() {

        $this->load_options();

        add_action( 'admin_init', [$this, 'register_settings'] );
        add_action( 'admin_menu', [$this, 'add_admin_menu'] );

        add_shortcode( 'streamnexus-video', [$this, 'handle_shortcode'] );

        add_action( 'wp_enqueue_scripts', [$this, 'enqueue_scripts'] );

    }

    private function load_options(): void {
        $this->default_player_url   = self::DEF_PLAYER_URL;
        $this->default_height_ratio = self::DEF_HEIGHT_RATIO;
        $this->debug_mode           = (string) get_option(self::OPT_DEBUG_MODE, '0') === '1';
    }

    /** -------------------- Admin: Settings API -------------------- */
    public function add_admin_menu(): void {
        add_options_page(
            esc_html__( 'StreamNexus Settings', 'streamnexus-io-embed-videos' ),
            esc_html__( 'StreamNexus', 'streamnexus-io-embed-videos' ),
                'manage_options',
                'streamnexus-settings',
                [$this, 'settings_page' ]
        );
    }

    public function register_settings(): void {
        // Register options
        register_setting( 'streamnexus', self::OPT_DEBUG_MODE, [
            'type' => 'string',
            'sanitize_callback' => fn($v) => $v === '1' ? '1' : '0',
            'default' => '0',
        ]);

        // Sections
        add_settings_section( 'snx_section_general',  esc_html__('General', 'streamnexus-io-embed-videos'), null, 'streamnexus-settings');

        // Fields
        add_settings_field( self::OPT_DEBUG_MODE, esc_html__('Debug Mode', 'streamnexus-io-embed-videos'), [$this, 'field_debug_mode'], 'streamnexus-settings', 'snx_section_general');
    }

    public function field_debug_mode(): void {
        printf(
            '<label><input type="checkbox" name="%1$s" value="1" %2$s /> %3$s</label>',
            esc_attr(self::OPT_DEBUG_MODE),
            checked($this->debug_mode, true, false),
            esc_html__('Show debug info to admins on the front-end.', 'streamnexus-io-embed-videos')
        );
    }

    public function sanitize_debug_mode($value): string {
        return $value === '1' ? '1' : '0';
    }


    public function settings_page(): void {
        $settings_updated = filter_input(INPUT_GET, 'settings-updated', FILTER_VALIDATE_BOOLEAN);

        if ($settings_updated) {
            // Reload options after successful save handled by options.php (nonce verified there).
            $this->load_options();
            echo '<div class="updated"><p>' . esc_html__('Settings saved.', 'streamnexus-io-embed-videos') . '</p></div>';
        }
        
        settings_errors('streamnexus', false, true);
        ?>
        <div class="wrap">
            <h1><?php echo esc_html__('StreamNexus Configuration', 'streamnexus-io-embed-videos'); ?></h1>
            <form method="post" action="options.php">
                <?php
                    settings_fields('streamnexus');
                    do_settings_sections('streamnexus-settings');
                    submit_button();
                ?>
            </form>

            <h3><?php echo esc_html__('How to use the shortcode', 'streamnexus-io-embed-videos'); ?></h3>
            <ul>
                <li><code>[streamnexus-video media_id="3d6b53c6-7057-43b8-908e-7a131cb0d7aa"]</code> — <?php echo esc_html__('Basic usage', 'streamnexus-io-embed-videos'); ?></li>
                <li><code>[streamnexus-video media_id="..." width="100%" aspect_ratio="16:9"]</code> — <?php echo esc_html__('Control width and ratio', 'streamnexus-io-embed-videos'); ?></li>
                <li><code>[streamnexus-video media_id="..." width="800" height="450"]</code> — <?php echo esc_html__('Fixed dimensions', 'streamnexus-io-embed-videos'); ?></li>
                <li><code>[streamnexus-video media_id="..." aspect_ratio="1:1"]</code> — <?php echo esc_html__('Square video', 'streamnexus-io-embed-videos'); ?></li>
                <li><code>[streamnexus-video media_id="..." height_ratio="0.4"]</code> — <?php echo esc_html__('Custom height', 'streamnexus-io-embed-videos'); ?></li>
            </ul>
        </div>
        <?php
    }

    /** -------------------- Front-end & AJAX -------------------- */
    public function enqueue_scripts(): void {
        wp_register_style(
            'streamnexus-style',
            plugins_url('css/streamnexus.css', __FILE__),
            [],
            self::VERSION
        );

        wp_enqueue_style('streamnexus-style');

        wp_enqueue_script(
            'streamnexus-js',
            plugins_url('js/streamnexus.js', __FILE__),
            ['jquery'],
            self::VERSION,
            true
        );

        wp_localize_script('streamnexus-js', 'streamnexus_ajax', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce'    => wp_create_nonce('streamnexus-video-check'),
            'i18n'     => [
                'unavailable' => esc_html__('This video is not available.', 'streamnexus-io-embed-videos'),
                'ajax_error'  => esc_html__('Error while checking the StreamNexus video.', 'streamnexus-io-embed-videos'),
            ],
        ]);

        if (function_exists('wp_set_script_translations')) {
            wp_set_script_translations('streamnexus-js', 'streamnexus-io-embed-videos', plugin_dir_path(__FILE__) . 'languages');
        }

        // Inline listener without heredoc/nowdoc
        $listener =
            "(function(){\n" .
            "    window.addEventListener('message', function(e){\n" .
            "        var d = e.data;\n" .
            "        if(!d || d.type !== 'snx-ratio' || !d.mediaId || !d.ratio) return;\n" .
            "        var el = document.querySelector('.streamnexus-video-container[data-media-id=\"'+ d.mediaId +'\"]');\n" .
            "        if(!el) return;\n\n" .
            "        // Only auto-adjust containers created in \"default\" mode\n" .
            "        if (el.dataset.snxAutoratio !== '1') return;\n\n" .
            "        // Ensure we are not in fixed height\n" .
            "        el.classList.remove('fixed-height');\n" .
            "        el.style.height = '';\n" .
            "        el.style.paddingBottom = (d.ratio * 100) + '%';\n" .
            "    });\n" .
            "})();";

        wp_add_inline_script('streamnexus-js', $listener, 'after');
    }

    
    public function handle_shortcode($atts): string {
        $atts = shortcode_atts([
            'media_id'     => '',
            'share_link'   => '',
            'width'        => '100%',
            'height'       => '',
            'aspect_ratio' => '',  
            'height_ratio' => '',
            'max_width'    => '',
            'autoplay'     => 'false',
        ], $atts, 'streamnexus-video');

        $media_id = '';
        if ($atts['share_link'] !== '') {
            $parsed = wp_parse_url($atts['share_link']);
            if (isset($parsed['query'])) {
                parse_str($parsed['query'], $q);
                $media_id = isset($q['mediaId']) ? (string)$q['mediaId'] : '';
            }
        } else {
            $media_id = (string) $atts['media_id'];
        }

        if ($media_id === '') {
            return '<div class="streamnexus-error-message">' . esc_html__('Error: No media ID or valid share link provided.', 'streamnexus-io-embed-videos') . '</div>';
        }

        $video_url = add_query_arg('mediaId', $media_id, $this->default_player_url);
        if ($atts['autoplay'] === 'true') {
            $video_url = add_query_arg('autoplay', 'true', $video_url);
        }

        // Determine aspect ratio
        $aspect_ratio_percent = null;
        $use_fixed_height = false;
        $auto_ratio = false; // we’ll mark containers that can be auto-updated

        if ($atts['height_ratio'] !== '') {
            $aspect_ratio_percent = (float) $atts['height_ratio'] * 100.0;
        } elseif ($atts['aspect_ratio'] !== '') {
            $parts = explode(':', $atts['aspect_ratio']);
            if (count($parts) === 2 && (float)$parts[0] > 0) {
                $aspect_ratio_percent = ((float)$parts[1] / (float)$parts[0]) * 100.0;
            } else {
                $aspect_ratio_percent = (float) $this->default_height_ratio * 100.0;
            }
        } elseif ($atts['height'] !== '') {
            $use_fixed_height = true;              // only fixed when height is explicitly set
        } else {
            // DEFAULT → use ratio box (will be corrected by postMessage)
            $aspect_ratio_percent = (float) $this->default_height_ratio * 100.0;
            $auto_ratio = true;                    // allow JS to adjust later
        }

        // Build container styles
        $width_style = '';
        if ($atts['width'] !== '100%') {
            $w = trim((string)$atts['width']);
            $width_style .= 'width: ' . (is_numeric($w) ? ((int)$w . 'px') : esc_attr($w)) . '; ';
        }
        if ($atts['max_width'] !== '') {
            $mw = trim((string)$atts['max_width']);
            $width_style .= 'max-width: ' . (is_numeric($mw) ? ((int)$mw . 'px') : esc_attr($mw)) . '; ';
        }

        // Debug information
        $debug_info = '';
        if ( $this->debug_mode && current_user_can( 'manage_options' ) ) {
            $debug_info  = '<div style="background:#f1f1f1;padding:10px;margin:10px 0;border:1px solid #ddd;">';
            $debug_info .= '<strong>' . esc_html__('Debug Information', 'streamnexus-io-embed-videos') . ':</strong><br>';
            $debug_info .= esc_html__( 'Media ID', 'streamnexus-io-embed-videos' ) . ': ' . esc_html($media_id) . '<br>';
            $debug_info .= esc_html__( 'Player URL', 'streamnexus-io-embed-videos' ) . ': ' . esc_html($video_url) . '<br>';
            $debug_info .= esc_html__( 'Aspect Ratio', 'streamnexus-io-embed-videos' ) . ': ' . esc_html($atts['aspect_ratio']) . '<br>';
            $debug_info .= esc_html__( 'Height Ratio', 'streamnexus-io-embed-videos' ) . ': ' . esc_html($atts['height_ratio']) . '<br>';
            $debug_info .= esc_html__( 'Calculated Padding', 'streamnexus-io-embed-videos' ) . ': ' . esc_html($aspect_ratio_percent) . '%<br>';
            $debug_info .= esc_html__( 'Fixed Height Mode', 'streamnexus-io-embed-videos' ) . ': ' . ($use_fixed_height ? 'Yes' : 'No') . '<br>';
            $debug_info .= '</div>';
        }

        $out = $debug_info;
        
        $classes = ['streamnexus-video-container'];
        $container_style = $width_style;
        $iframe_style = '';

        if ($use_fixed_height) {
            $classes[] = 'fixed-height';
            $h = trim((string)$atts['height']);
            $iframe_height = is_numeric($h) ? ((int)$h . 'px') : esc_attr($h);
            $iframe_style = 'height:' . $iframe_height . ';';
        } else {
            // ratio box
            $container_style .= 'padding-bottom:' . esc_attr($aspect_ratio_percent) . '%;';
        }

        $out .= '<div class="' . implode(' ', $classes) . '"'
            . ' data-media-id="' . esc_attr($media_id) . '"'
            . ($auto_ratio ? ' data-snx-autoratio="1"' : '')
            . ' style="' . $container_style . '">';
        $out .= '<iframe src="' . esc_url($video_url) . '" allowfullscreen loading="lazy" referrerpolicy="no-referrer-when-downgrade"';
        if ($iframe_style) {
            $out .= ' style="' . $iframe_style . '"';
        }
        $out .= '></iframe>';
        $out .= '</div>';
        $out .= '<div class="streamnexus-error-container" style="display:none;"></div>';

        return $out;
    }
}

/** Create JS directory on activation (for future use if needed) */
register_activation_hook( __FILE__ , function () {
    $js_dir = plugin_dir_path( __FILE__ ) . 'js';
    if ( ! file_exists( $js_dir ) ) wp_mkdir_p( $js_dir );
});

new StreamNexusIntegration();