<?php
if (!defined('ABSPATH')) {
    exit;
}

class StoreItContact_SubmissionSyncService {

    private $apiClient = null;
    private $settings = [];

    public function __construct() {
        $this->settings = get_option('store_it_contact_global_settings', []);
        $this->initializeApiClient();
    }

    /**
     * Initialize the API client if credentials are available
     * 
     * @return bool True if API client was initialized, false otherwise
     */
    private function initializeApiClient() {
        // Check if API credentials are available
        if (!empty($this->settings['username']) && 
            !empty($this->settings['password']) && 
            !empty($this->settings['client_id'])) {
            
            try {
                // Create API client instance
                $this->apiClient = new StoreItAPIClient(
                    $this->settings['username'],
                    $this->settings['password'],
                    $this->settings['client_id'],
                    $this->settings['grant_type'] ?? 'password',
                    $this->settings['base_url'] ?? 'https://api.store-it365.eu'
                );
                return true;
            } catch (StoreItException $e) {
                
                // Send notification email to admin
                $this->sendErrorEmail(
                    'Store-IT API Authentication Error',
                    '<p><strong>Error Message:</strong> ' . esc_html($e->getMessage()) . '</p>' .
                    '<p><strong>Status Code:</strong> ' . esc_html($e->statusCode ?? 'unknown') . '</p>' .
                    '<p>Please check your API credentials in the Store-IT Contact settings.</p>'
                );
                
                return false;
            }
        }
        
        return false;
    }

    /**
     * Process a form submission to create a Store-IT customer
     * 
     * @param string $formId The ID of the submitted form
     * @param array $submittedData The form data that was submitted
     * @return array|bool Success status and response data or false if API not available
     */
    public function processSubmission($formId, $submittedData) {
        // Check if API client is available
        if (!$this->apiClient) {
            $this->sendErrorEmail('API client not initialized', 'The Store-IT API client could not be initialized. Please check API credentials.');
            return false;
        }

        // Only proceed if we have an email (required for customer creation)
        if (empty($submittedData['email'])) {
            return [
                'success' => false,
                'message' => 'Email is required for customer creation'
            ];
        }
        
        try {            
            // Get the sitecode from the selected location, default to '01' if not set
            $siteCode = !empty($submittedData['location']) ? $submittedData['location'] : '01';
            
            /**
             * API calls
             */
            $customerData = $this->prepareCustomerData($submittedData, $formId);
            $customer_id = $this->apiClient->createCustomer($customerData, $siteCode);

            $task_data = $this->apiClient->createTask($customer_id, $submittedData['request_type'], $siteCode, $submittedData['message_field']);
            
            $emailData = $this->prepareEmailData($submittedData, $formId, $task_data['taskId'], $customer_id, $siteCode);
            $email_result = $this->apiClient->sendMail($emailData);

            return [
                'success' => true,
                'message' => 'Customer created successfully',
                'response' => [
                    'customer_id' => $customer_id, 
                    'task_data' => $task_data,
                    'email_data' => $email_result,
                    'site_code' => $siteCode
                ]
            ];
        } catch (StoreItException $e) {
            
            // Send notification email to admin
            $this->sendErrorEmail(
                'Store-IT API Error: ' . $e->getMessage(),
                $this->formatErrorDetails($e, $formId, $submittedData)
            );
            
            return [
                'success' => false,
                'message' => $e->getMessage(),
                'code' => $e->statusCode ?? 'unknown'
            ];
        }
    }
    
    /**
     * Prepare customer data for API submission based on form fields
     * 
     * @param array $submittedData The form data that was submitted
     * @param string $formId The ID of the form that was submitted
     * @return array The prepared customer data for the API
     */
    private function prepareCustomerData($submittedData, $formId) {
        // Basic customer data structure with required email
        $customerData = [
            'mailAddresses' => [
                [
                    'typeId' => 1,
                    'address' => $submittedData['email']
                ]
            ]
        ];
        
        // Map form fields to API fields
        $fieldMapping = [
            'first_name' => 'firstName',
            'surname' => 'lastName',
            'infix' => 'infix',
            'title_field' => 'title',
            'company_name' => 'companyName',
            'chamber_of_commerce' => 'chamberOfCommerceNumber',
            'vat_number' => 'vatNumber',
            'date_of_birth' => 'dateOfBirth',
            'language' => 'language',
        ];
        
        // Add mapped fields if they exist in submitted data
        foreach ($fieldMapping as $formField => $apiField) {
            if (!empty($submittedData[$formField])) {
                $customerData[$apiField] = $submittedData[$formField];
            }
        }

        // Handle phone numbers if present
        if (!empty($submittedData['phone_number'])) {
            $customerData['phoneNumbers'] = [
                [
                    'typeId' => 1,
                    'phone' => $submittedData['phone_number'],
                    'isSmsService' => false
                ]
            ];
        }
        
        // Handle address if present
        if (!empty($submittedData['street']) && !empty($submittedData['postal_code']) && !empty($submittedData['city'])) {
            $customerData['addresses'] = [
                [
                    'typeId' => 1,
                    'street' => $submittedData['street'],
                    'extraLine' => $submittedData['address_extra'] ?? '',
                    'postalCode' => $submittedData['postal_code'],
                    'city' => $submittedData['city'],
                    'country' => $submittedData['country'] ?? 'Nederland'
                ]
            ];
        }

        // set language
        $forms = get_option('store_it_contact_forms', []);
        $formData = isset($forms[$formId]) ? $forms[$formId] : [];
        if(isset($formData['language'])) {
            $customerData['language'] = strtoupper($formData['language']);
        }

        
        return $customerData;
    }

    /**
     * Prepare email data for API submission based on form fields
     * 
     * @param array $submittedData The form data that was submitted
     * @param string $formId The ID of the form that was submitted
     * @param int $taskId The task ID for email object references
     * @param int $customerId The customer ID
     * @param string $siteCode The site code
     * @return array The prepared email data for the API
     */
    private function prepareEmailData($submittedData, $formId, $taskId, $customerId, $siteCode) {
        // Get the form configuration to access email template and object settings
        $forms = get_option('store_it_contact_forms', []);
        $formData = isset($forms[$formId]) ? $forms[$formId] : [];
        
        // Get notification settings
        $notifications = isset($formData['notifications']) ? $formData['notifications'] : [];
        
        // Default values if not set
        $emailTemplate = isset($notifications['email_template']) ? $notifications['email_template'] : 91;
        $emailObject = isset($notifications['email_object']) ? $notifications['email_object'] : 'none';
        
        // Prepare email data structure for API - matching the expected format
        $emailData = [
            'templateId' => $emailTemplate,
            'customerId' => $customerId,
            'siteCode' => $siteCode,
            'mailObjects' => [],
        ];

        // Get correct task type data
        $globalSettings = get_option('store_it_contact_global_settings', []);
        $taskTypes = $globalSettings['task_types_data'];

        $taskType = array_filter($taskTypes, function($taskType) use ($submittedData) {
            return $taskType['typeId'] == $submittedData['request_type'];
        });

        $taskType = reset($taskType);

        // Only add mail objects for SendFile and SendTemplate task kinds
        if($taskType['taskKind'] == 'SendFile') {
            $emailData['mailObjects'] = [
                [
                    'objectId' => $submittedData['request_type'], // Using typeId directly
                    'fileType' => 5, // TaskAttachment
                ]
            ];
        } elseif($taskType['taskKind'] == 'SendTemplate') {
            $emailData['mailObjects'] = [
                [
                    'objectId' => $taskId, // Using the created task ID
                    'fileType' => 4, // TaskDocument
                ]
            ];
        }
        
        // // Add mail objects based on selection
        // if ($emailObject !== 'none') {
        //     if ($emailObject === 'task_document') {
        //         // Add task document as mail object (fileType 4 = TaskDocument)
        //         $emailData['mailObjects'][] = [
        //             'objectId' => $taskId,
        //             'fileType' => 4,
        //         ];
        //     } else if ($emailObject === 'task_attachment') {
        //         // Add task attachment as mail object (fileType 5 = TaskAttachment)
        //         $emailData['mailObjects'][] = [
        //             'objectId' => $taskId,
        //             'fileType' => 5,
        //         ];
        //     }
        // }
        
        return $emailData;
    }

    /**
     * Send an error notification email to the WordPress admin
     * 
     * @param string $subject Email subject
     * @param string $message Email message body
     * @return bool Whether the email was sent successfully
     */
    private function sendErrorEmail($subject, $message) {
        $admin_email = get_option('admin_email');
        $site_name = get_bloginfo('name');
        
        $headers = array(
            'Content-Type: text/html; charset=UTF-8',
            'From: ' . $site_name . ' <' . $admin_email . '>'
        );
        
        $email_subject = '[' . $site_name . '] ' . $subject;
        
        return wp_mail($admin_email, $email_subject, $message, $headers);
    }

    /**
     * Format error details for the notification email
     * 
     * @param StoreItException $exception The exception that was thrown
     * @param string $formId The ID of the form that was submitted
     * @param array $submittedData The form data that was submitted
     * @return string Formatted error details
     */
    private function formatErrorDetails($exception, $formId, $submittedData) {
        $details = '<h2>Store-IT API Error Details</h2>';
        $details .= '<p><strong>Time:</strong> ' . current_time('mysql') . '</p>';
        $details .= '<p><strong>Error Message:</strong> ' . esc_html($exception->getMessage()) . '</p>';
        $details .= '<p><strong>Status Code:</strong> ' . esc_html($exception->statusCode ?? 'unknown') . '</p>';
        $details .= '<p><strong>Form ID:</strong> ' . esc_html($formId) . '</p>';
        
        // Add basic submitted data (excluding sensitive information)
        $details .= '<h3>Submitted Data</h3>';
        $details .= '<table border="1" cellpadding="5" style="border-collapse: collapse;">';
        $details .= '<tr><th>Field</th><th>Value</th></tr>';
        
        foreach ($submittedData as $key => $value) {
            $details .= '<tr>';
            $details .= '<td><strong>' . esc_html($key) . '</strong></td>';
            $details .= '<td>' . esc_html($value) . '</td>';
            $details .= '</tr>';
        }
        
        $details .= '</table>';
        
        // Add debug info for technical users
        $details .= '<h3>Technical Details</h3>';
        $details .= '<p>Check server error logs for more detailed information.</p>';
        $details .= '<p><strong>PHP Version:</strong> ' . phpversion() . '</p>';
        $details .= '<p><strong>WordPress Version:</strong> ' . get_bloginfo('version') . '</p>';
        
        return $details;
    }
}
