<?php
if (!defined('ABSPATH')) {
    exit;
}

class StoreItException extends \Exception {
    protected $statusCode;
    protected $errorData;

    public function __construct($message, $statusCode = null, $errorData = null) {
        parent::__construct($message);
        $this->statusCode = $statusCode;
        $this->errorData = $errorData;
    }
}

class StoreItAPIClient {
    private $username;
    private $password;
    private $clientId;
    private $grantType;
    private $baseUrl;
    private $accessToken;
    private $tokenExpiry;
    private $tokenOptionName = 'store_it_api_token';
    private $tokenExpiryOptionName = 'store_it_api_token_expiry';

    public function __construct($username, $password, $clientId, $grantType = 'password', $baseUrl = 'https://api.store-it365.eu')
    {
        $this->username = $username;
        $this->password = $password;
        $this->clientId = $clientId;
        $this->grantType = $grantType;
        $this->baseUrl = rtrim($baseUrl, '/');
        
        $this->loadToken();
        if (!$this->hasValidToken()) {
            $this->authenticate();
        }
    }

    /**
     * Load token from WordPress options
     */
    private function loadToken() {
        $this->accessToken = get_option($this->tokenOptionName);
        $this->tokenExpiry = get_option($this->tokenExpiryOptionName);
    }

    /**
     * Save token to WordPress options
     */
    private function saveToken() {
        update_option($this->tokenOptionName, $this->accessToken);
        update_option($this->tokenExpiryOptionName, $this->tokenExpiry);
    }

    /**
     * Check if we have a valid token
     */
    private function hasValidToken() {
        if (empty($this->accessToken) || empty($this->tokenExpiry)) {
            return false;
        }

        // Allow a 5-minute buffer before expiration
        return $this->tokenExpiry > (time() + 300);
    }

    /**
     * Attempt an OAuth/Token-based authentication using configured credentials.
     *
     * @throws StoreItException
     */
    private function authenticate() {
        /*
         * We assume the token endpoint is typically something like:
         * https://token.store-it365.eu/auth
         *
         * If your settings expect a base domain only, you might
         * adapt this to your environment:
         */
        $tokenEndpoint = 'https://token.store-it365.eu/auth';

        $response = wp_remote_post($tokenEndpoint, array(
            'timeout' => 15,
            'body' => array(
                'username'   => $this->username,
                'password'   => $this->password,
                'grant_type' => $this->grantType,
                'client_id'  => $this->clientId,
            ),
        ));

        if (is_wp_error($response)) {
            throw new StoreItException('Authentication failed: ' . esc_html($response->get_error_message()));
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        $statusCode = wp_remote_retrieve_response_code($response);

        if ($statusCode !== 200 || empty($body['access_token'])) {
            throw new StoreItException('Failed to obtain access token', intval($statusCode));
        }

        $this->accessToken = $body['access_token'];
        
        // Calculate expiry time (3 hours from now)
        $this->tokenExpiry = time() + 3 * 3600;
        
        // Save token and expiry time to WordPress options
        $this->saveToken();
    }

    /**
     * Internal method to make an authenticated API request
     *
     * @throws StoreItException
     */
    private function makeRequest($method, $endpoint, $body = null, $queryParams = []) {
        // Check if token is valid before making request
        if (!$this->hasValidToken()) {
            $this->authenticate();
        }

        // Construct full URL
        $url = $this->baseUrl . $endpoint;
        if (!empty($queryParams)) {
            $url .= '?' . http_build_query($queryParams);
        }

        // Prepare args
        $args = [
            'timeout' => 15,
            'headers' => [
                'Authorization' => 'Bearer ' . $this->accessToken,
            ],
        ];

        if ($body) {
            $args['body'] = wp_json_encode($body);
            $args['headers']['Content-Type'] = 'application/json';
        }

        // Perform request
        $response = ($method === 'GET')
            ? wp_remote_get($url, $args)
            : wp_remote_post($url, $args);

        // Handle errors
        if (is_wp_error($response)) {
            throw new StoreItException('API request failed: ' . esc_html($response->get_error_message()));
        }

        $statusCode = wp_remote_retrieve_response_code($response);
        $responseBody = wp_remote_retrieve_body($response);
        $decodedBody = json_decode($responseBody, true);

        // Handle authentication errors - try to reauthenticate once
        if ($statusCode === 401) {
            $this->authenticate();
            return $this->makeRequest($method, $endpoint, $body, $queryParams);
        }

        if ($statusCode >= 400) {
            $message = isset($decodedBody['message']) ? $decodedBody['message'] : 'Unknown error';
            throw new StoreItException("API error (" . intval($statusCode) . "): " . esc_html($message), intval($statusCode));
        }

        return $decodedBody;
    }

    public function getTitleTypes() {
        return $this->makeRequest('GET', '/v1/api/titleTypes');
    }

    public function getUnitTypeGroups($siteCode = '01') {
        return $this->makeRequest('GET', '/v1/api/UnitTypeGroups', null, ['siteCode' => $siteCode]);
    }

    public function getUnitTypes($siteCode = '01') {
        return $this->makeRequest('GET', '/v1/api/UnitTypes', null, [
            'siteCode' => $siteCode,
            'vacantOnly' => 'true'
        ]);
    }

    public function getUnitTypesByGroup($group, $siteCode = '01') {
        return $this->makeRequest('GET', '/v1/api/UnitTypes', null, [
            'siteCode' => $siteCode,
            'groupId' => $group,
            'vacantOnly' => 'true'
        ]);
    }

    public function getUnitTypesStock($groupId = null, $siteCode = '01') {
        $params = ['siteCode' => $siteCode];
        if ($groupId) {
            $params['groupId'] = $groupId;
        }
        return $this->makeRequest('GET', '/v1/api/UnitTypes/stock', null, $params);
    }

    public function getUnitsByType($typeName, $siteCode = '01') {
        return $this->makeRequest('GET', '/v1/api/units/vacant', null, [
            'unitType' => $typeName,
            'siteCode' => $siteCode
        ]);
    }

    public function priceRequest($data) {
        return $this->makeRequest('POST', '/v2/api/pricerequests', $data);
    }

    public function contractPreview($data) {
        return $this->makeRequest('POST', '/v1/api/previews/contract', $data);
    }

    public function taskPreview($data) {
        return $this->makeRequest('POST', '/v1/api/previews/task', $data);
    }

    public function createCustomer($data, $siteCode = '01') {
        return $this->makeRequest('POST', '/v1/api/customers', $data, [
            'siteCode' => $siteCode,
            'alwaysUpdate' => 'false'
        ]);
    }

    public function createBooking($data) {
        return $this->makeRequest('POST', '/v1/api/bookings', $data);
    }

    public function createPayment($data) {
        return $this->makeRequest('POST', '/v1/api/payments', $data);
    }

    public function postAccessControls($data) {
        $response = $this->makeRequest('POST', '/v1/api/accesscontrols', $data);
        return wp_remote_retrieve_response_code($response);
    }

    public function postMails($data) {
        return $this->makeRequest('POST', '/v1/api/mails', $data);
    }

    public function postBookingSignature($data) {
        return $this->makeRequest('POST', '/v1/api/bookingsignatures', $data);
    }

    public function sendMail($data) {
        return $this->makeRequest('POST', '/v2/api/mails', $data);
    }

    public function postRecurrings($data) {
        return $this->makeRequest('POST', '/v1/api/recurrings', $data);
    }

    public function getFacilities() {
        return $this->makeRequest('GET', '/v1/api/facilities');
    }

    public function getTaskTypes() {
        return $this->makeRequest('GET', '/v2/api/tasktypes');
    }

    /**
     * Create a task
     * 
     * @param string $customerId
     * @param string $taskTypeId
     * @param string $siteCode
     * @param string $note
     * @return array
     * 
     * https://apidocs.store-it365.eu/#api-Tasks-PostTasks
     */
    public function createTask($customerId, $taskTypeId, $siteCode = '01', $note = null) {
        return $this->makeRequest('POST', '/v1/api/tasks', [
            'siteCode' => $siteCode,
            'customerId' => $customerId,
            'taskTypeId' => $taskTypeId,
            'note' => $note
        ]);
    }
}
