<?php
if (!defined('ABSPATH')) {
    exit;
}

class StoreItContactSubmissionsManager
{
    /**
     * Name of the option in wp_options that stores all submissions
     */
    private $optionName = 'store_it_contact_submissions';

    /**
     * Array of all submissions loaded from the DB
     * Structure:
     * [
     *   'form_ID' => [
     *       'submission_ID' => [
     *           'date' => '...',
     *           'fields' => [...],
     *       ],
     *   ],
     * ]
     */
    private $submissions = array();

    /**
     * Load all submissions from the option
     */
    public function loadSubmissions()
    {
        $this->submissions = get_option($this->optionName, array());
    }

    /**
     * Stores a new submission under the given form ID
     *
     * @param string $formId
     * @param array  $submittedData
     * @return string Newly generated submission ID
     */
    public function saveSubmission($formId, array $submittedData)
    {
        $this->loadSubmissions();

        // Ensure structure
        if (!isset($this->submissions[$formId])) {
            $this->submissions[$formId] = array();
        }

        // Generate a unique submission ID
        $submissionId = 'submission_' . time() . '_' . wp_rand(1000, 9999);

        // Prepare the submission entry
        $this->submissions[$formId][$submissionId] = array(
            'date'   => current_time('mysql'),
            'fields' => $submittedData,
        );

        update_option($this->optionName, $this->submissions);

        return $submissionId;
    }

    /**
     * Retrieve submissions by form ID
     *
     * @param string $formId
     * @return array
     */
    public function getSubmissions($formId)
    {
        $this->loadSubmissions();

        if (!isset($this->submissions[$formId])) {
            return array(); // No submissions for this form
        }

        // sort submissions by date
        usort($this->submissions[$formId], function($a, $b) {
            return strtotime($b['date']) - strtotime($a['date']);
        });

        return $this->submissions[$formId];
    }

    /**
     * Return a specific submission if it exists
     *
     * @param string $formId
     * @param string $submissionId
     * @return array|null
     */
    public function getSubmission($formId, $submissionId)
    {
        $this->loadSubmissions();

        if (isset($this->submissions[$formId][$submissionId])) {
            return $this->submissions[$formId][$submissionId];
        }
        return null;
    }
} 