<?php
if (!defined('ABSPATH')) {
    exit;
}

class StoreItContactFormsManager {

    /**
     * Name of the option in wp_options that stores all forms
     */
    private $optionName = 'store_it_contact_forms';

    /**
     * Array of all forms loaded from the DB
     */
    private $forms = array();

    /**
     * Load all forms from the option
     */
    public function loadForms() {
        $this->forms = get_option($this->optionName, array());
    }

    /**
     * Return the entire array
     * @return array
     */
    public function getForms() {
        return $this->forms;
    }

    /**
     * Return a single form by ID
     * @param string $formId
     * @return array|null
     */
    public function getForm($formId) {
        return isset($this->forms[$formId]) ? $this->forms[$formId] : null;
    }

    /**
     * Save or update a form in the forms array and persist to the DB
     * @param string|null $formId
     * @param string      $name
     * @param string      $title
     * @param string      $language
     * @param array       $requestTypes
     * @param array       $locations
     * @param array       $fields
     * @param array       $styling
     * @return string     ID of the saved form
     */
    public function saveForm(
        $formId,
        $name,
        $title,
        $language,
        $requestTypes = array(),
        $locations = array(),
        $fields = array(),
        $styling = array()
    ) {
        // Load forms first to ensure we have the latest data
        $this->loadForms();

        // Only create a new ID if none was provided (new form not edit)
        if (empty($formId)) {
            $formId = 'form_' . time() . '_' . wp_rand(1000, 9999);
        }

        $this->forms[$formId] = array(
            'name'          => $name,
            'title'         => $title,
            'language'      => $language,
            'request_types' => $requestTypes,
            'locations'     => $locations,
            'fields'        => $fields,
            'styling'       => array(
                'background_color'      => isset($styling['background_color']) ? $styling['background_color'] : '',
                'border_color'          => isset($styling['border_color']) ? $styling['border_color'] : '',
                'text_color'            => isset($styling['text_color']) ? $styling['text_color'] : '',
                'font_size'             => isset($styling['font_size']) ? $styling['font_size'] : 16,
                'accent_color'          => isset($styling['accent_color']) ? $styling['accent_color'] : '',
                'accent_contrast_color' => isset($styling['accent_contrast_color']) ? $styling['accent_contrast_color'] : '',
                'border_radius'         => isset($styling['border_radius']) ? $styling['border_radius'] : 0
            ),
            'success_message' => isset($styling['success_message']) ? $styling['success_message'] : '',
            'default_error_message' => isset($styling['default_error_message']) ? $styling['default_error_message'] : ''
        );

        update_option($this->optionName, $this->forms);

        return $formId;
    }

    /**
     * Delete a form from the forms array and update the option
     * @param string $formId
     */
    public function deleteForm($formId) {
        if (isset($this->forms[$formId])) {
            unset($this->forms[$formId]);
            update_option($this->optionName, $this->forms);
        }
    }

    /**
     * Save forms to the option
     * 
     * @param array $forms The forms data to save
     * @return bool Whether the save was successful
     */
    public function saveForms($forms) {
        $this->forms = $forms;
        return update_option($this->optionName, $this->forms);
    }

    /**
     * Duplicate a form by creating a copy with a new ID and modified name
     * @param string $formId The ID of the form to duplicate
     * @return string|false The new form ID if successful, false on failure
     */
    public function duplicateForm($formId) {
        // Load forms first to ensure we have the latest data
        $this->loadForms();
        
        // Check if the source form exists
        if (!isset($this->forms[$formId])) {
            return false;
        }
        
        // Get the source form data
        $sourceForm = $this->forms[$formId];
        
        // Create a new unique ID
        $newFormId = 'form_' . time() . '_' . wp_rand(1000, 9999);
        
        // Create a copy of the form with a modified name
        $newForm = $sourceForm;
        $newForm['name'] = $sourceForm['name'] . ' (Copy)';
        
        // Save the duplicated form
        $this->forms[$newFormId] = $newForm;
        update_option($this->optionName, $this->forms);
        
        return $newFormId;
    }
}