<?php
if (!defined('ABSPATH')) {
    exit;
}

class StoreItContactShortcodes {

    private $formsManager;
    private $submissionsManager;
    private $submissionSyncService;

    public function __construct($formsManager, $submissionsManager, $submissionSyncService = null) {
        $this->formsManager = $formsManager;
        $this->submissionsManager = $submissionsManager;
        $this->submissionSyncService = $submissionSyncService;
    }

    /**
     * Initialize shortcode hooks
     */
    public function init() {
        add_action('init', array($this, 'registerShortcodes'));
        add_action('init', array($this, 'handleFrontendSubmission'));
        add_action('wp_enqueue_scripts', array($this, 'enqueueScripts'));
    }

    /**
     * Register the [store_it_contact_form id="..."] shortcode
     */
    public function registerShortcodes() {
        add_shortcode('store_it_contact_form', array($this, 'renderContactFormShortcode'));
    }

    /**
     * Handle front-end submissions posted by the contact form
     */
    public function handleFrontendSubmission(){
        try {
            // Check if this is our form submission
            if (!empty($_POST['store_it_contact_form_id']) && isset($_POST['store_it_contact_submit_flag'])) {
                $formId = sanitize_text_field(wp_unslash($_POST['store_it_contact_form_id']));
                
                // Verify nonce for security
                if (!isset($_POST['store_it_contact_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['store_it_contact_nonce'])), 'store_it_contact_form_' . $formId)) {
                    // Redirect to form with error
                    wp_safe_redirect(add_query_arg([
                        'store_it_form_submit_error' => 1,
                        'store_it_contact_form_id' => $formId
                    ]) . '#error-store-it-contact-form-' . $formId);
                    exit;
                }
    
                // Gather all posted fields we care about
                $submittedData = array();
                
                // Load the form to see which fields are present
                $this->formsManager->loadForms();
                $theForm = $this->formsManager->getForm($formId);

                if ($theForm && !empty($theForm['fields'])) {
                    foreach ($theForm['fields'] as $fieldId => $config) {
                        // If not visible, skip
                        if (empty($config['visible']) || $config['visible'] !== 'visible') {
                            continue;
                        }
                        // Build the name attribute
                        $inputName = 'store_it_' . $fieldId;
                        if (isset($_POST[$inputName])) {
                            $submittedData[$fieldId] = sanitize_text_field(wp_unslash($_POST[$inputName]));
                        } else {
                            // If it wasn't submitted, store an empty string
                            $submittedData[$fieldId] = '';
                        }
                    }
                    
                    // Check if the location is "other" and replace with the standin facility if configured
                    if (isset($submittedData['location']) && $submittedData['location'] === 'other') {
                        // If we have a standin facility configured, use it instead of "other"
                        if (!empty($theForm['other_facility_standin'])) {
                            $submittedData['location'] = $theForm['other_facility_standin'];
                        }
                    }
                    
                    // Now we have a complete package of data, send it to the API
                    $syncService = new StoreItContact_SubmissionSyncService();
                    $result = $syncService->processSubmission($formId, $submittedData);

                    $submittedData['result'] = json_encode($result);

                    $submissionId = $this->submissionsManager->saveSubmission($formId, $submittedData);
                    
                    if (is_array($result) && isset($result['success']) && !$result['success']) {
                        // API returned an error, show error message
                        $errorDetails = !empty($result['message']) ? $result['message'] : null;
                        set_transient('store_it_contact_form_error_' . $formId, $errorDetails, 60 * 10);

                        // Redirect back to the form with error flag and form ID
                        wp_safe_redirect(add_query_arg([
                            'store_it_form_submit_error' => 1,
                            'store_it_contact_form_id' => $formId
                        ]) . '#error-store-it-contact-form-' . $formId);
                        exit;
                    } else {
                        // If we get a result back, store it in a transient
                        if ($result) {
                            set_transient('store_it_contact_submission_result_' . $formId, $result, 60 * 10);
                        }
                        // Redirect back to the same page to avoid resubmission
                        wp_safe_redirect(add_query_arg([
                            'store_it_form_submit_success' => 1,
                            'store_it_contact_form_id' => $formId
                        ]) . '#success-store-it-contact-form-' . $formId);
                        exit;
                    }
                }
            }
        } catch (Exception $e) {
            // Handle errors silently in production
        }
    }

    /**
     * Renders the contact form based on the passed-in form ID.
     * Usage: [store_it_contact_form id="FORM_ID"]
     *
     * @param array $atts
     * @return string
     */
    public function renderContactFormShortcode($atts) {

        // Ensure forms are loaded
        $this->formsManager->loadForms();

        $atts = shortcode_atts(
            array('id' => ''),
            $atts,
            'store_it_contact_form'
        );
        $formId = $atts['id'];

        $formData = $this->formsManager->getForm($formId);

        // If the form does not exist, return a message
        if (!$formData) {
            return '<p>' . esc_html__('Form not found.', 'store-it-contact') . '</p>';
        }

        // Make $formData available in the included file
        $formDataForView = $formData;

        // Extract the form data
        $name          = isset($formDataForView['name'])          ? $formDataForView['name']          : '';
        $title         = isset($formDataForView['title'])         ? $formDataForView['title']         : '';
        $language      = isset($formDataForView['language'])      ? $formDataForView['language']      : 'en';
        $requestTypes  = isset($formDataForView['request_types']) ? $formDataForView['request_types'] : array();
        $locations     = isset($formDataForView['locations'])     ? $formDataForView['locations']     : array();
        $fields        = isset($formDataForView['fields'])        ? $formDataForView['fields']        : array();
        $styling       = isset($formDataForView['styling'])       ? $formDataForView['styling']       : array();

        // Extract styling
        $bgColor             = !empty($styling['background_color'])      ? $styling['background_color']         : '#ffffff';
        $borderColor         = !empty($styling['border_color'])          ? $styling['border_color']             : '#cccccc';
        $textColor           = !empty($styling['text_color'])            ? $styling['text_color']               : '#000000';
        $fontSize            = !empty($styling['font_size'])             ? intval($styling['font_size'])        : 16;
        $accentColor         = !empty($styling['accent_color'])          ? $styling['accent_color']             : '#f00';
        $accentContrastColor = !empty($styling['accent_contrast_color']) ? $styling['accent_contrast_color']    : '#fff';
        $borderRadius        = !empty($styling['border_radius'])         ? intval($styling['border_radius'])    : 0;

        // Get all task types and facilities
        $global_settings = get_option('store_it_contact_global_settings', array());
        $allTaskTypes = isset($global_settings['task_types_data']) ? $global_settings['task_types_data'] : array();
        $allFacilities = isset($global_settings['facilities_data']) ? $global_settings['facilities_data'] : array();
        $allTitleTypes = isset($global_settings['title_types_data']) ? $global_settings['title_types_data'] : array();

        // Get reCAPTCHA settings
        $recaptchaSiteKey = isset($global_settings['recaptcha_site_key']) ? $global_settings['recaptcha_site_key'] : '';
        $recaptchaSecretKey = isset($global_settings['recaptcha_secret_key']) ? $global_settings['recaptcha_secret_key'] : '';

        // Get available task types
        $availableTaskTypes = array();
        if (!empty($allTaskTypes) && !empty($requestTypes)) {
            foreach ($allTaskTypes as $taskType) {
                if (in_array((string)$taskType['typeId'], $requestTypes, true)) {
                    $availableTaskTypes[] = $taskType;
                }
            }
            // Sort task types alphabetically by description
            usort($availableTaskTypes, function($a, $b) {
                return strcasecmp($a['description'], $b['description']);
            });
        }

        // Get available title types
        $availableTitleTypes = array();
        if (!empty($allTitleTypes)) {
            foreach ($allTitleTypes as $titleType) {
                if (isset($titleType['code']) && isset($titleType['description'])) {
                    $availableTitleTypes[] = $titleType;
                }
            }
            // Sort title types alphabetically by description
            usort($availableTitleTypes, function($a, $b) {
                return strcasecmp($a['description'], $b['description']);
            });
        }
        
        // Get available facilities
        $availableFacilities = array();
        if (!empty($allFacilities) && !empty($locations)) {
            foreach ($allFacilities as $facility) {
                if (in_array((string)$facility['siteCode'], $locations, true)) {
                    $availableFacilities[] = $facility;
                }
            }
            
            // Check if "other" is selected but not added yet (since it's not in the API data)
            if (in_array('other', $locations, true)) {
                // Get the custom name for "Other" option
                $otherName = isset($formDataForView['other_name']) ? $formDataForView['other_name'] : 'Other';
                
                // Add the custom "Other" option manually
                $availableFacilities[] = [
                    'siteCode' => 'other',
                    'description' => $otherName
                ];
            }
            
            // Sort facilities alphabetically by description
            usort($availableFacilities, function($a, $b) {
                return strcasecmp($a['description'], $b['description']);
            });
        }

        // Simple translations for placeholders/labels
        $placeholders = array(
            'en' => array('submit' => 'Send Message'),
            'nl' => array('submit' => 'Versturen'),
            'de' => array('submit' => 'Senden'),
            'fr' => array('submit' => 'Envoyer'),
            'it' => array('submit' => 'Invia'),
            'es' => array('submit' => 'Enviar'),
            'fi' => array('submit' => 'Lähetä')
        );

        // Fallback if language array not defined
        if (!array_key_exists($language, $placeholders)) {
            $language = 'en';
        }

        $submitLabel = isset($placeholders[$language]['submit']) ? $placeholders[$language]['submit'] : 'Send';
        

        // Enqueue reCAPTCHA script if configured
        if (!empty($recaptchaSiteKey) && !empty($recaptchaSecretKey)) {
            $this->enqueueRecaptchaScript();
        }

        // Simple output buffering to include a "view" file
        ob_start();

        include STOREIT_CONTACT_PLUGIN_DIR . 'includes/views/frontend/ContactForm.php';
        return ob_get_clean();
    }

    /**
     * Enqueue frontend scripts and styles
     */
    public function enqueueScripts() {
        // Enqueue frontend styles
        wp_enqueue_style(
            'store-it-contact-frontend',
            STOREIT_CONTACT_PLUGIN_URL . 'assets/css/frontend.css',
            array(),
            STOREIT_CONTACT_VERSION
        );
        
        wp_enqueue_script(
            'store-it-contact-frontend',
            STOREIT_CONTACT_PLUGIN_URL . 'assets/js/frontend.js',
            array(),
            STOREIT_CONTACT_VERSION,
            true
        );
    }

    /**
     * Enqueue reCAPTCHA scripts when needed
     */
    public function enqueueRecaptchaScript() {
        if (!wp_script_is('recaptcha', 'enqueued')) {
            wp_enqueue_script('recaptcha', 'https://www.google.com/recaptcha/api.js', array(), STOREIT_CONTACT_VERSION, true);
        }
        
        if (!wp_script_is('store-it-contact-recaptcha-validation', 'enqueued')) {
            wp_enqueue_script(
                'store-it-contact-recaptcha-validation',
                STOREIT_CONTACT_PLUGIN_URL . 'assets/js/recaptcha-validation.js',
                array('recaptcha'),
                STOREIT_CONTACT_VERSION,
                true
            );
        }
    }

}