<?php
if (!defined('ABSPATH')) {
    exit;
}

class StoreItContactAdmin {

    private $formsManager;
    private $submissionsManager;
    private $defaultFields;
    private $languageController;

    public function __construct($formsManager, $submissionsManager) {
        $this->formsManager       = $formsManager;
        $this->submissionsManager = $submissionsManager;
        $this->languageController = new StoreItContact_LanguageController();

        // Initialize default fields once with translations
        $this->initializeDefaultFields();

        add_action('admin_enqueue_scripts', array($this, 'enqueueAdminScripts'));
    }

    /**
     * Initialize default fields with current language
     * 
     * @param string $language Language code (default: 'en')
     */
    private function initializeDefaultFields($language = 'en') {
        $this->defaultFields = array(
            'location'      => array('label' => $this->languageController->translate('location', $language),      'visible' => 'visible', 'required' => 'required', 'placeholder' => '', 'description' => '', 'lock_required' => true),
            'request_type'  => array('label' => $this->languageController->translate('request_type', $language),  'visible' => 'visible', 'required' => 'required', 'placeholder' => '', 'description' => '', 'lock_required' => true),
            'title_field'   => array('label' => $this->languageController->translate('title_field', $language),   'visible' => 'visible', 'required' => 'notrequired', 'placeholder' => '', 'description' => ''),
            'first_name'    => array('label' => $this->languageController->translate('first_name', $language),    'visible' => 'visible', 'required' => 'notrequired', 'placeholder' => '', 'description' => ''),
            'surname'       => array('label' => $this->languageController->translate('surname', $language),       'visible' => 'visible', 'required' => 'notrequired', 'placeholder' => '', 'description' => ''),
            'phone_number'  => array('label' => $this->languageController->translate('phone_number', $language),  'visible' => 'visible', 'required' => 'notrequired', 'placeholder' => '', 'description' => ''),
            'email'         => array('label' => $this->languageController->translate('email', $language),         'visible' => 'visible', 'required' => 'required', 'placeholder' => '', 'description' => '', 'lock_required' => true),
            'message_field' => array('label' => $this->languageController->translate('message_field', $language), 'visible' => 'visible', 'required' => 'notrequired', 'placeholder' => '', 'description' => ''),
            'recaptcha'     => array('label' => $this->languageController->translate('recaptcha', $language),     'visible' => 'visible', 'required' => 'notrequired', 'placeholder' => '', 'description' => ''),
            'submit_button' => array('label' => $this->languageController->translate('submit_button', $language), 'visible' => 'visible', 'required' => 'notrequired', 'placeholder' => '', 'description' => ''),
        );
    }

    /**
     * Initialize all admin hooks
     */
    public function init() {
        add_action('admin_menu', array($this, 'registerAdminPages'));
        add_action('admin_init', array($this, 'handleFormSubmission'));
        add_action('admin_init', array($this, 'handleUrlActions'));
        add_action('admin_enqueue_scripts', array($this, 'enqueueAdminScripts'));
    }

    /**
     * Enqueue all admin scripts/styles
     */
    public function enqueueAdminScripts($hook) {
        // Only load on our plugin pages
        if (strpos($hook, 'store-it-contact') === false) {
            return;
        }

        // jQuery UI
        wp_enqueue_script('jquery-ui-sortable');
        wp_enqueue_script('jquery-ui-core');
        
        // Thickbox
        wp_enqueue_style('thickbox');
        wp_enqueue_script('thickbox');
        
        // Color picker
        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('wp-color-picker');
        
        // Custom admin script
        wp_enqueue_script(
            'store-it-contact-admin',
            plugins_url('/assets/js/admin.js', dirname(dirname(__FILE__))),
            array('jquery', 'jquery-ui-sortable', 'wp-color-picker', 'thickbox'),
            '1.0.0',
            true
        );

        // Add inline admin JavaScript for form functionality
        $this->addInlineAdminScript();

        // Admin styles
        wp_enqueue_style(
            'store-it-contact-admin',
            plugins_url('/assets/css/admin.css', dirname(dirname(__FILE__))),
            array('wp-color-picker', 'thickbox'),
            '1.0.0'
        );
    }

    /**
     * Register admin pages and subpages
     */
    public function registerAdminPages() {
        /**
         * Overview page
         */
        add_menu_page(
            __('Store-IT Contact', 'store-it-contact'),
            __('Store-IT Contact', 'store-it-contact'),
            'manage_options',
            'store-it-contact',
            array($this, 'renderOverviewPage'),
            'dashicons-email-alt'
        );

        /**
         * Settings subpage
         */
        add_submenu_page(
            'store-it-contact',
            __('Settings', 'store-it-contact'),
            __('Settings', 'store-it-contact'),
            'manage_options',
            'store-it-contact-global-settings',
            [$this, 'renderGlobalSettingsPage']
        );

        /**
         * Edit form subpage
         */
        add_submenu_page(
            'store-it-contact-global-settings',
            __('Edit Form', 'store-it-contact'),
            __('Edit Form', 'store-it-contact'),
            'manage_options',
            'store-it-contact-edit',
            array($this, 'renderEditPage')
        );
    }

    /**
     * Handle form submissions
     */
    public function handleFormSubmission() {
        if (!isset($_POST['store_it_contact_edit_form_nonce']) || 
            !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['store_it_contact_edit_form_nonce'])), 'store_it_contact_edit_form')) {
            return;
        }


        $this->formsManager->loadForms();
        $formId = isset($_GET['form_id']) ? sanitize_text_field(wp_unslash($_GET['form_id'])) : '';
        $forms = $this->formsManager->getForms();

        // Get existing form data if editing
        $existingData = array();
        if ($formId && isset($forms[$formId])) {
            $existingData = $forms[$formId];
        }

        // Initialize with existing data or defaults, ensuring fields are always present
        $formData = array_merge(
            array(
                'name' => 'New Form',
                'title' => '',
                'language' => 'en',
                'request_types' => array(),
                'styling' => array(
                    'background_color' => '#ffffff',
                    'border_color' => '#cccccc',
                    'text_color' => '#000000',
                    'font_size' => 16,
                    'accent_color' => '#f00',
                    'accent_contrast_color' => '#fff',
                    'border_radius' => 0
                ),
                'fields' => $this->defaultFields,
                'success_message' => __('Thank you! Your message has been submitted.', 'store-it-contact'),
                'default_error_message' => __('An error occurred while submitting the form. Please try again later.', 'store-it-contact')
            ),
            $existingData
        );

        /**
         * Update only the fields that were submitted based on the current tab
         * 
         * Settings tab
         */       
        if (isset($_POST['name'])) {
            $formData['name'] = sanitize_text_field(wp_unslash($_POST['name']));
            $formData['title'] = isset($_POST['title']) ? sanitize_text_field(wp_unslash($_POST['title'])) : '';
            $formData['language'] = isset($_POST['language']) ? sanitize_text_field(wp_unslash($_POST['language'])) : '';

            $formData['request_types'] = isset($_POST['request_types']) ? 
                array_map('sanitize_text_field', wp_unslash($_POST['request_types'])) : 
                array();

            $formData['locations'] = isset($_POST['locations']) ? 
                array_map('sanitize_text_field', wp_unslash($_POST['locations'])) : 
                array();
                
            // Save the custom "Other" option name
            $formData['other_name'] = isset($_POST['other_name']) ? 
                stripslashes(sanitize_text_field(wp_unslash($_POST['other_name']))) : 
                'Other';
            
            // Save the selected facility standin
            $formData['other_facility_standin'] = isset($_POST['other_facility_standin']) ? 
                sanitize_text_field(wp_unslash($_POST['other_facility_standin'])) : 
                '';
        }

        /**
         * Styling tab
         */
        if (isset($_POST['background_color'])) {
            $formData['styling'] = array(
                'background_color' => sanitize_text_field(wp_unslash($_POST['background_color'])),
                'border_color' => isset($_POST['border_color']) ? sanitize_text_field(wp_unslash($_POST['border_color'])) : '',
                'text_color' => isset($_POST['text_color']) ? sanitize_text_field(wp_unslash($_POST['text_color'])) : '',
                'font_size' => isset($_POST['font_size']) ? intval($_POST['font_size']) : 14,
                'accent_color' => isset($_POST['accent_color']) ? sanitize_text_field(wp_unslash($_POST['accent_color'])) : '',
                'accent_contrast_color' => isset($_POST['accent_contrast_color']) ? sanitize_text_field(wp_unslash($_POST['accent_contrast_color'])) : '',
                'border_radius' => isset($_POST['border_radius']) ? intval($_POST['border_radius']) : 0
            );
        }

        /**
         * Fields tab
         */
        if (isset($_POST['fields_data'])) {
            $rawFieldsData = sanitize_textarea_field(wp_unslash($_POST['fields_data']));
            $fieldsData = json_decode(stripslashes($rawFieldsData), true);
            if (is_array($fieldsData) && !empty($fieldsData)) {
                // Sanitize the nested array data
                $sanitizedFields = array();
                foreach ($fieldsData as $fieldKey => $fieldData) {
                    if (is_array($fieldData)) {
                        $sanitizedFields[sanitize_key($fieldKey)] = array_map('sanitize_text_field', $fieldData);
                    }
                }
                $formData['fields'] = $sanitizedFields;
            } else {
                // If fields data is empty or invalid, use default fields
                $formData['fields'] = $this->defaultFields;
            }
        }
        
        /**
         * Notifications tab
         */
        if (isset($_POST['success_message'])) {
            $formData['notifications'] = [
                'success_message' => sanitize_text_field(wp_unslash($_POST['success_message'])),
                'default_error_message' => isset($_POST['default_error_message']) ? wp_kses_post(wp_unslash($_POST['default_error_message'])) : '',
                'email_template' => isset($_POST['email_template']) ? intval($_POST['email_template']) : 91,
                'email_object' => isset($_POST['email_object']) ? sanitize_text_field(wp_unslash($_POST['email_object'])) : 'none'
            ];
        }

        // After updating the language in the form data
        if (isset($_POST['language'])) {
            $newLanguage = sanitize_text_field(wp_unslash($_POST['language']));
            $currentLanguage = isset($forms[$formId]['language']) ? $forms[$formId]['language'] : '';
            
            // Only update labels if the language has actually changed
            $languageChanged = $newLanguage !== $currentLanguage;
            
            $formData['language'] = $newLanguage;
            
            // Reinitialize default fields with the selected language
            $this->initializeDefaultFields($formData['language']);
            
            if (empty($formId) || !isset($forms[$formId]['fields']) || empty($forms[$formId]['fields'])) {
                $formData['fields'] = $this->defaultFields;
            } else if ($languageChanged) {
                // Only update field labels when language changes
                foreach ($formData['fields'] as $fieldId => $fieldConfig) {
                    if (isset($this->defaultFields[$fieldId])) {
                        $formData['fields'][$fieldId]['label'] = $this->defaultFields[$fieldId]['label'];
                    }
                }
                
                // Update success message with new language translation
                $translatedSuccessMessage = $this->languageController->translate('success_message', $newLanguage);
                if (!empty($translatedSuccessMessage)) {
                    if (!isset($formData['notifications'])) {
                        $formData['notifications'] = [];
                    }
                    $formData['notifications']['success_message'] = $translatedSuccessMessage;
                }

                // Update default error message with new language translation
                $translatedDefaultErrorMessage = $this->languageController->translate('default_error_message', $newLanguage);
                if (!empty($translatedDefaultErrorMessage)) {
                    if (!isset($formData['notifications'])) {
                        $formData['notifications'] = [];
                    }
                    $formData['notifications']['default_error_message'] = $translatedDefaultErrorMessage;
                }
            }
        }

        // When creating a new form, generate a new ID
        if (empty($formId)) {
            $formId = 'form_' . time();
        }

        $forms[$formId] = $formData;
        $this->formsManager->saveForms($forms);

        // Redirect back to the same tab
        $tab = isset($_GET['tab']) ? sanitize_text_field(wp_unslash($_GET['tab'])) : 'settings';
        wp_safe_redirect(add_query_arg(
            array(
                'page' => 'store-it-contact-edit',
                'form_id' => $formId,
                'tab' => $tab,
                'updated' => '1'
            ),
            admin_url('admin.php')
        ));
        exit;
    }

    /**
     * Handle URL actions (duplicate, etc.) - called early to allow proper redirects
     */
    public function handleUrlActions() {
        // Only process on our plugin pages
        if (!isset($_GET['page']) || strpos(sanitize_text_field(wp_unslash($_GET['page'])), 'store-it-contact') !== 0) {
            return;
        }

        // Handle duplicate action
        if (isset($_GET['action']) && sanitize_text_field(wp_unslash($_GET['action'])) === 'duplicate' && isset($_GET['form_id'])) {
            // Check user capabilities
            if (!current_user_can('manage_options')) {
                wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'store-it-contact'));
            }

            // Check nonce for security
            if (!isset($_GET['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'duplicate_form_' . sanitize_text_field(wp_unslash($_GET['form_id'])))) {
                wp_die(esc_html__('Security check failed.', 'store-it-contact'));
            }

            $duplicateId = sanitize_text_field(wp_unslash($_GET['form_id']));
            $this->formsManager->loadForms();
            $newFormId = $this->formsManager->duplicateForm($duplicateId);
            
            // Redirect back to overview page
            if ($newFormId) {
                $redirectUrl = admin_url('admin.php?page=store-it-contact&duplicated=1');
                wp_safe_redirect($redirectUrl);
                exit;
            } else {
                $redirectUrl = admin_url('admin.php?page=store-it-contact&error=1');
                wp_safe_redirect($redirectUrl);
                exit;
            }
        }
    }

    /**
     * Overview page: list forms, allow edit/delete, link to Add New
     */
    public function renderOverviewPage() {
        // We must load forms before rendering
        $this->formsManager->loadForms();

        // Handle possible delete action
        if (isset($_GET['action']) && sanitize_text_field(wp_unslash($_GET['action'])) === 'delete' && isset($_GET['form_id'])) {
            // Check user capabilities
            if (!current_user_can('manage_options')) {
                wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'store-it-contact'));
            }

            // Check nonce for security
            if (!isset($_GET['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'delete_form_' . sanitize_text_field(wp_unslash($_GET['form_id'])))) {
                wp_die(esc_html__('Security check failed.', 'store-it-contact'));
            }

            $deleteId = sanitize_text_field(wp_unslash($_GET['form_id']));
            $this->formsManager->deleteForm($deleteId);
        }

        $forms = $this->formsManager->getForms();

        // Include the admin "overview" view
        include STOREIT_CONTACT_PLUGIN_DIR . 'includes/views/admin/Overview.php';
    }

    /**
     * Get task types from global settings
     * @return array
     */
    private function getTaskTypes() {
        $globalSettings = get_option('store_it_contact_global_settings', []);
        return isset($globalSettings['task_types_data']) ? $globalSettings['task_types_data'] : [];
    }

    /**
     * Get facilities data from global settings, and add a custom "Other" option
     */
    private function getFacilities() {
        $globalSettings = get_option('store_it_contact_global_settings', []);
        $facilities = isset($globalSettings['facilities_data']) ? $globalSettings['facilities_data'] : [];
        
        // Use the custom name if editing a form, or default to "Other"
        $formId = '';
        $otherName = 'Other'; // Default
        
        // Only process form_id if it's in an admin context (these are read-only operations)
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Admin navigation parameters, read-only operation
        if (isset($_GET['form_id']) && is_admin() && current_user_can('manage_options')) {
            // These are read-only operations for admin interface, check page context
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Admin navigation parameters
            if (isset($_GET['page']) && strpos(sanitize_text_field(wp_unslash($_GET['page'])), 'store-it-contact') === 0) {
                // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Admin navigation parameters
                $formId = sanitize_text_field(wp_unslash($_GET['form_id']));
                if ($formId) {
                    $forms = $this->formsManager->getForms();
                    if (isset($forms[$formId]['other_name'])) {
                        $otherName = $forms[$formId]['other_name'];
                    }
                }
            }
        }
        
        // Add custom "Other" option to facilities
        $facilities[] = [
            'siteCode' => 'other',
            'description' => $otherName
        ];
        
        return $facilities;
    }

    /**
     * Edit page: add or edit a contact form, plus new sub-tabs for Settings, Styling, Fields, and Entries
     */
    public function renderEditPage() {
        $this->formsManager->loadForms();

        // Only process GET parameters in admin context for read-only operations
        $formId = '';
        $tab = 'settings'; // default to 'settings'
        
        if (is_admin() && current_user_can('manage_options')) {
            // These are read-only operations for admin interface, check page context
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Admin navigation parameters
            if (isset($_GET['page']) && strpos(sanitize_text_field(wp_unslash($_GET['page'])), 'store-it-contact') === 0) {
                // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Admin navigation parameters
                $formId = isset($_GET['form_id']) ? sanitize_text_field(wp_unslash($_GET['form_id'])) : '';
                // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Admin navigation parameters
                $tab = isset($_GET['tab']) ? sanitize_text_field(wp_unslash($_GET['tab'])) : 'settings';
            }
        }
        $forms     = $this->formsManager->getForms();
        $editMode  = false;
        $name      = 'New Form';
        $title     = '';
        $language  = 'en';  
        $requestTypes = array();
        $styling = array(
            'background_color' => '#ffffff',
            'border_color' => '#cccccc',
            'text_color' => '#000000',
            'font_size' => 16,
            'accent_color' => '#f00',
            'accent_contrast_color' => '#fff',
            'border_radius' => 0
        );

        // Check if editing existing form
        if ($formId && isset($forms[$formId])) {
            $editMode   = true;
            $name       = $forms[$formId]['name'];
            $title      = $forms[$formId]['title'];
            $language   = $forms[$formId]['language'];
            $requestTypes = $forms[$formId]['request_types'];
            $locations = isset($forms[$formId]['locations']) ? $forms[$formId]['locations'] : array();
            $other_name = isset($forms[$formId]['other_name']) ? $forms[$formId]['other_name'] : 'Other';
            $other_facility_standin = isset($forms[$formId]['other_facility_standin']) ? $forms[$formId]['other_facility_standin'] : '';

            if (isset($forms[$formId]['styling'])) {
                $styling = array_merge($styling, $forms[$formId]['styling']);
            }

            if (isset($forms[$formId]['notifications'])) {
                $notifications = $forms[$formId]['notifications'];
            }
        } else {
            $locations = array();
            $other_name = 'Other';
            $other_facility_standin = '';
        }

        // Get task types for the settings tab
        $taskTypes = $this->getTaskTypes();
        
        // Get facilities for the settings tab
        $facilities = $this->getFacilities();

        // If editing an existing form, load its fields, otherwise use defaults
        $fields = $this->defaultFields;
        if ($editMode && isset($forms[$formId]['fields']) && !empty($forms[$formId]['fields'])) {
            $fields = $forms[$formId]['fields'];
        }

        // Include the edit form view
        include STOREIT_CONTACT_PLUGIN_DIR . 'includes/views/admin/EditForm.php';
    }

    /**
     * Handles the sync action in the Global Settings page
     * @return array Returns array with success status and error message if any
     */
    private function handleSync($globalSettings) {
        try {
            // Create API client with credentials from settings
            $apiClient = new StoreItAPIClient(
                $globalSettings['username'],
                $globalSettings['password'],
                $globalSettings['client_id'],
                $globalSettings['grant_type'],
                $globalSettings['base_url']
            );

            // Fetch title types and facilities data
            $facilitiesData = $apiClient->getFacilities();
            $titleTypesData = $apiClient->getTitleTypes();
            $taskTypesData = $apiClient->getTaskTypes();

            // Fetch recaptcha data

            // Update the settings with new data
            $globalSettings['facilities_data'] = $facilitiesData;
            $globalSettings['title_types_data'] = $titleTypesData;
            $globalSettings['task_types_data'] = $taskTypesData;
            $globalSettings['last_sync_timestamp'] = current_time('mysql');
            update_option('store_it_contact_global_settings', $globalSettings);

            return [
                'success' => true,
                'message' => null
            ];
        } catch (StoreItException $e) {
            
            // Return detailed error information
            return [
                'success' => false,
                'message' => sprintf(
                    // translators: %1$s is the error message, %2$s is the HTTP status code
                    __('Sync failed: %1$s (Status code: %2$s)', 'store-it-contact'),
                    $e->getMessage(),
                    $e->statusCode ?? 'unknown'
                )
            ];
        }
    }

    /**
     * Renders the Global Settings page.
     */
    public function renderGlobalSettingsPage() {
        // Store everything in a single WP option
        $optionName = 'store_it_contact_global_settings';
        
        if (isset($_POST['store_it_global_settings_nonce']) 
            && wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['store_it_global_settings_nonce'])), 'store_it_global_settings_form')) {
            
            // Which tab are we on?
            $activeTab = isset($_GET['tab']) ? sanitize_text_field(wp_unslash($_GET['tab'])) : 'general';

            /**
             * General tab
             */
            if ($activeTab === 'general') {
                $globalSettings = get_option($optionName, []);  // Get current settings first
                $globalSettings['base_url']   = sanitize_text_field(wp_unslash($_POST['base_url'] ?? ''));
                $globalSettings['username']   = sanitize_text_field(wp_unslash($_POST['username'] ?? ''));
                $globalSettings['password']   = sanitize_text_field(wp_unslash($_POST['password'] ?? ''));
                $globalSettings['grant_type'] = sanitize_text_field(wp_unslash($_POST['grant_type'] ?? ''));
                $globalSettings['client_id']  = sanitize_text_field(wp_unslash($_POST['client_id'] ?? ''));
                update_option($optionName, $globalSettings);
                
                // Check if we can run synchronization
                $canSync = !empty($globalSettings['base_url']) 
                    && !empty($globalSettings['username']) 
                    && !empty($globalSettings['password']) 
                    && !empty($globalSettings['grant_type']) 
                    && !empty($globalSettings['client_id']);
                
                if ($canSync) {
                    // Run synchronization after saving settings
                    $syncResult = $this->handleSync($globalSettings);
                    if ($syncResult['success']) {
                        add_settings_error(
                            'store_it_contact_messages',
                            'settings_updated_and_synced',
                            __('API Credentials Updated and Synchronized Successfully.', 'store-it-contact'),
                            'success'
                        );
                    } else {
                        add_settings_error(
                            'store_it_contact_messages',
                            'settings_updated_sync_failed',
                            sprintf(
                                // translators: %s is the error message from the sync failure
                                __('API Credentials Updated but Sync Failed: %s', 'store-it-contact'),
                                $syncResult['message']
                            ),
                            'error'
                        );
                    }
                } else {
                    add_settings_error(
                        'store_it_contact_messages',
                        'settings_updated',
                        __('API Credentials Updated.', 'store-it-contact'),
                        'success'
                    );
                }
            }

            /**
             * Sync tab
             */
            if ($activeTab === 'sync') {
                $globalSettings = get_option($optionName, []); // Get current settings
                // Only sync if all required creds are filled in
                $canSync = !empty($globalSettings['base_url']) 
                    && !empty($globalSettings['username']) 
                    && !empty($globalSettings['password']) 
                    && !empty($globalSettings['grant_type']) 
                    && !empty($globalSettings['client_id']);

                if ($canSync) {
                    $syncResult = $this->handleSync($globalSettings);
                    if ($syncResult['success']) {
                        // Refresh the settings after sync
                        $globalSettings = get_option($optionName, []);
                        add_settings_error(
                            'store_it_contact_messages',
                            'sync_success',
                            __('Sync completed successfully.', 'store-it-contact'),
                            'success'
                        );
                    } else {
                        add_settings_error(
                            'store_it_contact_messages',
                            'sync_error',
                            $syncResult['message'],
                            'error'
                        );
                    }
                } else {
                    add_settings_error(
                        'store_it_contact_messages',
                        'sync_error',
                        __('Cannot sync. Please fill in all API credentials first.', 'store-it-contact'),
                        'error'
                    );
                }
            }

            /**
             * reCAPTCHA tab
             */
            if ($activeTab === 'recaptcha') {
                $globalSettings = get_option($optionName, []);  // Get current settings first
                $globalSettings['recaptcha_site_key'] = sanitize_text_field(wp_unslash($_POST['recaptcha_site_key'] ?? ''));
                $globalSettings['recaptcha_secret_key'] = sanitize_text_field(wp_unslash($_POST['recaptcha_secret_key'] ?? ''));
                update_option($optionName, $globalSettings);
                
                add_settings_error(
                    'store_it_contact_messages',
                    'recaptcha_updated',
                    __('reCAPTCHA Settings Updated.', 'store-it-contact'),
                    'success'
                );
            }
        }

        // Get fresh settings after any possible updates
        $globalSettings = get_option($optionName, [
            'base_url'           => '',
            'username'           => '',
            'password'           => '',
            'grant_type'        => '',
            'client_id'         => '',
            'last_sync_timestamp'=> '',
            'title_types_data'  => '',
            'facilities_data'   => '',
            'recaptcha_site_key' => '',
            'recaptcha_secret_key' => ''
        ]);

        // Get active tab
        $activeTab = isset($_GET['tab']) ? sanitize_text_field(wp_unslash($_GET['tab'])) : 'general';

        // Include the view file
        include STOREIT_CONTACT_PLUGIN_DIR . 'includes/views/admin/GlobalSettings.php';
    }

    /**
     * Add inline admin JavaScript for form functionality
     */
    private function addInlineAdminScript() {
        $script = "
        document.addEventListener('DOMContentLoaded', function() {
            // Form validation and UI logic
            var form = document.querySelector('form');
            if (!form) return;
            
            var locationCheckboxes = document.querySelectorAll('input[name=\"locations[]\"]');
            var requestTypeCheckboxes = document.querySelectorAll('input[name=\"request_types[]\"]');
            var otherNameInput = document.getElementById('other_name');
            var otherFacilityStandinSelect = document.getElementById('other_facility_standin');

            function updateOtherFields() {
                var otherCheckbox = document.querySelector('input[name=\"locations[]\"][value=\"other\"]');
                var locationsOtherChecked = otherCheckbox ? otherCheckbox.checked : false;

                if(locationsOtherChecked && otherNameInput && otherFacilityStandinSelect) {
                    otherNameInput.setAttribute('required', 'required');
                    otherFacilityStandinSelect.setAttribute('required', 'required');
                    otherFacilityStandinSelect.removeAttribute('disabled');
                    otherNameInput.removeAttribute('disabled');
                } else if(otherNameInput && otherFacilityStandinSelect) {
                    otherNameInput.removeAttribute('required');
                    otherFacilityStandinSelect.removeAttribute('required');
                    otherFacilityStandinSelect.setAttribute('disabled', 'disabled');
                }
            }

            updateOtherFields();
            
            if (form) {
                form.addEventListener('change', updateOtherFields);
                
                form.addEventListener('submit', function(e) {
                    var hasError = false;
                    
                    // Clear previous errors
                    var errorMessages = document.querySelectorAll('.error-message');
                    errorMessages.forEach(function(msg) {
                        msg.style.display = 'none';
                    });
                    
                    // Validate locations
                    var checkedLocations = document.querySelectorAll('input[name=\"locations[]\"]:checked');
                    if (checkedLocations.length === 0) {
                        var locationError = document.getElementById('location-error');
                        if (locationError) {
                            locationError.style.display = 'block';
                            hasError = true;
                        }
                    }
                    
                    // Validate request types  
                    var checkedRequestTypes = document.querySelectorAll('input[name=\"request_types[]\"]:checked');
                    if (checkedRequestTypes.length === 0) {
                        var requestTypeError = document.getElementById('request-types-error');
                        if (requestTypeError) {
                            requestTypeError.style.display = 'block';
                            hasError = true;
                        }
                    }
                    
                    if (hasError) {
                        e.preventDefault();
                        return false;
                    }
                });
            }
        });";
        
        wp_add_inline_script('store-it-contact-admin', $script);
    }
}