<?php
// Security: Prevent direct file access
if (!defined('ABSPATH')) {
    exit;
}

add_action('admin_menu', 'hdsb_stickybanner_menu');
function hdsb_stickybanner_menu()
{
    add_menu_page(
        __('Sticky Banner Settings', 'sticky-banner'),
        __('Sticky Banner', 'sticky-banner'),
        'manage_options',
        'hdsb-stickybanner-settings',
        'hdsb_stickybanner_settings_page',
        'dashicons-megaphone'
    );
}

add_action('admin_init', 'hdsb_stickybanner_settings');
function hdsb_stickybanner_settings()
{
    // Register settings with sanitization callbacks
    register_setting('hdsb-stickybanner-settings-group', 'hdsb_stickybanner_enabled', array(
        'type' => 'boolean',
        'sanitize_callback' => 'rest_sanitize_boolean',
        'default' => false
    ));

    register_setting('hdsb-stickybanner-settings-group', 'hdsb_stickybanner_colour', array(
        'type' => 'string',
        'sanitize_callback' => 'sanitize_hex_color',
        'default' => '#910A06'
    ));

    register_setting('hdsb-stickybanner-settings-group', 'hdsb_stickybanner_text_colour', array(
        'type' => 'string',
        'sanitize_callback' => 'sanitize_hex_color',
        'default' => '#ffffff'
    ));

    register_setting('hdsb-stickybanner-settings-group', 'hdsb_stickybanner_text', array(
        'type' => 'string',
        'sanitize_callback' => 'wp_kses_post',
        'default' => ''
    ));

    register_setting('hdsb-stickybanner-settings-group', 'hdsb_stickybanner_position', array(
        'type' => 'string',
        'sanitize_callback' => 'hdsb_sanitize_position',
        'default' => 'top'
    ));

    register_setting('hdsb-stickybanner-settings-group', 'hdsb_stickybanner_btn_text', array(
        'type' => 'string',
        'sanitize_callback' => 'sanitize_text_field',
        'default' => ''
    ));

    register_setting('hdsb-stickybanner-settings-group', 'hdsb_stickybanner_btn_link', array(
        'type' => 'string',
        'sanitize_callback' => 'esc_url_raw',
        'default' => ''
    ));

    register_setting('hdsb-stickybanner-settings-group', 'hdsb_stickybanner_cookie_expiry', array(
        'type' => 'integer',
        'sanitize_callback' => 'absint',
        'default' => 7
    ));

    register_setting('hdsb-stickybanner-settings-group', 'hdsb_stickybanner_hide_on_pages', array(
        'type' => 'string',
        'sanitize_callback' => 'hdsb_sanitize_page_ids',
        'default' => ''
    ));

    register_setting('hdsb-stickybanner-settings-group', 'hdsb_stickybanner_hide_close_btn', array(
        'type' => 'boolean',
        'sanitize_callback' => 'rest_sanitize_boolean',
        'default' => false
    ));
}

/**
 * Sanitize position value
 */
function hdsb_sanitize_position($value)
{
    $valid_positions = array('top', 'bottom');
    return in_array($value, $valid_positions, true) ? $value : 'top';
}

/**
 * Sanitize page IDs
 */
function hdsb_sanitize_page_ids($value)
{
    if (empty($value)) {
        return '';
    }

    $ids = array_map('trim', explode(',', $value));
    $ids = array_filter($ids, 'is_numeric');
    $ids = array_map('absint', $ids);

    return implode(',', $ids);
}

function hdsb_stickybanner_settings_page()
{
    // Security: Check user capabilities
    if (!current_user_can('manage_options')) {
        wp_die(__('You do not have sufficient permissions to access this page.', 'sticky-banner'));
    }
?>
    <div class="wrap hdsb-settings-page">
        <!-- Page Header -->
        <div class="hdsb-page-header">
            <div>
                <h1><?php esc_html_e('Sticky Banner', 'sticky-banner'); ?></h1>
                <p class="hdsb-subtitle"><?php esc_html_e('Create beautiful announcement banners for your website', 'sticky-banner'); ?></p>
            </div>
        </div>

        <!-- Settings Form -->
        <form id="hdsb-settings-form" method="post" action="options.php">
            <?php settings_fields('hdsb-stickybanner-settings-group'); ?>
            <?php do_settings_sections('hdsb-stickybanner-settings-group'); ?>

            <!-- Enable Banner Toggle with Save Button -->
            <div class="hdsb-enable-section">
                <div class="hdsb-enable-inner">
                    <div class="hdsb-toggle-wrapper">
                        <input type="checkbox" id="hdsb_stickybanner_enabled" name="hdsb_stickybanner_enabled" value="1" <?php checked(get_option('hdsb_stickybanner_enabled', true), true); ?> class="hdsb-toggle-input" />
                        <label for="hdsb_stickybanner_enabled" class="hdsb-toggle-label">
                            <span class="hdsb-toggle-slider"></span>
                        </label>
                        <span class="hdsb-toggle-text"><?php esc_html_e('Show sticky banner on site', 'sticky-banner'); ?></span>
                    </div>
                    <div class="hdsb-enable-save">
                        <?php submit_button(__('Save Changes', 'sticky-banner'), 'primary', 'submit', false, array('form' => 'hdsb-settings-form')); ?>
                    </div>
                </div>
            </div>

            <!-- Preview Section (Full Width) -->
            <div class="hdsb-preview-section">
                <div id="preview_banner" class="hdsb-stickybanner" style="position: relative;">
                    <span id="preview_banner_inner_text" class="hdsb-stickybanner-text"><?php esc_html_e('This is what your banner will look like.', 'sticky-banner'); ?></span>
                    <a id="preview_btn" class="hdsb-stickybanner-btn" href="#"><?php esc_html_e('Learn more', 'sticky-banner'); ?></a>
                    <button id="preview_close_btn" class="hdsb-stickybanner-close" aria-label="<?php esc_attr_e('Close banner', 'sticky-banner'); ?>" style="pointer-events: none;">
                        <span aria-hidden="true">&times;</span>
                    </button>
                </div>
            </div>

            <!-- 2-Column Layout -->
            <div class="hdsb-two-column-layout">
                <!-- Left Column: Design Settings -->
                <div class="hdsb-column hdsb-column-main">
                    <div class="hdsb-form-grid">
                        <!-- Color Row: 2 columns -->
                        <div class="hdsb-form-row hdsb-color-row">
                            <div class="hdsb-form-field">
                                <label for="hdsb_stickybanner_colour"><?php esc_html_e('Background Colour', 'sticky-banner'); ?></label>
                                <input type="text" class="hdsb-stickybanner-bg-colour-picker" id="hdsb_stickybanner_colour" name="hdsb_stickybanner_colour" value="<?php echo esc_attr(get_option('hdsb_stickybanner_colour', '#910A06')); ?>" data-default-color="#910A06" />
                            </div>

                            <div class="hdsb-form-field">
                                <label for="hdsb_stickybanner_text_colour"><?php esc_html_e('Text Colour', 'sticky-banner'); ?></label>
                                <input type="text" class="hdsb-stickybanner-text-colour-picker" id="hdsb_stickybanner_text_colour" name="hdsb_stickybanner_text_colour" value="<?php echo esc_attr(get_option('hdsb_stickybanner_text_colour', '#ffffff')); ?>" data-default-color="#ffffff" />
                            </div>
                        </div>

                        <!-- Announcement Text -->
                        <div class="hdsb-form-field">
                            <label for="hdsb_stickybanner_text"><?php esc_html_e('Announcement Text', 'sticky-banner'); ?></label>
                            <input type="text" id="hdsb_stickybanner_text" name="hdsb_stickybanner_text" value="<?php echo esc_attr(get_option('hdsb_stickybanner_text')); ?>" placeholder="<?php esc_attr_e('Enter your announcement message...', 'sticky-banner'); ?>" />
                            <p class="description"><?php esc_html_e('The message to display in the banner. Basic HTML is allowed.', 'sticky-banner'); ?></p>
                        </div>

                        <!-- Button Text -->
                        <div class="hdsb-form-field">
                            <label for="hdsb_stickybanner_btn_text"><?php esc_html_e('Button Text', 'sticky-banner'); ?></label>
                            <input type="text" id="hdsb_stickybanner_btn_text" name="hdsb_stickybanner_btn_text" placeholder="<?php esc_attr_e('Learn more', 'sticky-banner'); ?>" value="<?php echo esc_attr(get_option('hdsb_stickybanner_btn_text')); ?>" />
                            <p class="description"><?php esc_html_e('If empty, "Learn more" will be used', 'sticky-banner'); ?></p>
                        </div>

                        <!-- Button Link -->
                        <div class="hdsb-form-field">
                            <label for="hdsb_stickybanner_btn_link"><?php esc_html_e('Button Link', 'sticky-banner'); ?></label>
                            <input type="url" id="hdsb_stickybanner_btn_link" name="hdsb_stickybanner_btn_link" placeholder="https://example.com" value="<?php echo esc_url(get_option('hdsb_stickybanner_btn_link')); ?>" />
                            <p class="description"><?php esc_html_e('Required for button to show. Leave empty to hide the button.', 'sticky-banner'); ?></p>
                        </div>
                    </div>
                </div>

                <!-- Right Column: Display Settings -->
                <div class="hdsb-column hdsb-column-sidebar">
                    <div class="hdsb-sidebar-card">
                        <h3><?php esc_html_e('Display Settings', 'sticky-banner'); ?></h3>

                        <div class="hdsb-form-grid">
                            <!-- Hide Close Button -->
                            <div class="hdsb-form-field">
                                <label><?php esc_html_e('Close Button', 'sticky-banner'); ?></label>
                                <label class="hdsb-checkbox-label">
                                    <input type="checkbox" id="hdsb_stickybanner_hide_close_btn" name="hdsb_stickybanner_hide_close_btn" value="1" <?php checked(get_option('hdsb_stickybanner_hide_close_btn'), true); ?> />
                                    <span><?php esc_html_e('Hide close button', 'sticky-banner'); ?></span>
                                </label>
                                <p class="description"><?php esc_html_e('Visitors won\'t be able to close the banner.', 'sticky-banner'); ?></p>
                            </div>

                            <!-- Position -->
                            <div class="hdsb-form-field">
                                <label><?php esc_html_e('Position', 'sticky-banner'); ?></label>
                                <div class="hdsb-radio-group">
                                    <label class="hdsb-radio-label">
                                        <input type="radio" name="hdsb_stickybanner_position" value="top" <?php checked('top', get_option('hdsb_stickybanner_position', 'top')); ?>>
                                        <span><?php esc_html_e('Top', 'sticky-banner'); ?></span>
                                    </label>
                                    <label class="hdsb-radio-label">
                                        <input type="radio" name="hdsb_stickybanner_position" value="bottom" <?php checked('bottom', get_option('hdsb_stickybanner_position', 'top')); ?>>
                                        <span><?php esc_html_e('Bottom', 'sticky-banner'); ?></span>
                                    </label>
                                </div>
                            </div>

                            <!-- Hide on Pages -->
                            <div class="hdsb-form-field">
                                <label for="hdsb_stickybanner_hide_on_pages"><?php esc_html_e('Hide on Pages', 'sticky-banner'); ?></label>
                                <input type="text" id="hdsb_stickybanner_hide_on_pages" name="hdsb_stickybanner_hide_on_pages" value="<?php echo esc_attr(get_option('hdsb_stickybanner_hide_on_pages')); ?>" placeholder="<?php esc_attr_e('e.g., 2, 5, 12', 'sticky-banner'); ?>" />
                                <p class="description"><?php esc_html_e('Enter page IDs separated by commas.', 'sticky-banner'); ?></p>
                            </div>

                            <!-- Cookie Expiry -->
                            <div class="hdsb-form-field">
                                <label for="hdsb_stickybanner_cookie_expiry"><?php esc_html_e('Cookie Expiry (days)', 'sticky-banner'); ?></label>
                                <input type="number" id="hdsb_stickybanner_cookie_expiry" name="hdsb_stickybanner_cookie_expiry" value="<?php echo esc_attr(get_option('hdsb_stickybanner_cookie_expiry', 7)); ?>" min="1" max="365" />
                                <p class="description"><?php esc_html_e('Days banner stays hidden after closing (default: 7).', 'sticky-banner'); ?></p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Bottom Save Button -->
            <?php submit_button(__('Save Changes', 'sticky-banner')); ?>
        </form>

        <!-- Footer -->
        <div class="hdsb-footer">
            <p>
                <?php
                printf(
                    /* translators: %s: Hidden Depth website link */
                    esc_html__('Created by %s', 'sticky-banner'),
                    '<a href="' . esc_url('https://hiddendepth.ie/?utm_source=wp_plugin&utm_medium=website&utm_campaign=sticky_banner') . '" target="_blank" rel="noopener noreferrer">' . esc_html__('Hidden Depth', 'sticky-banner') . '</a>'
                );
                ?>
                &nbsp;•&nbsp;
                <a class="hdsb_donate_link" href="<?php echo esc_url('https://www.paypal.com/cgi-bin/webscr?cmd=_s-xclick&hosted_button_id=22P45UAZ84JQL&source=url'); ?>" target="_blank" rel="noopener noreferrer">
                    <?php esc_html_e('Support Development', 'sticky-banner'); ?>
                </a>
            </p>
        </div>
    </div>

    <!-- Script to apply styles to Preview Banner -->
    <script type="text/javascript">
        (function() {
            'use strict';

            // DOM elements
            var previewText = document.getElementById('preview_banner_inner_text');
            var previewBtn = document.getElementById('preview_btn');
            var previewCloseBtn = document.getElementById('preview_close_btn');
            var textInput = document.getElementById('hdsb_stickybanner_text');
            var btnTextInput = document.getElementById('hdsb_stickybanner_btn_text');
            var bgColorInput = document.getElementById('hdsb_stickybanner_colour');
            var textColorInput = document.getElementById('hdsb_stickybanner_text_colour');
            var hideCloseBtnCheckbox = document.getElementById('hdsb_stickybanner_hide_close_btn');

            // Default values
            var defaultBannerText = <?php echo wp_json_encode(__('This is what your banner will look like.', 'sticky-banner')); ?>;
            var defaultBtnText = <?php echo wp_json_encode(__('Button text', 'sticky-banner')); ?>;

            // Update preview text
            function updatePreviewText() {
                var value = textInput.value.trim();
                previewText.textContent = value !== '' ? value : defaultBannerText;
            }

            // Update preview button text
            function updatePreviewButtonText() {
                var value = btnTextInput.value.trim();
                previewBtn.textContent = value !== '' ? value : defaultBtnText;
            }

            // Update preview colors
            function updatePreviewColors() {
                var bgColor = bgColorInput.value || '#910A06';
                var textColor = textColorInput.value || '#ffffff';

                var style = document.getElementById('preview_banner_styles');
                if (!style) {
                    style = document.createElement('style');
                    style.id = 'preview_banner_styles';
                    document.head.appendChild(style);
                }

                style.textContent =
                    '#preview_banner.hdsb-stickybanner { background: ' + bgColor + ' !important; }' +
                    '#preview_banner .hdsb-stickybanner-text { color: ' + textColor + ' !important; }' +
                    '#preview_banner .hdsb-stickybanner-btn { ' +
                        'color: ' + bgColor + ' !important; ' +
                        'background: ' + textColor + ' !important; ' +
                        'padding: 4px 8px; margin: 0 10px; text-decoration: none;' +
                    '}';
            }

            // Update preview close button visibility
            function updatePreviewCloseButton() {
                if (hideCloseBtnCheckbox && previewCloseBtn) {
                    previewCloseBtn.style.display = hideCloseBtnCheckbox.checked ? 'none' : 'block';
                }
            }

            // Initialize preview
            updatePreviewText();
            updatePreviewButtonText();
            updatePreviewColors();
            updatePreviewCloseButton();

            // Add event listeners
            textInput.addEventListener('input', updatePreviewText);
            textInput.addEventListener('change', updatePreviewText);

            btnTextInput.addEventListener('input', updatePreviewButtonText);
            btnTextInput.addEventListener('change', updatePreviewButtonText);

            bgColorInput.addEventListener('input', updatePreviewColors);
            bgColorInput.addEventListener('change', updatePreviewColors);

            textColorInput.addEventListener('input', updatePreviewColors);
            textColorInput.addEventListener('change', updatePreviewColors);

            hideCloseBtnCheckbox.addEventListener('change', updatePreviewCloseButton);

            // Color pickers trigger events when WordPress color picker changes
            if (jQuery) {
                jQuery(document).ready(function($) {
                    $('.hdsb-stickybanner-bg-colour-picker').wpColorPicker({
                        change: function(event, ui) {
                            setTimeout(function() {
                                bgColorInput.value = ui.color.toString();
                                updatePreviewColors();
                            }, 0);
                        },
                        clear: function() {
                            bgColorInput.value = '#910A06';
                            updatePreviewColors();
                        }
                    });

                    $('.hdsb-stickybanner-text-colour-picker').wpColorPicker({
                        change: function(event, ui) {
                            setTimeout(function() {
                                textColorInput.value = ui.color.toString();
                                updatePreviewColors();
                            }, 0);
                        },
                        clear: function() {
                            textColorInput.value = '#ffffff';
                            updatePreviewColors();
                        }
                    });

                    // Also listen to the iris color picker widget directly for immediate updates
                    $('.hdsb-stickybanner-bg-colour-picker').on('irischange', function(event, ui) {
                        bgColorInput.value = ui.color.toString();
                        updatePreviewColors();
                    });

                    $('.hdsb-stickybanner-text-colour-picker').on('irischange', function(event, ui) {
                        textColorInput.value = ui.color.toString();
                        updatePreviewColors();
                    });
                });
            }
        })();
    </script>
<?php
}
