<?php
/**
 * Uninstall STCW Headless Assistant
 *
 * Removes all plugin data and artifacts when the plugin is uninstalled.
 * This ensures a clean removal with no traces left behind.
 *
 * @package    STCW_Headless_Assistant
 * @since      1.3.1
 */

// If uninstall not called from WordPress, exit.
if ( ! defined( 'WP_UNINSTALL_PLUGIN' ) ) {
	exit;
}

// Initialize WordPress filesystem.
global $wp_filesystem;
if ( empty( $wp_filesystem ) ) {
	require_once ABSPATH . 'wp-admin/includes/file.php';
	WP_Filesystem();
}

/**
 * Recursively delete a directory and all its contents using WP_Filesystem.
 *
 * @param string $dir Directory path to delete.
 * @return bool True on success, false on failure.
 */
function stcw_headless_uninstall_delete_directory( $dir ) {
	global $wp_filesystem;

	if ( ! $wp_filesystem->is_dir( $dir ) ) {
		return false;
	}

	$items = $wp_filesystem->dirlist( $dir );
	if ( false === $items ) {
		return false;
	}

	foreach ( $items as $item ) {
		$path = trailingslashit( $dir ) . $item['name'];

		if ( 'd' === $item['type'] ) {
			stcw_headless_uninstall_delete_directory( $path );
		} else {
			$wp_filesystem->delete( $path, false, 'f' );
		}
	}

	return $wp_filesystem->rmdir( $dir );
}

// Get the cache directory where exports are stored.
$stcw_headless_cache_dir = WP_CONTENT_DIR . '/cache/stcw-headless-exports';

// Delete the entire stcw-headless-exports directory and all contents.
if ( $wp_filesystem->is_dir( $stcw_headless_cache_dir ) ) {
	stcw_headless_uninstall_delete_directory( $stcw_headless_cache_dir );
}
